#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliDynamicSymbol
import AaaCliLib
AaaShowMethodModel = CliDynamicSymbol.CliDynamicPlugin( "AaaShowMethodModel" )

######### "show aaa ..." #########
#
# Implemented
# -----------
#
# show aaa methods authentication
# show aaa methods authorization
# show aaa methods accounting
# show aaa methods all
def _methodsToList( ml ):
   methods = []
   for idx in sorted( ml.method.keys() ):
      methods.append( ml.method[ idx ] )
   return methods

def _authzMethodsToList( ml ):
   methods = []
   for idx in sorted( ml.defaultMethod.keys() ):
      methods.append( ml.defaultMethod[ idx ] )
   return methods

def addAuthenMethodInfo( model, methodList, nameOverride=None ):
   name = nameOverride if nameOverride is not None else methodList.name
   model[ name ] = AaaShowMethodModel.AaaMethodListsMethods()
   model[ name ].methods = _methodsToList( methodList )

def addAuthzMethodInfo( model, methodList, nameOverride=None ):
   name = nameOverride if nameOverride is not None else methodList.name
   model[ name ] = AaaShowMethodModel.AaaMethodListsMethods()
   model[ name ].methods = _authzMethodsToList( methodList )

def showAuthenMethodLists( mode, args ):
   result = AaaShowMethodModel.AaaMethodListsAuthen()
   cfg = AaaCliLib.configAaa( mode )
   addAuthenMethodInfo( result.loginAuthenMethods, cfg.defaultLoginMethodList )
   for v in cfg.loginMethodList.values():
      addAuthenMethodInfo( result.loginAuthenMethods, v )
   addAuthenMethodInfo( result.enableAuthenMethods, cfg.defaultEnableMethodList )
   for v in cfg.enableMethodList.values():
      addAuthenMethodInfo( result.enableAuthenMethods, v )
   addAuthenMethodInfo( result.dot1xAuthenMethods, cfg.defaultDot1xMethodList )

   return result

def _cmdMethodListGenerator( getMlFunc, cmpMlFunc ):
   """ Helper function used by showAuthzMethodList and showAcctMethodList.
   When different command levels have the same method list configuration
   display them on one line so that the most common configs don't take
   up 16 lines of space. """
   startIndex = 0
   startIndexMl = getMlFunc( "command00" )
   i = 1
   for i in range( 1, 16 ):
      name = f"command{i:02d}"
      ml = getMlFunc( name )
      if not cmpMlFunc( startIndexMl, ml ):
         index = startIndex
         startIndex = i
         startIndexMl = ml
         # returning the command range which have same ml
         yield index, i - 1
   yield startIndex, i

def showAuthzMethodLists( mode, args ):
   result = AaaShowMethodModel.AaaMethodListsAuthz()
   cfg = AaaCliLib.configAaa( mode )
   mlg = _cmdMethodListGenerator( lambda name:
                                 cfg.authzMethod[ name ],
                                  lambda ml0, ml1:
                                     _authzMethodsToList( ml0 ) == \
                                     _authzMethodsToList( ml1 ) )
   for start, end in mlg:
      ml = cfg.authzMethod[ f"command{start:02d}" ]
      if start == end:
         name = f"privilege{start}"
      else:
         name = f"privilege{start}-{end}"
      addAuthzMethodInfo( result.commandsAuthzMethods, ml, name )
   addAuthzMethodInfo( result.execAuthzMethods,
                  cfg.authzMethod[ "exec" ], None )
   return result

def showAcctMethodLists( mode, args ):
   result = AaaShowMethodModel.AaaMethodListsAcct()
   cfg = AaaCliLib.configAaa( mode )
   def _methodsDictToList( mlDict ):
      keys = sorted( mlDict )
      return [ mlDict[ key ] for key in keys ]

   def _cmpAcctMl( ml0, ml1 ):
      if ml0.defaultAction != ml1.defaultAction:
         return False
      if ml0.consoleAction != ml1.consoleAction:
         return False
      if ml0.consoleUseOwnMethod != ml1.consoleUseOwnMethod:
         return False
      if _methodsDictToList( ml0.defaultMethod ) != \
             _methodsDictToList( ml1.defaultMethod ):
         return False
      if _methodsDictToList( ml0.consoleMethod ) != \
             _methodsDictToList( ml1.consoleMethod ):
         return False
      return True
   mlg = _cmdMethodListGenerator( lambda name: cfg.acctMethod[ name ],
                                  _cmpAcctMl )

   def _showAcctMl( ml ):
      acctMethods = AaaShowMethodModel.AaaMethodListsAcctMethods()
      if ml.defaultAction != 'none':
         acctMethods.defaultAction = ml.defaultAction
         acctMethods.defaultMethods = _methodsDictToList( ml.defaultMethod )

      # pylint: disable=protected-access
      acctMethods._consoleUseOwnMethod = ml.consoleUseOwnMethod
      if ml.consoleUseOwnMethod and ml.consoleAction != 'none':
         acctMethods.consoleAction = ml.consoleAction
         acctMethods.consoleMethods = _methodsDictToList( ml.consoleMethod )

      return acctMethods

   for start, end in mlg:
      if start == end:
         name = f"privilege{start}"
      else:
         name = f"privilege{start}-{end}"
      result.commandsAcctMethods[ name ] = _showAcctMl(
                                            cfg.acctMethod[ f"command{start:02d}" ] )

   result.execAcctMethods[ "exec" ] = _showAcctMl( cfg.acctMethod[ "exec" ] )
   result.systemAcctMethods[ "system" ] = _showAcctMl( cfg.acctMethod[ "system" ] )
   result.dot1xAcctMethods[ "dot1x" ] = _showAcctMl( cfg.acctMethod[ "dot1x" ] )
   return result

def showAllMethodLists( mode, args ):
   result = AaaShowMethodModel.AaaMethodLists()
   result.authentication = showAuthenMethodLists( mode, args )
   result.authorization = showAuthzMethodLists( mode, args )
   result.accounting = showAcctMethodLists( mode, args )
   return result

