#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Bool
from CliModel import Dict
from CliModel import List
from CliModel import Model
from CliModel import Str
from CliModel import Submodel

class AaaMethodListsMethods( Model ):
   methods = List( valueType=str,
                   help="List of methods for a particular mode" )

   def renderMethods( self, name ):
      print( f"  name={name} methods={', '.join( self.methods )}" )

class AaaMethodListsAcctMethods( Model ):
   defaultAction = Str( help="Present if accounting for default actions is enabled",
                        optional=True )
   defaultMethods = List( valueType=str,
                          help="List of methods for default accounting" )
   _consoleUseOwnMethod = Bool( help="If console method is possible, private" )
   consoleAction = Str( help="Present if accounting for console actions is enabled",
                        optional=True )
   consoleMethods = List( valueType=str,
                          help="List of methods for a console accounting" )

   def renderMethods( self, name ):
      if not self.defaultAction:
         print( f"  name={name} default-action=none" )
      else:
         print( f"  name={name} default-action={self.defaultAction} "
                f"default-methods={', '.join( self.defaultMethods )}" )

      if self._consoleUseOwnMethod:
         if not self.consoleAction:
            print( ' ' * len( '  name=' + name ), "console-action=none" )
         else:
            print( f"{' ' * len( '  name=' + name)} "
                   f"console-action={self.consoleAction} "
                   f"console-methods={', '.join( self.consoleMethods )}" )

class AaaMethodListsAuthen( Model ):
   loginAuthenMethods = Dict( valueType=AaaMethodListsMethods,
                              help="A Mapping between login authentication type and "
                                   "methods configured" )
   enableAuthenMethods = Dict( valueType=AaaMethodListsMethods,
                                help="A Mapping between enable authentication type "
                                     "and methods configured" )
   dot1xAuthenMethods = Dict( valueType=AaaMethodListsMethods,
                              help="A Mapping between dot1x authentication type and "
                                   "methods configured" )

   def renderAuthenType( self, listType, authType, authenMethodInfo ):
      print( f"Authentication method {listType} for {authType}:" )
      for name, methods in sorted( authenMethodInfo.items(), key=lambda x: x[ 0 ] ):
         methods.renderMethods( name )

   def render( self ):
      self.renderAuthenType( "lists", "LOGIN", self.loginAuthenMethods )
      self.renderAuthenType( "list", "ENABLE", self.enableAuthenMethods )
      self.renderAuthenType( "list", "DOT1X", self.dot1xAuthenMethods )

class AaaMethodListsAuthz( Model ):
   commandsAuthzMethods = Dict( valueType=AaaMethodListsMethods,
                                help="A Mapping between command authorization type "
                                     "and methods configured" )
   execAuthzMethods = Dict( valueType=AaaMethodListsMethods,
                            help="A Mapping between exec authorization type "
                                 "and methods configured" )

   def renderAuthzType( self, listType, authzType, authzMethodInfo ):
      print( f"Authorization method {listType} for {authzType}:" )
      for name, methods in sorted( authzMethodInfo.items(), key=lambda x: x[ 0 ] ):
         methods.renderMethods( name )

   def render( self ):
      self.renderAuthzType( "lists", "COMMANDS", self.commandsAuthzMethods )
      self.renderAuthzType( "list", "EXEC", self.execAuthzMethods )

class AaaMethodListsAcct( Model ):
   commandsAcctMethods = Dict( valueType=AaaMethodListsAcctMethods,
                               help="A Mapping between command accounting type "
                                     "and methods configured" )
   execAcctMethods = Dict( valueType=AaaMethodListsAcctMethods,
                           help="A Mapping between exec accounting type "
                                 "and methods configured" )
   systemAcctMethods = Dict( valueType=AaaMethodListsAcctMethods,
                             help="A Mapping between system accounting type "
                                  "and methods configured" )
   dot1xAcctMethods = Dict( valueType=AaaMethodListsAcctMethods,
                            help="A Mapping between dot1x accounting type "
                                 "and methods configured" )

   def renderAcctType( self, listType, acctType, acctMethodInfo ):
      print( f"Accounting method {listType} for {acctType}:" )
      for name, methods in sorted( acctMethodInfo.items(), key=lambda x: x[ 0 ] ):
         methods.renderMethods( name )

   def render( self ):
      self.renderAcctType( "lists", "COMMANDS", self.commandsAcctMethods )
      self.renderAcctType( "list", "EXEC", self.execAcctMethods )
      self.renderAcctType( "list", "SYSTEM", self.systemAcctMethods )
      self.renderAcctType( "list", "DOT1X", self.dot1xAcctMethods )

class AaaMethodLists( Model ):
   authentication = Submodel( valueType=AaaMethodListsAuthen,
                              help="AAA Authentication information" )
   authorization = Submodel( valueType=AaaMethodListsAuthz,
                              help="AAA Authorization information" )
   accounting = Submodel( valueType=AaaMethodListsAcct,
                              help="AAA Accounting information" )

   def render( self ):
      self.authentication.render()
      self.authorization.render()
      self.accounting.render()

