#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import CliCommon
from CliModel import Dict, Enum, Float, Int, List, Model, Str
import SecretCli
import TableOutput
import Tac

import datetime
import time
import math

ACCT_NAME_LEN = 8

class Users( Model ):

   class User( Model ):
      username = Str( help="The username of this account" )
      role = Str( help="What role the account has been given"
                  " (e.g. 'network-admin')", optional=True,
                  default="network-operator" )
      privLevel = Int( help=( f"The configured privilege level for this user"
                              f" (between {CliCommon.MIN_PRIV_LVL} and "
                              f"{CliCommon.MAX_PRIV_LVL} inclusive)" ) )
      sshAuthorizedKey = Str( help="SSH public key for this user", optional=True )
      secondarySshKeys = List( valueType=str,
                               help="SSH secondary keys for this user",
                               optional=True )

      def render( self ):
         print( f"user: {self.username}" )
         print( f"       role: {self.role}" )
         print( f"       privilege level: {self.privLevel}" )
         if self.sshAuthorizedKey:
            print( f"       ssh public key: {self.sshAuthorizedKey}" )
         for sshKey in sorted( self.secondarySshKeys ):
            print( f"       ssh public key: {sshKey}" )

   users = Dict( valueType=User, help="Maps a username to details for this user" )

   def render( self ):
      for _, user in sorted( self.users.items() ):
         user.render()

class AaaUsers( Model ):
   class AaaUser( Model ):
      password = Str( "The encrypted password of this account" )

   enablePassword = Str( help="The encrypted enable password", optional=True )
   users = Dict( valueType=AaaUser, help="Maps a username to details for this user" )

   def render( self ):
      if not self.enablePassword:
         self.enablePassword = "(None set)"
      print( f"Enable password (encrypted): {self.enablePassword}" )

      if not self.users:
         print( "No logins configured" )
      else:
         passwdLen = len( SecretCli.encrypt( "A", hashAlgorithm="md5" ) )
         fmtStr = f"%-{ACCT_NAME_LEN}s  %-{passwdLen}s"
         print( fmtStr % ( "Username", "Encrypted passwd" ) )
         print( fmtStr % ( "-" * ACCT_NAME_LEN, "-" * passwdLen ) )
         for user in sorted( self.users ):
            print( fmtStr % ( user, self.users[ user ].password ) )

class AaaLockedUser( Model ):
   lockoutEndTime = Float( help="Time that user's account will be unlocked" )
   lockoutStartTime = Float( help="Time that user's account was locked" )

class AaaLockedUsers( Model ):
   lockedUsers = Dict( valueType=AaaLockedUser,
                       help="Maps a username to details for this user" )

   def render( self ):
      headings = ( 'User', 'Start Time', 'End Time', 'Expires In' )
      table = TableOutput.createTable( headings )
      for user, userInfo in sorted( self.lockedUsers.items() ):
         startTime = time.ctime( userInfo.lockoutStartTime )
         endTime = time.ctime( userInfo.lockoutEndTime )
         # math.floor removes microseconds
         expirationTime = math.floor( userInfo.lockoutEndTime - Tac.utcNow() )
         expirationTimeStr = datetime.timedelta( seconds=expirationTime )
         table.newRow( user, startTime, endTime, expirationTimeStr )
      # 'table.output()' adds an empty line to terminate the table. The
      # tests don't expect this line, so remove it.
      print( "\n".join( table.output().split( "\n" )[ : -1 ] ) )

class ShowRolesList( Model ):
   sequenceNumber = Int( help="Sequence number designates a rule's placement "
                              "in the role, value ranges from 1 to 256" )
   cmdPermission = Enum( values=( "permit", "deny" ), help="If match is found "
                   "against a rule, decision is based on that rule and all other "
                   "rules followed are ignored. If no match is found, the command "
                   "is denied by default" )
   mode = Str( help="The regex of a cli mode name for which this rule applies, "
               "if absent, rule is valid in all modes", optional=True )
   cmdRegex = Str( help="Regular expression that corresponds to one "
                        "or more CLI commands" )

class Rules( Model ):
   rules = List( valueType=ShowRolesList, help="A mapping of an authorization "
                     "role name to its rules" )

class ShowRoles( Model ):
   roles = Dict( valueType=Rules, help="Display the contents of built-in "
                 "and user-defined roles" )
   defaultRole = Str( help="Default role" )
   def render( self ):
      print( 'The default role is ' + self.defaultRole + '\n' )
      for role, info in sorted( self.roles.items() ):
         print( 'role:', role )
         for rule in info.rules:
            print( '       ', rule.sequenceNumber, rule.cmdPermission,
                   ( f'mode {rule.mode} ' if rule.mode else '' ) + 'command',
                   rule.cmdRegex )

