# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Dict, Float, Model
import Arnet
import Tac
import TableOutput
from CliMode.Intf import IntfMode
from IntfModels import Interface
from Ark import timestampToStr

class InterfaceRateWatermark( Model ):
   intfName = Interface(
      help="Name of the interface that the watermark rate applies" )
   inBitRateWatermark = Float( help="Input bps rate", optional=True )
   inBitRateWatermarkPct = Float(
      help="Input packets rate percentage", optional=True )
   outBitRateWatermark = Float( help="Output ps rate", optional=True )
   outBitRateWatermarkPct = Float(
         help='Output packets rate percentagepercentage of total capacity.',
         optional=True )

   @staticmethod
   def createTable():
      # Port
      portMinLen = 9
      portMaxLen = 13

      # In rate pct, Out rate Pct
      ratePctMinLen = 10

      # In Mbps, Out Mbps
      mbpsMinLen = 10

      columnList = [
            "Port", "In Mbps Watermark", "%", "Out Mbps Watermark", "%" ]
      table = TableOutput.createTable( columnList )

      # Port
      fPort = TableOutput.Format( justify="left", minWidth=portMinLen,
                      maxWidth=portMaxLen, isHeading=True )
      fPort.noPadLeftIs( True )
      fPort.padLimitIs( True )

      # In rate pct, Out rate Pct
      fRatePct = TableOutput.Format(
         justify="right", dotAlign=True, minWidth=ratePctMinLen )
      fRatePct.noPadLeftIs( True )
      fRatePct.padLimitIs( True )

      # In Mbps, Out Mbps
      fMbps = TableOutput.Format(
         justify="right", dotAlign=True, minWidth=mbpsMinLen )
      fMbps.noPadLeftIs( True )
      fMbps.padLimitIs( True )

      table.formatColumns( fPort, fMbps, fRatePct, fMbps, fRatePct )
      table.formatRows( TableOutput.Format( isHeading=True, border=False ) )

      return table

   def insertRow( self, table ):
      # Setting interval in the format mm:ss

      inBitRateWatermarkMbps = self.inBitRateWatermark / 1000 / 1000
      outBitRateWatermarkMbps = self.outBitRateWatermark / 1000 / 1000
      inBitRateWatermarkStr = f'{inBitRateWatermarkMbps:.1f}'
      outBitRateWatermarkStr = f'{outBitRateWatermarkMbps:.1f}'

      # Calculating and setting In and Out rate %
      if self.inBitRateWatermarkPct is None:
         inRatePctStr = '-'
      else:
         inRatePctStr = f'{self.inBitRateWatermarkPct:.1f}%'

      if self.outBitRateWatermarkPct is None:
         outRatePctStr = '-'
      else:
         outRatePctStr = f'{self.outBitRateWatermarkPct:.1f}%'

      # Creating a new row.
      table.newRow(
            IntfMode.getShortname( self.intfName ),
            inBitRateWatermarkStr,
            inRatePctStr,
            outBitRateWatermarkStr,
            outRatePctStr )

class InterfaceRateWatermarks( Model ):
   interfaces = Dict( keyType=Interface, valueType=InterfaceRateWatermark,
                      help="Interface rate watermarks within the configured window" )
   windowStart = Float( help="Start of the watermark measurement window" )
   windowEnd = Float( help="End of the watermark measurement window" )

   @staticmethod
   def createTable():
      watermarkTextLen = 25
      timeTextLen = 12
      table = TableOutput.TableFormatter()
      watermarkWindowText = TableOutput.Format(
         justify='left', minWidth=watermarkTextLen )
      watermarkWindowText.noPadLeftIs( True )
      watermarkWindowText.padLimitIs( True )
      timeText = TableOutput.Format( justify='right', minWidth=timeTextLen )
      timeText.noPadLeftIs( True )
      timeText.padLimitIs( True )

      table.formatColumns( watermarkWindowText, timeText )
      table.formatRows( TableOutput.Format( border=False ) )
      table.formatRows( TableOutput.Format( border=False, noBreak=True ) )

      return table

   def setWatermarkUpdateTime( self ):
      table = self.createTable()
      table.newRow(
         "Watermark update time:",
         timestampToStr( self.windowEnd, relative=True,
                         millisecondsPrecision=False ) )
      return table

   def render( self ):
      intfs = list( self.interfaces )
      if not intfs:
         return

      windowTable = self.setWatermarkUpdateTime()

      sortedIntfs = Arnet.sortIntf( intfs )
      table = InterfaceRateWatermark.createTable()
      for intf in sortedIntfs:
         self.interfaces[ intf ].insertRow( table )

      print( windowTable.output() )
      print( table.output() )
