# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Ark
import TableOutput
from CliModel import (
      Model,
      Str,
      Int,
      Bool,
      List,
      Dict,
      Enum,
      Float,
      Submodel,
      DeferredModel,
   )
from ArnetModel import (
   IpGenericPrefix,
)
from CliPlugin.AleFibCliModel import (
   stateEnum,
   support,
   routeTypeEnum,
   decimalPlaces,
   largeInt,
   AleL3AdjVia,
)

pickTypeEnum = tuple( Tac.Type( 'Routing::Hardware::CounterKeyType' ).attributes )
fibRouteTypeEnum = tuple( set( Tac.Type( 'Routing::Fib::RouteType' ).attributes ) |
                          set( Tac.Type( 'Routing6::Fib::RouteType' ).attributes ) )

class FecProxyInfo( Model ):
   sourceFec = Int( help='Source Fec Id for this proxy set' )
   cmdErr = Str( optional=True, help="Error Message" )
   proxyFecList = List( valueType=largeInt, help='List of proxy FECs for source' )

class FecProxyInfoList( DeferredModel ):
   cmdErr = Str( optional=True, help="Error Message" )
   fecAliases = List( optional=True,
                      valueType=FecProxyInfo, help='List of FEC proxies' )

class AlePdckValues( Model ):
   value1 = Int( help='PDCK value 1' )
   value2 = Int( help='PDCK value 2' )

class AleL2AdjCounterBindingsInfo( Model ):
   pdcks = List( valueType=AlePdckValues,
      help='List of platform-dependent counter keys bound to the L2 Adjacency' )

class AlePdckCounterBindingsInfo( Model ):
   pick = Int( help="Platform-independent counter key bound to the PDCK" )
   pickType = Enum( values=pickTypeEnum, optional=True,
                    help='Specifies counter key Type' )
   pickIndex = Int( help="PICK Index" )

class AleL2AdjCounterBindings( DeferredModel ):
   cmdErr = Str( optional=True, help="Error Message" )
   l2AdjCounterBindings = Dict( keyType=largeInt,
                                valueType=AleL2AdjCounterBindingsInfo,
                                optional=True,
                                help="Dictionary to map Layer 2 adjacency index to "
                                     "its counter related bindings" )

class AlePdckCounterBindings( DeferredModel ):
   cmdErr = Str( optional=True, help="Error Message" )
   pdcksCounterBindings = Dict( keyType=str, valueType=AlePdckCounterBindingsInfo,
                               optional=True,
                               help="Dictionary to map Ale Platform-dependent "
                                    "counter key (PDCK) to PDCK counter related "
                                    "bindings" )

class FibFecInfo( Model ):
   cmdErr = Str( optional=True, help="Error flag" )
   fecType = Enum( values=routeTypeEnum, optional=True, help="FEC type" )
   vias = Int( optional=True, help="Number of vias for the route" )
   fecVias = List( valueType=AleL3AdjVia, optional=True,
                   help="Via list" )
   backupFecVias = List( valueType=AleL3AdjVia, optional=True,
                         help="Backup Via list" )

class FibFec( DeferredModel ):
   cmdErr = Str( optional=True, help="Error flag" )
   routeType = Enum( values=fibRouteTypeEnum, optional=True, help="Route type" )
   fec = Dict( keyType=largeInt, valueType=FibFecInfo, optional=True,
               help="Mapping fecId to its FEC Info" )
   routeCompressed = Bool( optional=True,
                           help="Route is compressed" )
   compressedByRoute = IpGenericPrefix( optional=True,
                                        help="IP Address of the parent route" )

class FibRoute( DeferredModel ):
   cmdErr = Str( optional=True, help="Error flag" )
   vrfName = Str( optional=True, help="VRF Name" )
   route = Dict( keyType=IpGenericPrefix, valueType=FibFec, optional=True,
                 help="Mapping prefix to its FibRoute Info" )

class AdjResourceOptimizationThresholds( Model ):
   low = Int( help="Threshold for starting resource optimization" )
   high = Int( help="Threshold for stopping resource optimization" )

class AdjResourceOptimization( Model ):
   supported = Enum( values=support,
                     help="Adjacency resource optimization support" )
   enabled = Enum( values=stateEnum,
                   optional=True,
                   help="Adjacency resource optimization enabled" )
   thresholds = Submodel( valueType=AdjResourceOptimizationThresholds,
                          optional=True,
                          help="Adjacency resource optimization thresholds" )

class FibSummaryStatus( Model ):
   __revision__ = 2
   adjShare = Enum( values=stateEnum, optional=True,
                    help="Adjacency sharing status" )
   bfdEvent = Enum( values=stateEnum, optional=True, help="BFD peer event" )
   deletionDelay = Float( help="Deletion delay" )
   pbrSupport = Enum( values=support, optional=True, help="PBR support" )
   urpfSupport = Enum( values=support, help="URPF support" )
   icmpStatus = Enum( values=stateEnum, help="ICMP status" )
   maxAleEcmp = Int( optional=True, help="Max Ale ECMP" )
   ucmpWeightDeviation = Float( optional=True, help="UCMP weight deviation" )
   maxRoutes = Int( help="Maximum number of routes" )
   protectDefaultRoute = Enum( values=stateEnum,
                               help="Protect default route status" )
   fibCompression = Enum( values=stateEnum,
                          help="FIB compression status" )
   adjResourceOptimization = Submodel( valueType=AdjResourceOptimization,
                           optional=True,
                           help="Adjacency resource optimization" )
   maxFecHierarchyLevels = Int( help="Maximum FEC hierarchy levels" )

   def degrade( self, dictRepr, revision ):
      # Removed 'parentDrop' in revision 2. By default is disabled
      if revision == 1:
         dictRepr[ 'parentDrop' ] = 'disabled'
      return dictRepr

   def render( self ):
      print( 'Fib summary' )
      print( '-----------' )
      if self.adjShare:
         print( f'Adjacency sharing: { self.adjShare }' )
      if self.bfdEvent:
         print( f'BFD peer event: { self.bfdEvent }' )
      print( f'Deletion Delay: { self.deletionDelay }' )
      if self.protectDefaultRoute:
         print( f'Protect default route: { self.protectDefaultRoute }' )
      if self.pbrSupport:
         print( f'PBR: { self.pbrSupport }' )
      if self.urpfSupport:
         print( f'URPF: { self.urpfSupport }' )
      if self.icmpStatus:
         print( f'ICMP unreachable: { self.icmpStatus }' )
      if self.maxAleEcmp is not None:
         print( f'Max Ale ECMP: { self.maxAleEcmp }' )
      if self.maxFecHierarchyLevels is not None:
         print( f'Maximum FEC hierarchy levels: { self.maxFecHierarchyLevels }' )
      if self.ucmpWeightDeviation is not None:
         print( f'UCMP weight deviation: '
         f'{str(round(self.ucmpWeightDeviation, decimalPlaces))}' )
      print( f'Maximum number of routes: { self.maxRoutes }' )
      if self.fibCompression:
         print( f'Fib compression: { self.fibCompression }' )
      if self.adjResourceOptimization is not None:
         if self.adjResourceOptimization.supported == 'supported':
            if self.adjResourceOptimization.enabled is not None:
               print( f'Resource optimization for adjacency '
                      f'programming: { self.adjResourceOptimization.enabled }' )
            if self.adjResourceOptimization.thresholds is not None:
               print( f'Adjacency resource optimization thresholds: '
                      f'low { self.adjResourceOptimization.thresholds.low },'
                      f' high { self.adjResourceOptimization.thresholds.high }' )
         else:
            print( f'Resource optimization for adjacency'
                   f' programming: { self.adjResourceOptimization.supported }' )

class AleResEcmpSummary( Model ):
   vrfName = Str( help="VRF Name" )
   numPrefixes = Int( help="Number of programmed resilient ECMP prefixes" )
   numProgrammed = Int( help="Number of routes programmed as resilient" )
   numUnprogrammed = Int( help="Number of routes not programmed as resilient" )
   numOrdered = Int( help="Number of ordered resilient routes" )

   def render( self ):
      print( f'VRF: { self.vrfName }' )
      print( f'Number of resilient prefixes: { self.numPrefixes }' )
      print( f'Number of programmed resilient routes: { self.numProgrammed }' )
      print( f'Number of unprogrammed resilient routes: { self.numUnprogrammed }' )
      print( f'Number of ordered resilient routes: { self.numOrdered }' )

class AleResEcmpRouteInfo( Model ):
   prefix = IpGenericPrefix( help="The route's prefix" )
   resilient = Bool( help="The route is programmed as resilient" )
   fecId = Int( help="The non-resilient FEC ID" )
   resFecId = Int( help="The resilient FEC ID" )
   reason = Str( help="Resason why route is not programmed as resilient" )
   orderedVias = Bool( help="The vias in the route's ECMP FEC are programmed in"
                            " a deterministic order" )

class AleResEcmpRouteInfos( Model ):
   capacity = Int( optional=True, help="The resilient capacity" )
   redundancy = Int( optional=True, help="The resilient redundancy" )
   routes = List( valueType=AleResEcmpRouteInfo,
                  help="List of resilient routes information" )
   orderedVias = Bool( optional=True,
                       help="The resilient prefix is configured as ordered" )

class AleResEcmpAllOrSpecific( Model ):
   vrfName = Str( help="VRF name" )
   resPrefixesToRouteInfos = Dict( keyType=IpGenericPrefix,
                                   valueType=AleResEcmpRouteInfos,
                                   help="Map from resilient ECMP prefixes to their "
                                         "covered routes" )
   policyBasedResRouteInfos = Submodel( valueType=AleResEcmpRouteInfos,
                                        optional=True,
                                        help="Policy marked resilient routes "
                                             "information" )
   _oneRoute = Bool( help="Render one specific route" )

   def render( self ):
      print( f'VRF: { self.vrfName }' )

      if not self._oneRoute:
         print( 'Codes: * - route not resilient, ^ - ordering not guaranteed' )

         def renderRouteInfos( routeInfos ):
            for routeInfo in routeInfos.routes:
               markers = ''
               resFecIdStr = ''
               if routeInfo.resilient:
                  resFecIdStr = f', resilient FEC ID: { routeInfo.resFecId }'
               else:
                  markers += '*'
               # If the prefix is configured as orderd but it's covering more than
               # one route or the covered route is not ordered, mark it as 'ordering
               # not guaranteed'
               if routeInfos.orderedVias:
                  if len( routeInfos.routes ) != 1 or not routeInfo.orderedVias:
                     markers += '^'

               print( f'\t{markers}{routeInfo.prefix},'
                     f' FEC ID: {routeInfo.fecId}{resFecIdStr}' )

         # Printing all routes
         for resPrefix in self.resPrefixesToRouteInfos:
            routeInfos = self.resPrefixesToRouteInfos[ resPrefix ]
            orderedKeyword = ', ordered' if routeInfos.orderedVias else ''
            print( f'{resPrefix}: capacity: {routeInfos.capacity}, '
                  f'redundancy: {routeInfos.redundancy}{orderedKeyword}' )
            renderRouteInfos( routeInfos )
         if self.policyBasedResRouteInfos:
            routeInfos = self.policyBasedResRouteInfos
            # policy based RECMP does not support ordered vias
            print( f'Policy marked: capacity: { routeInfos.capacity }, '
                  f'redundancy: { routeInfos.redundancy }' )
            renderRouteInfos( routeInfos )

      else:
         # Printing a specific route
         if self.policyBasedResRouteInfos:
            coveringPrefix = None
            routeInfos = self.policyBasedResRouteInfos
         else:
            coveringPrefix, routeInfos = \
                           next( iter( self.resPrefixesToRouteInfos.items() ) )
         routeInfo = routeInfos.routes[ 0 ]

         print( f'{ routeInfo.prefix }:' )
         if routeInfo.resilient:
            print( f'\tResilient: yes, capacity: '
                  f'{ routeInfos.capacity }, redundancy: { routeInfos.redundancy }' )
         else:
            print( '\tResilient: no' )

         if coveringPrefix:
            # policy based do not have a coveringPrefix.
            print( f'\tCovered by: { coveringPrefix }' )
         print( f'\tFEC ID: { routeInfo.fecId }' )
         print( f'\tResilient FEC ID: { routeInfo.resFecId }' )
         print( f'\tOrdered: { "yes" if routeInfo.orderedVias else "no" }' )
         print( f'\tReason: { routeInfo.reason }' )

class VrfDropRoute( Model ):
   vrfName = Str( help="Name of a VRF" )
   insertTime = Float( help='Drop route creation time' )
   prefix = IpGenericPrefix( help="The route's prefix" )

class DropRouteStatus( Model ):
   routingTelemetry = Enum( values=stateEnum,
                    help="Routing telemetry status" )
   vrfDropRoutes = List( optional=True, valueType=VrfDropRoute,
                              help="List of all drop routes" )

   def render( self ):
      if self.routingTelemetry == 'disabled':
         print( 'Route Monitoring is disabled.\n' )
         return
      if not self.vrfDropRoutes:
         return
      table = TableOutput.createTable( ( "Prefix", "VRF Name", "Creation Time" ) )
      tableMaxWidth = table.printableWidth()
      tableMaxWidth = max( tableMaxWidth, 80 )

      f1 = TableOutput.Format( justify="left", maxWidth=20, minWidth=12 )
      f1.noPadLeftIs( True )
      f1.padLimitIs( True )

      f2 = TableOutput.Format( justify="left", maxWidth=30, minWidth=6 )
      f2.noPadLeftIs( True )
      f2.padLimitIs( True )

      f3 = TableOutput.Format( justify="left", maxWidth=30, minWidth=6 )
      f3.noPadLeftIs( True )
      f3.padLimitIs( True )

      table.formatColumns( f1, f2 )

      for vrfDropRoute in self.vrfDropRoutes:
         creationTime = Ark.timestampToStr( vrfDropRoute.insertTime, False )
         table.newRow( vrfDropRoute.prefix, vrfDropRoute.vrfName, creationTime )
      print( table.output() )

class VrfRouteEntry( Model ):
   vrfName = Str( help="Name of a VRF" )
   insertTime = Float( help='Route entry creation time' )
   prefix = IpGenericPrefix( help="The route's prefix" )

class InconsistentRouteStatus( Model ):
   routingTelemetry = Enum( values=stateEnum,
                    help="Routing telemetry status" )
   vrfInconsistentRoutes = List( optional=True, valueType=VrfRouteEntry,
                              help="List of all inconsistent routes" )

   def render( self ):
      if self.routingTelemetry == 'disabled':
         print( 'Route Monitoring is disabled.\n' )
         return
      if not self.vrfInconsistentRoutes:
         return
      print( "Some of the inconsistent routes may be transient." )
      print( "Run the command multiple times to find non-transient "
             "inconsistencies.\n" )

      table = TableOutput.createTable( ( "Prefix", "VRF Name", "Creation Time" ) )
      tableMaxWidth = table.printableWidth()
      tableMaxWidth = max( tableMaxWidth, 80 )

      f1 = TableOutput.Format( justify="left", maxWidth=20, minWidth=12 )
      f1.noPadLeftIs( True )
      f1.padLimitIs( True )

      f2 = TableOutput.Format( justify="left", maxWidth=30, minWidth=6 )
      f2.noPadLeftIs( True )
      f2.padLimitIs( True )

      f3 = TableOutput.Format( justify="left", maxWidth=30, minWidth=6 )
      f3.noPadLeftIs( True )
      f3.padLimitIs( True )

      table.formatColumns( f1, f2 )

      for vrfInconsistentRoute in self.vrfInconsistentRoutes:
         creationTime = Ark.timestampToStr( vrfInconsistentRoute.insertTime, False )
         table.newRow(
               vrfInconsistentRoute.prefix,
               vrfInconsistentRoute.vrfName, creationTime )
      print( table.output() )
