# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import sys

import AgentCommandRequest
import Arnet
from CliPlugin import AleCliLib
from CliPlugin.AleFibCliModel import (
      AleL2Adj,
      stateEnum,
      support,
)
from CliPlugin.AleFibCli import fecIdValid
from CliDynamicSymbol import CliDynamicPlugin
import Tac
import AleFibCliDiffUtilLib
import SharedMem

AleFibCliDynamicModel = CliDynamicPlugin( "AleFibCliDynamicModel" )

def handleShowIpIpv6HardwareFibSummary( mode, args ):
   v4 = 'ip' in args
   if v4:
      routingHwConfig = AleCliLib.routingHwConfig
      routingHwStatus = AleCliLib.routingHwStatus
      routingHwRouteStatus = AleCliLib.routingHardwareRouteStatus
   else:
      routingHwConfig = AleCliLib.routing6HwConfig
      routingHwStatus = AleCliLib.routing6HwStatus
      routingHwRouteStatus = AleCliLib.routing6HardwareRouteStatus

   def _enabledState( enabled ):
      return stateEnum[ 0 ] if enabled else stateEnum[ 1 ]

   def _supportedState( supported ):
      return support[ 0 ] if supported else support[ 1 ]

   fibSummary = AleFibCliDynamicModel.FibSummaryStatus()

   # Attributes in both V4 and V6
   fibSummary.deletionDelay = routingHwRouteStatus.deletionDelay
   fibSummary.protectDefaultRoute = _enabledState(
         routingHwConfig.protectDefaultRoute )
   fibSummary.urpfSupport = _supportedState( routingHwStatus.urpfSupported )
   fibSummary.icmpStatus = _enabledState( routingHwConfig.icmpUnreachable )
   fibSummary.maxRoutes = routingHwRouteStatus.maxNumRoutes
   fibSummary.fibCompression = _enabledState( routingHwConfig.fibSuppression )
   fibSummary.maxFecHierarchyLevels = \
         AleCliLib.routingHardwareRouteStatus.hierarchicalFecsMaxLevel
   if v4:
      fibSummary.adjShare = _enabledState( routingHwRouteStatus.adjSharing )
      fibSummary.bfdEvent = _enabledState(
            routingHwRouteStatus.bfdPeerEventEnabled )
      fibSummary.pbrSupport = _supportedState( routingHwStatus.pbrSupported )
      fibSummary.maxAleEcmp = routingHwRouteStatus.effectiveMaxLogicalEcmp
      fibSummary.ucmpWeightDeviation = routingHwConfig.ucmpWeightDeviation
      fibSummary.adjResourceOptimization = \
         AleFibCliDynamicModel.AdjResourceOptimization()
      adjResourceOpt = fibSummary.adjResourceOptimization
      adjResourceOpt.supported = _supportedState(
         routingHwStatus.adjResourceOptimizationSupported )
      if routingHwStatus.adjResourceOptimizationSupported:
         adjResourceOpt.enabled = _enabledState(
               routingHwConfig.adjResourceOptimizationEnabled )
         adjResourceOpt.thresholds = \
            AleFibCliDynamicModel.AdjResourceOptimizationThresholds()
         thresholds = fibSummary.adjResourceOptimization.thresholds
         thresholdConfig = routingHwConfig.adjResourceOptimizationThresholds
         thresholds.low = thresholdConfig.stopOptimizationThreshold
         thresholds.high = thresholdConfig.startOptimizationThreshold

   return fibSummary

def doShowL3FibRoutesV4Diff( mode, args ):
   cmd = [ '/usr/bin/AleFibCliDiffUtil' ]

   if os.environ.get( 'SIMULATION_VMID' ):
      em = mode.session_.entityManager
      ( routeStatus, vrfMap, routeHwStatus, arpSmash, routing6HwRouteStatus,
        arpVrfMap ) = AleFibCliDiffUtilLib.mountEntities( em )

      AleFibCliDiffUtilLib.runCmd( em, routeStatus, vrfMap, routeHwStatus, arpSmash,
                                   routing6HwRouteStatus, arpVrfMap )
   else:
      Tac.run( cmd, stdout=sys.stdout, stderr=sys.stderr, asRoot=True )

def doShowL3FibRoutesV4( mode, args ):
   cmd = "FR4#"
   vrfName = args.get( 'VRF' )
   if vrfName:
      cmd += vrfName
   cmd += "#"
   prefix = args.get( 'PREFIX' )
   if prefix is not None:
      cmd += prefix.stringValue
   cmd += "#"
   return AgentCommandRequest.runCliPrintSocketCommand(
         AleCliLib.entityManager, "ale", "VrfRootSmCliCallbackWithFormat",
         cmd, mode=mode, asyncCommand=True, model=AleFibCliDynamicModel.FibRoute )

def doShowL3FibRoutesV6( mode, args ):
   cmd = "FR6#"
   vrfName = args.get( 'VRF' )
   if vrfName:
      cmd += vrfName
   cmd += "#"
   prefix = args.get( 'PREFIX' )
   if prefix is not None:
      cmd += prefix.stringValue
   cmd += "#"
   return AgentCommandRequest.runCliPrintSocketCommand(
         AleCliLib.entityManager, "ale", "VrfRootSmCliCallbackWithFormat",
         cmd, mode=mode, asyncCommand=True, model=AleFibCliDynamicModel.FibRoute )

def doShowL3FibFec( mode, args ):
   cmd = "FF4#" if 'ip' in args else "FF6#"
   vrfName = args.get( 'VRF' )
   if vrfName:
      cmd += vrfName
   cmd += "#"
   fecId = args.get( 'FECID' )
   if fecId:
      if not fecIdValid( fecId ):
         mode.addError( "fecId not valid" )
         return None
      cmd += str( fecId )
   cmd += "#"

   return AgentCommandRequest.runCliPrintSocketCommand(
         AleCliLib.entityManager, "ale", "VrfRootSmCliCallbackWithFormat",
         cmd, mode=mode, asyncCommand=True, model=AleFibCliDynamicModel.FibFec )

def getVrfNameAndResilientEcmpStatus( vrfName, ipv4 ):
   defaultVrfName = Tac.Type( 'L3::VrfName' ).defaultVrf
   if not vrfName:
      vrfName = defaultVrfName

   vrfTableStatus = AleCliLib.vrfTableStatus
   if vrfName not in vrfTableStatus.vrfName:
      return ( vrfName, None )

   if ipv4:
      resilientEcmpStatus = AleCliLib.resilientEcmpStatus
      mountPath = 'routing/vrf/resilientEcmpStatus/%s'
      mountType = 'Ale::ResilientEcmpStatus'
   else:
      resilientEcmpStatus = AleCliLib.resilientEcmpStatus6
      mountPath = 'routing6/vrf/resilientEcmpStatus/%s'
      mountType = 'Ale::ResilientEcmpStatus6'

   if vrfName != defaultVrfName:
      shmemEm = SharedMem.entityManager( sysdbEm=AleCliLib.entityManager )
      resilientEcmpStatus = shmemEm.doMount( mountPath % vrfName,
                                             mountType,
                                             AleCliLib.smashReaderInfo )

   return ( vrfName, resilientEcmpStatus )

def doShowResilientEcmpSummary( mode, args ):
   vrfName = args.get( 'VRF' )
   ipv4 = 'ip' in args

   vrfName, resilientEcmpStatus = getVrfNameAndResilientEcmpStatus( vrfName, ipv4 )
   if not resilientEcmpStatus:
      mode.addError( f"VRF { vrfName } not found" )
      return None

   programmedRoutes = 0
   unprogrammedRoutes = 0
   orderedRoutes = 0
   routePrefixToResEcmpInfo = resilientEcmpStatus.routePrefixToResilientEcmpInfo
   for pfx in routePrefixToResEcmpInfo:
      resilientEcmpInfo = routePrefixToResEcmpInfo[ pfx ]
      if resilientEcmpInfo.resilientFecId.localFecId > 0:
         programmedRoutes += 1
      else:
         unprogrammedRoutes += 1
      if resilientEcmpInfo.orderedVias:
         orderedRoutes += 1

   if ipv4:
      routingHwConfig = AleCliLib.routingHwConfig
   else:
      routingHwConfig = AleCliLib.routing6HwConfig

   aleResEcmpSummary = AleFibCliDynamicModel.AleResEcmpSummary()
   aleResEcmpSummary.vrfName = vrfName
   aleResEcmpSummary.numPrefixes = len( routingHwConfig.resilientEcmpPrefix )
   aleResEcmpSummary.numProgrammed = programmedRoutes
   aleResEcmpSummary.numUnprogrammed = unprogrammedRoutes
   aleResEcmpSummary.numOrdered = orderedRoutes

   return aleResEcmpSummary

def doShowL3AleDropRoutes( mode, args ):
   vrfDropRoutes = []
   af = 'ipv4' if 'ip' in args else 'ipv6'
   argsVrfName = args.get( 'VRF' )
   dropRoutes = AleFibCliDynamicModel.DropRouteStatus()

   def _enabledState( enabled ):
      return stateEnum[ 0 ] if enabled else stateEnum[ 1 ]

   dropRoutes.routingTelemetry = _enabledState(
                     AleCliLib.routingHwConfig.routingTelemetryEnabled )

   routeStatus = AleCliLib.routeMonitorSharkStatus
   if not routeStatus.vrfs:
      return dropRoutes
   if argsVrfName and routeStatus.vrfs.get( argsVrfName ) is None:
      return dropRoutes
   if argsVrfName and argsVrfName != 'all':
      if routeStatus.vrfs[ argsVrfName ].dropRoutes is None:
         return dropRoutes
      routeStateVrf = routeStatus.vrfs[ argsVrfName ]
      for dropRoute in routeStateVrf.dropRoutes.drop.values():
         prefix = dropRoute.prefix
         if prefix.af != af:
            continue
         vrfDropRoutes.append(
            AleFibCliDynamicModel.VrfDropRoute(
               vrfName=argsVrfName,
               insertTime=dropRoute.state.time.seconds(),
               prefix=prefix ) )
   else:
      for vrf in routeStatus.vrfs.values():
         if vrf.dropRoutes is None:
            continue
         for dropRoute in vrf.dropRoutes.drop.values():
            prefix = dropRoute.prefix
            if prefix.af != af:
               continue
            vrfDropRoutes.append(
               AleFibCliDynamicModel.VrfDropRoute(
                  vrfName=vrf.name,
                  insertTime=dropRoute.state.time.seconds(),
                  prefix=prefix ) )

   vrfDropRoutes = sorted(
         vrfDropRoutes, key=lambda x:( x[ 'vrfName' ], x[ 'prefix' ] ) )
   dropRoutes.vrfDropRoutes = vrfDropRoutes
   return dropRoutes

def doShowResilientEcmpStatus( mode, args ):
   # pylint: disable=protected-access
   def createAleResEcmpRouteInfo( info ):

      ReviewResilientAdjResult = Tac.Type( 'Ale::ReviewResilientAdjResult' )
      reasonStr = {
         ReviewResilientAdjResult.resEcmpSuccess:
            'Success',
         ReviewResilientAdjResult.noLongerResilientFailure:
            'Route no longer resilient',
         ReviewResilientAdjResult.noFecIdFailure:
            'No FEC ID found for route',
         ReviewResilientAdjResult.reviewAdjFailure:
            'Failed to review adjacency for route',
         ReviewResilientAdjResult.invalidViasFailure:
            'Route has invalid vias',
         ReviewResilientAdjResult.nonForwardRouteFailure:
            'Non-forward route not made resilient',
         ReviewResilientAdjResult.emptyViasFecFailure:
            'FEC has empty via set',
         ReviewResilientAdjResult.kernelRouteFailure:
            'Kernel route not made resilient',
         ReviewResilientAdjResult.unprogrammedRouteFailure:
            'Route is not programmed',
         ReviewResilientAdjResult.mixedAfFecUnsupported:
            'Mixed IPv4/IPv6 route cannot be made resilient',
      }

      result = AleFibCliDynamicModel.AleResEcmpRouteInfo()
      result.prefix = Arnet.IpGenPrefix( str( info.key ) )
      result.resilient = info.resilientFecId.localFecId != 0
      result.fecId = info.fibFecId.localFecId
      result.resFecId = info.resilientFecId.localFecId if result.resilient else 0
      result.orderedVias = info.orderedVias
      result.reason = reasonStr.get( info.reviewResilientAdjResult, '' )

      return result

   def updateAllOrSpecificOutput( prefix, routePrefixToResEcmpInfo ):
      coveringPrefix = \
         Arnet.IpGenPrefix( str( routePrefixToResEcmpInfo.coveringResEcmpPrefix ) )

      routeInfo = createAleResEcmpRouteInfo( routePrefixToResEcmpInfo )

      routeInfoList = AleFibCliDynamicModel.AleResEcmpRouteInfos()

      if routePrefixToResEcmpInfo.policyBasedResilientEcmp:
         resEcmpConfig = routingHwConfig.policyBasedResilientEcmp
         if output.policyBasedResRouteInfos:
            routeInfoList = output.policyBasedResRouteInfos
      else:
         if not coveringPrefix in routingHwConfig.resilientEcmpPrefix:
            mode.addErrorAndStop(
               f"RECMP config not found for"
               f" covering prefix { coveringPrefix } of { prefix }" )
            return

         resEcmpConfig = routingHwConfig.resilientEcmpPrefix[ coveringPrefix ]
         if coveringPrefix in output.resPrefixesToRouteInfos:
            routeInfoList = output.resPrefixesToRouteInfos[ coveringPrefix ]
      if not output._oneRoute or routeInfo.resilient:
         routeInfoList.capacity = resEcmpConfig.capacity
         routeInfoList.redundancy = resEcmpConfig.redundancy
         routeInfoList.orderedVias = resEcmpConfig.orderedVias

      routeInfoList.routes.append( routeInfo )

      if routePrefixToResEcmpInfo.policyBasedResilientEcmp:
         output.policyBasedResRouteInfos = routeInfoList
      else:
         output.resPrefixesToRouteInfos[ coveringPrefix ] = routeInfoList

   vrfName = args.get( 'VRF' )
   prefix = args.get( 'PREFIX' )
   ipv4 = 'ip' in args

   vrfName, resilientEcmpStatus = getVrfNameAndResilientEcmpStatus( vrfName, ipv4 )
   if not resilientEcmpStatus:
      mode.addError( f"VRF { vrfName } not found" )
      return None

   output = AleFibCliDynamicModel.AleResEcmpAllOrSpecific()
   output._oneRoute = prefix is not None

   output.vrfName = vrfName
   if ipv4:
      routingHwConfig = AleCliLib.routingHwConfig
   else:
      routingHwConfig = AleCliLib.routing6HwConfig

      # The matcher consistently returns an Ip6AddrWithMask
      prefix = Arnet.Ip6Prefix( str( prefix ) ) if output._oneRoute else None

   if output._oneRoute:
      # Specific prefix case
      recmpInfo = resilientEcmpStatus.routePrefixToResilientEcmpInfo.get( prefix )
      if not recmpInfo:
         mode.addError( f"Route { prefix } not found" )
         return None

      updateAllOrSpecificOutput( prefix, recmpInfo )
   else:
      # All prefixes case
      for pfx, recmpInfo in \
            resilientEcmpStatus.routePrefixToResilientEcmpInfo.items():
         updateAllOrSpecificOutput( pfx, recmpInfo )

   # pylint: enable=protected-access
   return output

def doShowL3AleInconsistentRoutes( mode, args ):
   vrfInconsistentRoutes = []
   af = 'ipv4' if 'ip' in args else 'ipv6'
   argsVrfName = args.get( 'VRF' )
   inconsistentRoutes = AleFibCliDynamicModel.InconsistentRouteStatus()

   def _enabledState( enabled ):
      return stateEnum[ 0 ] if enabled else stateEnum[ 1 ]

   inconsistentRoutes.routingTelemetry = _enabledState(
         AleCliLib.routingHwConfig.routingTelemetryEnabled )

   routeStatus = AleCliLib.routeMonitorSharkStatus
   if not routeStatus.vrfs:
      return inconsistentRoutes
   if argsVrfName and routeStatus.vrfs.get( argsVrfName ) is None:
      return inconsistentRoutes
   if argsVrfName and argsVrfName != 'all':
      if routeStatus.vrfs[ argsVrfName ].staleRoutes is None:
         return inconsistentRoutes
      routeStateVrf = routeStatus.vrfs[ argsVrfName ]
      for inconsistentRoute in routeStateVrf.staleRoutes.stale.values():
         prefix = inconsistentRoute.prefix
         if prefix.af != af:
            continue
         vrfInconsistentRoutes.append(
            AleFibCliDynamicModel.VrfRouteEntry(
               vrfName=argsVrfName,
               insertTime=inconsistentRoute.state.time.seconds(),
               prefix=prefix ) )
   else:
      for vrf in routeStatus.vrfs.values():
         if vrf.staleRoutes is None:
            continue
         for inconsistentRoute in vrf.staleRoutes.stale.values():
            prefix = inconsistentRoute.prefix
            if prefix.af != af:
               continue
            vrfInconsistentRoutes.append(
               AleFibCliDynamicModel.VrfRouteEntry(
                  vrfName=vrf.name,
                  insertTime=inconsistentRoute.state.time.seconds(),
                  prefix=prefix ) )

   vrfInconsistentRoutes = sorted(
         vrfInconsistentRoutes, key=lambda x:( x[ 'vrfName' ], x[ 'prefix' ] ) )
   inconsistentRoutes.vrfInconsistentRoutes = vrfInconsistentRoutes
   return inconsistentRoutes

def doShowL3AleAdjProxy( mode, args ):
   cmd = 'AAP#'
   vrfName = args.get( 'VRF' )
   if vrfName:
      cmd += vrfName
   cmd += '#'
   fecId = args.get( 'FECID' )
   if fecId:
      if not fecIdValid( fecId ):
         mode.addError( "fecId not valid" )
         return None
      cmd += str( fecId )
   cmd += '#'
   if 'proxy' in args:
      cmd += 'proxy'
   cmd += '#'
   return AgentCommandRequest.runCliPrintSocketCommand(
         AleCliLib.entityManager, "ale", "VrfRootSmCliCallbackWithFormat",
         cmd, mode=mode, model=AleFibCliDynamicModel.FecProxyInfoList )

def doShowL3AleL2Adj( mode, args ):

   cmd = "AA2#"
   adjIndex = args.get( 'ADJ_INDEX' )
   if adjIndex:
      cmd += str( adjIndex )
   cmd += '#'

   return AgentCommandRequest.runCliPrintSocketCommand(
         AleCliLib.entityManager, "ale", "VrfRootSmCliCallbackWithFormat",
         cmd, mode=mode, asyncCommand=True, model=AleL2Adj )

def doShowAleL2AdjCounterBindings( mode, args ):
   # CEB2# (Counter-Egress-Bindings-l2)
   cmd = "CEB2#"
   adjIndex = args.get( 'ADJ_INDEX' )
   if adjIndex:
      cmd += str( adjIndex )
   cmd += '#'

   return AgentCommandRequest.runCliPrintSocketCommand(
         AleCliLib.entityManager, "ale", "VrfRootSmCliCallbackWithFormat",
         cmd, mode=mode, asyncCommand=True,
         model=AleFibCliDynamicModel.AleL2AdjCounterBindings )

def doShowAlePdckCounterBindings( mode, args ):
   # PDCK Key is in the value1/value2 format
   # cmd code is CEB3# (Counter-Egress-Bindings-Pdck)
   pdck_value = args.get( 'PDCK', '' )
   cmd = f"CEB3#{ pdck_value }#"

   # its not an async agent command request
   return AgentCommandRequest.runCliPrintSocketCommand(
         AleCliLib.entityManager, "ale", "VrfRootSmCliCallbackWithFormat",
         cmd, mode=mode, model=AleFibCliDynamicModel.AlePdckCounterBindings )
