# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from ArnetModel import (
      IpGenericPrefix,
   )
from CliModel import (
      Bool,
      Dict,
      Int,
      Model,
      List,
      Str,
      Submodel,
   )
from IpLibConsts import (
      DEFAULT_VRF,
   )
import TableOutput
import Tac

largeInt = int

def operationStr( requestedDelete, failed ):
   operStr = []
   if failed:
      operStr.append( "failed" )
   if requestedDelete:
      operStr.append( "delete" )

   return "/".join( operStr ) or "add/modify"

class AcknowledgementRouteVersionId( Model ):
   versionId = Int( help="Route Version Id to be acknowledged" )
   requestedDelete = Bool( help="Route due to be deleted" )
   failed = Bool( help="Route install failed" )

class AcknowledgementRouteEntry( Model ):
   requested = Submodel( optional=True,
                         valueType=AcknowledgementRouteVersionId,
                         help="Requested Route Version Id" )
   acknowledged = Submodel( optional=True,
                            valueType=AcknowledgementRouteVersionId,
                            help="Acknowledged Route Version Id" )

class AcknowledgementRouteStatus( Model ):
   cmdErr = Str( optional=True, help="Error Message" )
   vrfName = Str( optional=True, help="VRF name" )
   fibRoutes = Dict( keyType=IpGenericPrefix,
                    valueType=AcknowledgementRouteEntry,
                    help="Mapping prefix to its FibRoute Info" )

   def render( self ):
      table = TableOutput.createTable( ( "Prefix",
                                         "Requested ID",
                                         "Requested Operation",
                                         "Acknowledged ID",
                                         "Acknowledged Operation" )
                                     )
      tableMaxWidth = table.printableWidth()
      tableMaxWidth = max( tableMaxWidth, 80 )

      f1 = TableOutput.Format( justify="left", maxWidth=50, minWidth=12 )
      f1.noPadLeftIs( True )
      f1.padLimitIs( True )

      f2 = TableOutput.Format( justify="right", maxWidth=9, minWidth=6, wrap=True )
      f2.padLimitIs( True )

      f3 = TableOutput.Format( justify="left", maxWidth=10, minWidth=6, wrap=True )
      f3.noPadLeftIs( True )
      f3.padLimitIs( True )

      f4 = TableOutput.Format( justify="right", maxWidth=12, minWidth=6, wrap=True )
      f4.padLimitIs( True )

      f5 = TableOutput.Format( justify="left", maxWidth=12, minWidth=6, wrap=True )
      f5.noPadLeftIs( True )
      f5.padLimitIs( True )

      table.formatColumns( f1, f2, f3, f4, f5 )

      for prefix, fibRoute in sorted( self.fibRoutes.items() ):
         requestedVersion = ""
         requestedOperation = ""

         if fibRoute.requested:
            requested = fibRoute.requested
            requestedVersion = requested.versionId
            requestedOperation = operationStr( requested.requestedDelete,
                                               requested.failed )

         acknowledgedVersion = ""
         acknowlegedOperation = ""

         if fibRoute.acknowledged:
            acknowledged = fibRoute.acknowledged
            acknowledgedVersion = acknowledged.versionId
            acknowlegedOperation = operationStr( acknowledged.requestedDelete,
                                                 acknowledged.failed )

         table.newRow(
               prefix,
               requestedVersion,
               requestedOperation,
               acknowledgedVersion,
               acknowlegedOperation )

      if not self.vrfName:
         self.vrfName = DEFAULT_VRF

      print( table.output() )

class AcknowledgementFecVersionId( Model ):
   versionId = Int( help="FEC version Id to be acknowledged" )
   requestedDelete = Bool( help="FEC is due to be deleted" )
   failed = Bool( help="FEC install failed" )

class AcknowledgementFecEntry( Model ):
   requested = Submodel( optional=True,
                         valueType=AcknowledgementFecVersionId,
                         help="FEC version Id to be acknowledged" )
   acknowledged = Submodel( optional=True,
                            valueType=AcknowledgementFecVersionId,
                            help="FEC version Id that was acknowledged" )

class AcknowledgementFecStatus( Model ):
   cmdErr = Str( optional=True, help="Error Message" )
   vrfName = Str( help="VRF name" )
   fecEntries = Dict( keyType=largeInt,
                      valueType=AcknowledgementFecEntry,
                      help="Mapping fecId to its version ID info" )

   def render( self ):
      table = TableOutput.createTable( ( "FecId",
                                         "Adj Type",
                                         "Requested ID",
                                         "Requested Operation",
                                         "Acknowledged ID",
                                         "Acknowledged Operation" ) )
      tableMaxWidth = table.printableWidth()
      tableMaxWidth = max( tableMaxWidth, 80 )

      f1 = TableOutput.Format( justify="left", maxWidth=50, minWidth=12 )
      f1.noPadLeftIs( True )
      f1.padLimitIs( True )

      f2 = TableOutput.Format( justify="left", maxWidth=25, minWidth=6 )
      f2.noPadLeftIs( True )
      f2.padLimitIs( True )

      f3 = TableOutput.Format( justify="right", maxWidth=9, minWidth=6, wrap=True )
      f3.padLimitIs( True )

      f4 = TableOutput.Format( justify="left", maxWidth=10, minWidth=6, wrap=True )
      f4.noPadLeftIs( True )
      f4.padLimitIs( True )

      f5 = TableOutput.Format( justify="right", maxWidth=12, minWidth=6, wrap=True )
      f5.padLimitIs( True )

      f6 = TableOutput.Format( justify="left", maxWidth=12, minWidth=6, wrap=True )
      f6.noPadLeftIs( True )
      f6.padLimitIs( True )

      table.formatColumns( f1, f2, f3, f4, f5, f6 )

      for fecId, fecEntry in sorted( self.fecEntries.items() ):
         requestedVersion = ""
         requestedOperation = ""

         if fecEntry.requested:
            requested = fecEntry.requested
            requestedVersion = requested.versionId
            requestedOperation = operationStr( requested.requestedDelete,
                                               requested.failed )

         acknowledgedVersion = ""
         acknowlegedOperation = ""

         if fecEntry.acknowledged:
            acknowledged = fecEntry.acknowledged
            acknowledgedVersion = acknowledged.versionId
            acknowlegedOperation = operationStr( acknowledged.requestedDelete,
                                                 acknowledged.failed )

         smashFecId = Tac.Type( 'Smash::Fib::FecId' )( fecId )
         # truncate the "Adj" from the tunnelFibAdj or fibGenAdj
         # before printing the adj Type.
         adjType = smashFecId.adjType()
         if adjType[ -3 : ] == "Adj":
            adjType = adjType[ : -3 ]

         table.newRow(
               fecId,
               adjType,
               requestedVersion,
               requestedOperation,
               acknowledgedVersion,
               acknowlegedOperation )

      print( 'VRF: ' + self.vrfName )
      print( table.output() )

class VlanListType( Model ):
   vlans = List( valueType=int, help="VLAN IDs" )

class GatewayMacStatus( Model ):
   macs = Dict( keyType=str, valueType=VlanListType,
                     help="Mapping MAC address to its VLANs" )

   def render( self ):
      table = TableOutput.createTable( ( "VLAN",
                                         "MAC Address" )
                                     )
      tableMaxWidth = table.printableWidth()
      tableMaxWidth = max( tableMaxWidth, 80 )

      f1 = TableOutput.Format( justify="right", maxWidth=12, minWidth=4 )
      f1.noPadLeftIs( True )
      f1.padLimitIs( True )

      f2 = TableOutput.Format( justify="left", maxWidth=25, minWidth=14, wrap=True )
      f2.padLimitIs( True )

      table.formatColumns( f1, f2 )

      for mac in self.macs:
         for vlan in self.macs[ mac ].vlans:
            table.newRow(
                  vlan,
                  mac, )

      print( table.output() )
