# Copyright (c) 2006-2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import sys
import io
import re

import ConfigMount
import BasicCliSession
import BasicCliModes
import Tac
import difflib
from TypeFuture import TacLazyType
from CliPlugin import CliCliModel
from CliPlugin.AliasCli import AliasConfigMode, maybeAddAliasSessionOnCommitHandler

Alias = TacLazyType( 'Cli::Config::Alias' )

cliConfig = None

def verifyAllowedAliasOrStop( mode, alias ):
   '''We don't allow aliasing of certain keywords or empty regex aliases.'''
   alias = alias.lower()
   if alias in ( 'no', 'default', 'alias', '' ):
      mode.addErrorAndStop( f'{alias!r} is not allowed as an alias name' )

def compileRegexOrStop( mode, pattern ):
   '''Compile regex pattern to make sure it's valid.
   Stop further handler execusion if it's not.
   '''
   verifyAllowedAliasOrStop( mode, pattern )
   try:
      return re.compile( pattern )
   except re.error as e:
      mode.addErrorAndStop( str( e ) )
      return None

#------------------------------------------------------------------------------------
# alias ALIAS | REGEX
#------------------------------------------------------------------------------------
def gotoAliasMode( mode, args ):
   alias = args[ 'ALIAS' ]
   verifyAllowedAliasOrStop( mode, alias )
   childMode = mode.childMode( AliasConfigMode, cmdAlias=alias )
   mode.session_.gotoChildMode( childMode )

def gotoRegexAliasMode( mode, args ):
   pattern = args[ 'REGEX' ]
   regex = compileRegexOrStop( mode, pattern )
   childMode = mode.childMode( AliasConfigMode, cmdAlias=pattern, regex=regex )
   mode.session_.gotoChildMode( childMode )

#------------------------------------------------------------------------------------
# alias commands path PATH
#-----------------------------------------------------------------------------------
def onSessionCommitFn( mode, onSessionCommit ):
   if onSessionCommit and mode.session_ and not mode.session_.isEapiClient():
      mode.session_.loadDynamicAliases()

def commandsPathCmd( mode, args ):
   cmdsPath = args.get( 'PATH', '' )
   if mode.session_:
      cliConfig.commandsPath = cmdsPath
      if not mode.session_.isEapiClient():
         mode.session_.loadDynamicAliases()
      mode.session_.maybeCallConfigSessionOnCommitHandler( "commandsPath",
                                                           onSessionCommitFn )

#------------------------------------------------------------------------------------
# alias ALIAS COMMAND
#------------------------------------------------------------------------------------
def removeAlias( mode, cmdAlias, collection ):
   '''Remove an alias from a given collection and any associated comments.
   '''
   lastMode = mode.session_.modeOfLastPrompt()
   if ( isinstance( lastMode, AliasConfigMode ) and lastMode.alias
        and lastMode.alias.cmdAlias == cmdAlias ):
      # If we are deleting ourselves, make sure we don't write it back when
      # we exit.
      lastMode.alias = None
   # Remove associated comment if any. commentKey should be same as what user
   # entered initially.
   alias = collection.get( cmdAlias )
   if alias:
      BasicCliModes.removeCommentWithKey( 'alias-' + alias.cmdAlias )
      del collection[ cmdAlias ]

def modifyAliasCmd( mode, args ):
   alias = args[ 'ALIAS' ]
   command = args[ 'COMMAND' ]
   verifyAllowedAliasOrStop( mode, alias )
   theAlias = Alias( alias )
   theAlias.originalCmd[ AliasConfigMode.INC_STEP ] = command
   cliConfig.alias[ alias.lower() ] = theAlias

def modifyAliasCmdNot( mode, args ):
   removeAlias( mode, args[ 'ALIAS' ].lower(), cliConfig.alias )

#------------------------------------------------------------------------------------
# alias REGEX TEMPLATE
#------------------------------------------------------------------------------------
def modifyRegexAliasCmd( mode, args ):
   pattern = args[ 'REGEX' ]
   regex = compileRegexOrStop( mode, pattern )
   index = BasicCliSession.getRegexAliasIndex( pattern, cliConfig.regexAlias )
   if index:
      theAlias = Tac.nonConst( cliConfig.regexAlias[ index ] )
   else:
      index = BasicCliSession.addCompiledRegex( regex, cliConfig.regexAlias )
      theAlias = Alias( regex.pattern )

   theAlias.originalCmd[ AliasConfigMode.INC_STEP ] = args[ 'TEMPLATE' ]
   cliConfig.regexAlias[ index ] = theAlias
   maybeAddAliasSessionOnCommitHandler( mode )

def modifyRegexAliasCmdNot( mode, args ):
   BasicCliSession.delRegexAlias( args[ 'REGEX' ], cliConfig.regexAlias )
   maybeAddAliasSessionOnCommitHandler( mode )

#------------------------------------------------------------------------------------
# command COMMAND
#------------------------------------------------------------------------------------
def aliasAddCmd( mode, args ):
   originalCmd = args[ 'COMMAND' ]
   if len( mode.alias.originalCmd ) < mode.MAX_CMD:
      theSeq = max( mode.alias.originalCmd, default=0 ) + mode.INC_STEP
      mode.alias.originalCmd[ theSeq ] = originalCmd
   else:
      mode.addError( f"Exceeding maximum number of commands allowed: {mode.MAX_CMD}"
                    )

def aliasRemoveCmd( mode, args ):
   arg = args[ 'COMMAND' ]
   for seq, cmd in mode.alias.originalCmd.items():
      if cmd == arg:
         del mode.alias.originalCmd[ seq ]
         return

#------------------------------------------------------------------------------------
# [ no | default ] SEQ COMMAND
#------------------------------------------------------------------------------------
def aliasEditCmd( self, args ):
   self.alias.originalCmd[ args[ 'SEQ' ] ] = args[ 'COMMAND' ]

def aliasEditCmdNot( self, args ):
   del self.alias.originalCmd[ args[ 'SEQ' ] ]

#------------------------------------------------------------------------------------
# abort
#------------------------------------------------------------------------------------
def aliasAbortCmd( self, args ):
   self.removeComment()
   self.alias = None
   self.session_.gotoParentMode()

#------------------------------------------------------------------------------------
# show [ pending ]
#------------------------------------------------------------------------------------
def _showList( mode, alias, output=None ):
   if output is None:
      output = sys.stdout
   if alias is None or not alias.originalCmd:
      return
   cmd = alias.cmdAlias
   if mode.compiledRegex: # Quote alias if regex.
      cmd = f'"{cmd}"'
   output.write( f'alias {cmd}\n' )
   for seq, oCmd in alias.originalCmd.items():
      output.write( f'   {seq}\t{oCmd}\n' )

def showPending( mode, args ):
   _showList( mode, mode.alias )

#------------------------------------------------------------------------------------
# show active
#------------------------------------------------------------------------------------
def showActive( mode, args ):
   _showList( mode, mode.getAliasConfig( mode.alias.cmdAlias.lower() ) )

#------------------------------------------------------------------------------------
# show diff
#------------------------------------------------------------------------------------
def showDiff( mode, args ):
   # generate diff between active and pending
   activeOutput = io.StringIO()
   _showList( mode, mode.getAliasConfig( mode.alias.cmdAlias.lower() ),
               output=activeOutput )
   pendingOutput = io.StringIO()
   _showList( mode, mode.alias, output=pendingOutput )
   diff = difflib.unified_diff( activeOutput.getvalue().splitlines(),
                                pendingOutput.getvalue().splitlines(), lineterm='' )
   print( '\n'.join( diff ) )

# -----------------------------------------------------------------------------------
# show aliases
# -----------------------------------------------------------------------------------
def _createAliasesModel( aliasCollection ):
   AliasModel = CliCliModel.Aliases.Alias
   return { alias.cmdAlias: AliasModel( commands=dict( alias.originalCmd ) )
            for alias in aliasCollection.values() }

def showAliases( mode, args ):
   testString = args.get( 'STRING' )
   if testString:
      aliases = {}
      dynamicAliases = {}
   else:
      aliases = _createAliasesModel( cliConfig.alias )
      dynamicAliases = _createAliasesModel( cliConfig.dynamicAlias )

   # For regex aliases, order matters,
   # and we filter them if 'STRING' has been specified.
   regexIndices = [ i for i, alias in cliConfig.regexAlias.items()
                    if not testString or re.match( alias.cmdAlias, testString ) ]

   regexAliases = []
   RegexAlias = CliCliModel.Aliases.RegexAlias
   for index in sorted( regexIndices ):
      alias = cliConfig.regexAlias[ index ]
      regexAliases.append( RegexAlias( regex=alias.cmdAlias,
                                       commands=dict( alias.originalCmd ) ) )

   return CliCliModel.Aliases( commandsPath=cliConfig.commandsPath,
                               aliases=aliases,
                               regexAliases=regexAliases,
                               dynamicAliases=dynamicAliases )

#------------------------------------------------------------------------------------
# Plugin Func
#------------------------------------------------------------------------------------
def Plugin( entityManager ):
   global cliConfig

   cliConfig = ConfigMount.mount( entityManager, 'cli/config', 'Cli::Config', 'w' )
