#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
# pylint: disable=consider-using-f-string
# pylint: disable=use-a-generator
# pylint: disable=useless-object-inheritance
# pylint: disable=no-else-continue
# pylint: disable=consider-using-in
# pylint: disable=import-outside-toplevel
import Arnet
from BgpLib import (
   doLogClearIpBgp,
   bgpConfigAttrAfList,
   createKey,
   PeerConfigKey,
)
from Bunch import Bunch
import Cell
from CliCommon import AlreadyHandledError
from CliMode import BgpMacVrfConfigMode
from CliPlugin import ArBgpCliCommon
from CliPlugin import IntfCli
from CliPlugin import ShowTaskSchedulerModel
from CliPlugin.ArBgpCli import (
   ArBgpAsyncCliCommand,
   ArBgpCliCommand,
   SessionTrackerConfigMode
)
from CliPlugin.ArBgpCliModels import (
   BgpRfdPolicy,
   BgpRfdPolicies,
   BgpSummaryAll,
)
from CliPlugin.BgpCliModels import (
   ShowNeighborsHistoryModel,
   ShowNeighborsHistorySocketModel,
   BgpOrrPosition,
   BgpOrrPositionByVrf,
   BgpOrrPositionSource,
   ShowBgpConvergenceModel,
   ShowTrafficPolicyFieldSetCommunityMappingsModel,
   ShowVrfSelectionPolicyFieldSetCommunityMappingsModel,
)
from CliPlugin.IraServiceCli import getEffectiveProtocolModel
from CliPlugin.MaintenanceModels import (
    MaintenanceInterfaceBgpPeerStatus,
    MaintenanceInterfaceBgpStatus
)
from CliPlugin.RcfLibCliModels import (
   ShowAgentRcfDetailModel,
)
import CliPlugin.RcfCliLib
from CliPlugin.RoutingBgpShowCli import (
   ArBgpShowOutput,
   AllBgpPeerGroups,
   doShowBgpOutputNotSupported,
   doShowIpBgpFieldSet,
   peerListVrfModel,
   routeSummaryVrfModel,
   summaryVrfModel,
   statisticsVrfModel,
   showBgpCombinedRegexpHelper,
   queuedWithdrawalsVrfModel,
   cliShowBgpInstanceVpnSummaryVrfModel,
)
from CliPlugin.RouteMapCli import routeMapContainerModel
from CliPlugin import VrfCli
from CliPlugin.PolicyEvalModels import (
   createPolicyEvalModel,
   RcfDebugStateOutOfSync,
)

import ConfigMount
import LazyMount
from io import StringIO
import SharedMem
import Smash
import Tac
import Tracing
import difflib
import itertools
import json
import math

from natsort import natsorted
from ArnetLib import bgpFormatAsn, formatRd
from CliDynamicSymbol import CliDynamicPlugin
from CliModel import cliPrinted
from IpLibConsts import DEFAULT_VRF
from IpLibTypes import ProtocolAgentModelType as ProtoAgentModel
from RouteMapLib import matchPermitEnum, printRouteMapEntryAttributes

from CliSavePlugin.BgpMacVrfConfigCliSave import saveMacVrfConfig
from CliSavePlugin.RoutingBgpCliSave import (
   saveAfConfig,
   saveAfConfigCallbackDict,
   showBgpConfigInstanceCallbacks,
   saveBgpConfig,
   saveBgpVpnAfConfig,
   saveBgpVpnConfig,
   saveUcmpConfig,
   getLsImportCmds,
   getLsRoleCmds,
)

from TypeFuture import TacLazyType

ArBgpCliModels = CliDynamicPlugin( "ArBgpCliModels" )
BgpMaintenanceModels = CliDynamicPlugin( "BgpMaintenanceModels" )
BgpMaintenanceShowCliHandler = CliDynamicPlugin( "BgpMaintenanceShowCliHandler" )
DynamicPrefixListModel = CliDynamicPlugin( "DynamicPrefixListModel" )
RoutingBgpShowCliHandler = CliDynamicPlugin( "RoutingBgpShowCliHandler" )

traceHandle = Tracing.Handle( 'ArBgpCliHandler' )
t0 = traceHandle.trace0
t1 = traceHandle.trace1

routingHwStatusCommon = None
routingHwStatus = None
config = None
vrfConfigDir = None
mapConfig = None
asnConfig = None
allIntfStatusDir = None
arBgpMapConfig = None
arBgpConfig = None
arBgpVrfConfigDir = None
macVrfConfigDir = None
l3Config = None
securityConfig = None
ucmpConfig = None
ucmpVrfConfigDir = None
arUcmpConfig = None
arUcmpVrfConfigDir = None
aclCpConfig = None
rdConfigDir = None
lsImportConfig = None
lsRoleConfig = None
orrPositionStatusRoot = None
orrPositionsSourceRoot = None
defaultReceiverRmDir = None
sessionTrackerConfigDir = None
sessionTrackerIntfConfigDir = None
agentBgpThreadConfig = None
dynPfxListConfigDir = None
dynPfxListStatusDir = None
vrfNameStatus = None
allVrfRmStatusSet = None
allVrfPolicyStateSet = None
rtrGeneralConfigDir = None
peerInfoTable = None
rcfStatus = None
defaultInitiatorRmDir = None
commandTagIdState = None
routerGeneralConfig = None

GenPolicyName = TacLazyType( 'Routing::Policy::GenPolicyName' )
PolicyType = TacLazyType( 'Routing::Policy::PolicyType' )
defaultRmName = TacLazyType( 'BgpMaintenance::Defaults' ).rmName
BgpPeerState = Tac.Type( 'Routing::Bgp::BgpPeerState' )
sessionTrackerDefaults = TacLazyType( 'Routing::Bgp::SessionTrackerDefaults' )

def convertPeerAddr( kwargs ):
   key = 'peerAddr'
   peerAddr = kwargs.get( key, None )
   llIntf = ''
   if peerAddr is not None:
      peerKey = PeerConfigKey( peerAddr )
      if peerKey.type == 'peerIpv4':
         peerAddr = peerKey.v4Addr
      elif peerKey.type in [ 'peerIpv6', 'peerIpv6Ll' ]:
         peerAddr = peerKey.v6Addr.stringValue
         llIntf = Arnet.IntfId( peerKey.llIntf ).stringValue
         kwargs[ 'llIntf' ] = llIntf
      kwargs[ key ] = peerAddr

def handlerEnterSessionTrackerConfigMode( mode, args ):
   trackerName = args[ 'NAME' ]
   childMode = mode.childMode( SessionTrackerConfigMode,
                               trackerName=trackerName,
                               mode=mode )
   mode.session_.gotoChildMode( childMode )

def noOrDefaultHandlerEnterSessionTrackerConfigMode( mode, args ):
   trackerName = args[ 'NAME' ]
   del sessionTrackerConfigDir.sessionTrackerConfig[ trackerName ]

def handlerSessionTrackerRecoveryDelayCmd( mode, args ):
   seconds = args[ 'DELAYSEC' ]
   mode.updateRecoveryDelay( seconds )

def noOrDefaultHandlerSessionTrackerRecoveryDelayCmd( mode, args ):
   seconds = sessionTrackerDefaults.recoveryDelaySec
   mode.updateRecoveryDelay( seconds, False )

def handlerSessionTrackerNeighborAllCmd( mode, args ):
   mode.updateNeighborAll( True )

def noOrDefaultHandlerSessionTrackerNeighborAllCmd( mode, args ):
   mode.updateNeighborAll( sessionTrackerDefaults.trackAllPeersUp )

def handlerSessionTrackerIntfConfigCmd( mode, args ):
   intf = mode.intf.name
   trackerName = args[ 'NAME' ]
   intfConfig = sessionTrackerIntfConfigDir.intfConfig.get( intf, None )
   if intfConfig is None:
      intfConfig = sessionTrackerIntfConfigDir.newIntfConfig( intf )
   intfConfig.sessionTracker = trackerName

def noOrDefaultHandlerSessionTrackerIntfConfigCmd( mode, args ):
   intf = mode.intf.name
   del sessionTrackerIntfConfigDir.intfConfig[ intf ]

def handlerShowBgpSchedulerReset( mode, args ):
   # Since it's deprecated, this should never get called, but
   # if it is, let's make sure that the user knows that it didn't
   # work.
   mode.addError( 'Use "clear agent bgp task scheduler"' )

def handlerShowIpBgpFieldSet( mode, args ):
   return doShowIpBgpFieldSet(
      mode,
      afi='ipv4',
      fieldSetName=args[ "FSNAME" ],
      detail=args.get( "detail" ),
      vrfName=args.get( "VRF" ) )

def handlerShowIpv6BgpFieldSet( mode, args ):
   return doShowIpBgpFieldSet(
      mode,
      afi='ipv6',
      fieldSetName=args[ "FSNAME" ],
      detail=args.get( "detail" ),
      vrfName=args.get( "VRF" ) )

def handlerShowBgpIpFieldSet( mode, args ):
   return doShowIpBgpFieldSet(
      mode,
      afi=args[ "AFI" ],
      fieldSetName=args[ "FSNAME" ],
      detail=args.get( "detail" ),
      vrfName=args.get( "VRF" ) )

def handlerShowIpBgpAndRegexp( mode, args ):
   return doShowIpBgpAndRegexp( mode, aspRegex=args[ "ASPREGEX" ],
                                commRegex=args[ "COMMREGEX" ],
                                vrfName=args.get( "VRF" ) )

@ArBgpShowOutput( 'doShowIpBgpAndRegexp', arBgpModeOnly=True, vrfLookup=True )
def doShowIpBgpAndRegexp( mode, aspRegex=None, commRegex=None, vrfName=None ):
   result = showBgpCombinedRegexpHelper( mode, aspRegex, commRegex, vrfName=vrfName )
   return result

# -------------------------------------------------------------------------------
# "[ no | default ] bgp session tracker < name >" config command in
# "config-if" mode.
# -------------------------------------------------------------------------------
class SessionTrackerIntf( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      del sessionTrackerIntfConfigDir.intfConfig[ self.intf_.name ]

def getIsisHostname( mode, systemId, instance=0, vrfName='default' ):
   '''
   Return hostname for the Isis system-id if available
   else return the Isis system-id as string
   '''
   shmemEm = SharedMem.entityManager( sysdbEm=mode.entityManager )
   smi = Smash.mountInfo( 'reader' )
   path = 'routing/isis/hostnamemap/%s' % vrfName
   hnameEnt = shmemEm.doMount( path, 'IsisExportImpl::IsisHostnameMap',
                               smi )

   # Construct a routerId
   rId = Tac.Value( "Mpls::RouterId" )
   rId.stringValue = systemId
   rId.igpInstanceId = instance
   hostnameArray = hnameEnt.hostnameMap.get( rId )
   return hostnameArray.smashString() if hostnameArray else str( systemId )
# pylint: enable=consider-using-f-string

def populatePositionSource( mode, positionName, positionModel, vrfName,
                            orrPositionsSourceByVrf ):
   if not orrPositionsSourceByVrf:
      positionModel.sources = None
      return
   orrPositionSource = orrPositionsSourceByVrf.orrPositionSource.get( positionName )
   if not orrPositionSource:
      positionModel.sources = None
      return
   for topo, igpRtrId in sorted( orrPositionSource.source.items() ):
      bgpOrrPositionSource = BgpOrrPositionSource()
      # Only isis supported now
      bgpOrrPositionSource.protocol = 'isis'
      if igpRtrId:
         source = getIsisHostname( mode, igpRtrId.stringValue, topo.instance,
                                   vrfName=vrfName )
      else:
         source = 'local'
      bgpOrrPositionSource.source = source
      bgpOrrPositionSource.instance = topo.instance
      bgpOrrPositionSource.isisLevel = topo.protoId
      positionModel.sources.append( bgpOrrPositionSource )

# -------------------------------------------------------------------------------
# show bgp optimal-route-reflection position
# -------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowBgpOrrPosition', arBgpModeOnly=True, requiresAgent=False )
def handlerDoShowBgpOrrPosition( mode, args ):
   model = BgpOrrPositionByVrf()
   model.vrfName = Tac.Type( 'Vrf::Constants' ).defaultVrfName
   defaultVrfId = Tac.Type( 'Vrf::VrfIdMap::VrfId' ).defaultVrf
   orrPositionStatusByVrf = \
      orrPositionStatusRoot.orrPositionStatusByVrf.get( defaultVrfId )
   orrPositionsSourceByVrf = \
      orrPositionsSourceRoot.orrPositionsSourceByVrf.get( defaultVrfId )
   if orrPositionStatusByVrf:
      for positionName in sorted( orrPositionStatusByVrf.position ):
         position = orrPositionStatusByVrf.position[ positionName ]
         positionModel = BgpOrrPosition()
         positionModel.nlriTypes = None
         positionModel.name = positionName
         isActive = position.statusCode == 'idAssigned'
         positionModel.active = isActive
         if isActive:
            positionModel.numericId = position.id
            populatePositionSource( mode, positionName, positionModel,
                                    model.vrfName,
                                    orrPositionsSourceByVrf )
         else:
            positionModel.inactiveReason = position.inactiveReason()
            positionModel.sources = None
         model.position.append( positionModel )
   return model

def showArBgpDump( mode, args ):
   # Run this command only if the protocol model is multiAgent
   if ( not mode.session_.shouldPrint() or
        l3Config.protocolAgentModel != ProtoAgentModel.multiAgent ):
      # shouldPrint is set to False when displaying json and we are
      # not a converted command.
      return
   cmd = ArBgpCliCommand( mode, 'show support' )
   if 'iar' in args:
      if 'drain-undrain' in args:
         # show ip bgp neighbors iar vrf all
         cmd = ArBgpAsyncCliCommand( mode, 'show peer', vrfName='all' )
         cmd.addParam( 'iar' )
      else:
         # show bgp instance internal events iar history vrf all
         cmd = ArBgpAsyncCliCommand( mode, 'show instance-internal', vrfName='all' )
         cmd.addParam( 'iar-history' )
   if 'memory' in args:
      cmd.addParam( 'memory' )
   if 'import-error' in args:
      cmd.addParam( 'importError' )
   if 'verbose' in args:
      cmd.addParam( 'verbose' )
   if 'evpn' in args:
      cmd.addParam( 'evpn' )
   if 'graceful-restart' in args:
      cmd = ArBgpAsyncCliCommand( mode, 'show support graceful-restart' )
   if 'line-numbers' in args:
      cmd.addParam( 'dumpLineNumbers' )
   if 'rcf' in args:
      cmd.addParam( 'rcf' )
   if 'label-management' in args:
      cmd = ArBgpAsyncCliCommand( mode, 'show support label' )
   if 'link-state' in args:
      cmd = ArBgpAsyncCliCommand( mode, 'show support link-state' )
   dumpfile = args.get( 'DUMPFILE' )
   if dumpfile:
      cmd.addParam( 'dumpfile', dumpfile.localFilename() )
      if 'compress' in args:
         cmd.addParam( 'compress' )
      cmd.run( timeout=None )
   else:
      cmd.run()

# -------------------------------------------------------------------------------
# "show bgp instance [vrf <vrfName>]
#     (hidden options) [internal [dump-duplicates]]
# -------------------------------------------------------------------------------
@VrfCli.VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames )
def doShowBgpInstanceInternal( mode, internal, vrfName=None ):
   cmd = ArBgpAsyncCliCommand( mode, 'show instance-internal', vrfName=vrfName )
   if 'dump-duplicates' in internal:
      cmd.addParam( 'dump-duplicates' )
   elif 'brib' in internal and 'count' in internal:
      cmd.addParam( 'brib-count' )
      if 'detail' in internal:
         cmd.addParam( 'brib-detail' )
      elif 'stash' in internal:
         cmd.addParam( 'stash' )
   elif 'iar' in internal:
      assert 'events' in internal
      cmd.addParam( 'iar-history' )
   elif 'events' in internal:
      assert 'tracking' in internal
      if 'start' in internal:
         cmd.addParam( 'events-start' )
         if 'advertise-threshold' in internal:
            cmd.addParam( 'advertise-threshold' )
            cmd.addParam( internal.get( 'numRibRoutes' ) )
            cmd.addParam( internal.get( 'numRibOutRoutes' ) )
      elif 'stop' in internal:
         cmd.addParam( 'events-stop' )
      elif 'summary' in internal:
         cmd.addParam( 'events-summary' )
      elif 'detail' in internal:
         cmd.addParam( 'events-detail' )
   cmd.run()

def doShowBgpInstance( mode, internal=None, vrfName=None ):
   if internal:
      return doShowBgpInstanceInternal( mode, internal, vrfName=vrfName )

   # We can use the same command string ('show instance') here because
   # the new async command callback will register using
   # 'agentCommandCallbackWithFormatIs' instead of 'agentCommandCallbackIs'.
   # This helps with adding JSON format to the command.
   cmd = ArBgpAsyncCliCommand( mode, 'show instance', vrfName=vrfName, revision=3 )
   # The callback is shared with 'show bgp convergence'
   cmd.addParam( 'instance' )
   cmd.run()
   return cliPrinted( RoutingBgpShowCliHandler.cliShowBgpInstanceVrfModel )

# -------------------------------------------------------------------------------
# "show bgp instance vpn summary [vrf <vrfName>]
# -------------------------------------------------------------------------------
def doShowBgpInstanceVpnSummary( mode, vrfName=None ):
   cmd = ArBgpAsyncCliCommand( mode, 'show instance', vrfName=vrfName )
   cmd.addParam( 'summary' )
   cmd.run()
   return cliPrinted( cliShowBgpInstanceVpnSummaryVrfModel )

def handlerShowBgpScheduler( mode, args ):
   return doShowBgpScheduler( mode,
                              history=args.get( "history" ),
                              internal=args.get( "internal" ),
                              verbose=args.get( "verbose" ) )

@ArBgpShowOutput( 'doShowIpBgpSummaryLldp', arBgpModeOnly=True )
def handlerShowIpBgpSummaryLldp( mode, args ):
   doShow = doShowIpv6BgpSummary if 'ipv6' in args else doShowIpBgpSummary
   return doShow( mode, lldp=True, vrfName=args.get( "VRF" ) )

def handlerShowTechBgpDebugCmd( mode, args ):
   showArBgpDebug( mode, args.get( 'ipUrib' ) )

# pylint: disable=consider-using-f-string
# -------------------------------------------------------------------------------
# "show bgp scheduler [verbose] [internal] [history]"
# -------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowBgpScheduler', arBgpModeOnly=True )
def doShowBgpScheduler( mode, history=None, internal=None, verbose=None ):
   cmd = ArBgpAsyncCliCommand( mode, 'show scheduler' )
   # This command does not use the ArBgp ACR framework, remove any
   # arguments it may have added
   cmd.params = []
   for option in [ arg for arg in ( history, internal, verbose )
                   if arg is not None ]:
      cmd.addParam( option )
   cmd.run()
   # Return the output from the agent directly to the user. Use
   # cliPrinted() to let the infrastructure know that we've done this.
   return cliPrinted( ShowTaskSchedulerModel.Overall )

# -------------------------------------------------------------------------------
# "show bgp configuration (active | unsupported | detail)"
# -------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowBgpConfig', arBgpModeOnly=True, requiresAgent=False )
def doShowBgpConfig( mode, option=None ):
   if arBgpConfig.asNumber == 0:
      return

   class Mode:
      def __init__( self, parent=None, modeCmd=None ):
         self.modeCmd_ = modeCmd
         self.parent_ = parent
         self.children_ = []
         if not parent:
            self.indent_ = ''
            self.output_ = []
         else:
            parent.children_.append( self )
            self.indent_ = parent.indent_
            self.output_ = parent.output_
         self.asdotConfigured = asnConfig.isAsdotConfigured()

      def __enter__( self ):
         if self.parent_ and len( self.parent_.children_ ) > 1:
            self.addCommand( '!' )
         if self.modeCmd_:
            self.addCommand( self.modeCmd_ )
            self.indent_ += '   '
         return self

      def __exit__( self, t, value, traceback ):
         pass

      def __getitem__( self, k ):
         return self

      def getOrCreateModeInstance( self, *args, **kwargs ):
         return self

      def addCommand( self, cmd ):
         self.output_.append( f'{self.indent_}{cmd}' )

      # pylint: disable-msg=W0621
      def addRouteMapConfigCommands( self, mapConfig, arBgpMapConfig=None ):
         for rmName in sorted( mapConfig.routeMap ):
            routeMap = mapConfig.routeMap[ rmName ]
            if arBgpMapConfig and rmName not in arBgpMapConfig.routeMap:
               # Skip routeMaps that are not in use by ArBgp
               continue
            for seqno in sorted( routeMap.mapEntry ):
               mapEntry = routeMap.mapEntry[ seqno ]
               with Mode( self, 'route-map %s %s %s' %
                           ( rmName, matchPermitEnum[ mapEntry.permit ], seqno ) ) \
                     as routeMapMode:
                  cmds = []
                  printRouteMapEntryAttributes( mapEntry, output=cmds )
                  for cmd in cmds:
                     routeMapMode.addCommand( cmd )

      def addBgpConfigCommands( self, bgpConfig, bgpVrfConfigDir,
                                ucmpConfig, vrfUcmpConfigDir ):
         options = Bunch( saveAll=False )
         with Mode( self, 'router bgp %s' % bgpFormatAsn( bgpConfig.asNumber,
                           self.asdotConfigured ) ) as routerBgpMode:
            cmds = saveBgpConfig( bgpConfig, bgpConfig, routingHwStatus,
                                  "", securityConfig, aclCpConfig,
                                  commandTagIdState,
                                  asdotConfigured=self.asdotConfigured )
            for callback in showBgpConfigInstanceCallbacks:
               cmdDict = callback( bgpConfig ) if callback else {}
               for submodeCmd, gloabalCmds in cmdDict.items():
                  if submodeCmd is None:
                     for cmd in gloabalCmds:
                        cmds.append( cmd )

            for cmd in cmds:
               routerBgpMode.addCommand( cmd )

            # UCMP
            cmds = saveUcmpConfig( routingHwStatus, ucmpConfig, ucmpConfig )
            for cmd in cmds:
               routerBgpMode.addCommand( cmd )

            # pylint: disable=consider-using-in
            def skipAfMode( af, cmds ):
               if not cmds:
                  if option == 'active':
                     # Skip the af mode cmds when there are no submode cmds
                     # present for the active configuration option.
                     return True
                  elif option == 'detail' or option == 'unsupported':
                     # Skip the af mode cmds if the af is neither ipv4 nor ipv6
                     # for both detail and unsupported configuration options.
                     # Always print both ipv4 and ipv6 af mode cmds since we
                     # know those af modes are supported in both gated and
                     # ArBgp.
                     if af != 'ipv4' and af != 'ipv6':
                        return True
                  else:
                     assert 0
               return False
            # pylint: enable=consider-using-in

            # load BGP-LS import config
            importcmds = getLsImportCmds( lsImportConfig )
            for af in bgpConfigAttrAfList():
               cmds = saveAfConfig( bgpConfig, bgpConfig, "", af,
                                    self.asdotConfigured )
               callBack = saveAfConfigCallbackDict.get( af )
               cmds += callBack( bgpConfig, af ) if callBack else []
               if skipAfMode( af, cmds ):
                  continue
               with Mode( routerBgpMode, 'address-family %s' % af ) \
                     as routerBgpAfMode:
                  if af == 'link-state':
                     cmds += importcmds
                     # reset the list, so that it does not get printed again futher
                     # down
                     importcmds = []
                  for cmd in cmds:
                     routerBgpAfMode.addCommand( cmd )

            # load BGP-LS role config
            rolecmds = getLsRoleCmds( lsRoleConfig )

            # if there were no link-state af specific commands just load the
            # import config commands
            if importcmds or rolecmds:
               with Mode( routerBgpMode, 'address-family link-state' ) as lsAfMode:
                  for cmd in importcmds:
                     lsAfMode.addCommand( cmd )
                  for cmd in rolecmds:
                     lsAfMode.addCommand( cmd )

            # Handle 'vrf default' mode
            dreCallBack = saveAfConfigCallbackDict.get( 'default-route-export' )
            defVrfCmds = {}
            defVrfCmds[ None ] = []
            rdRemoteDomainConfig = rdConfigDir[ 'bgpRemoteDomain' ]
            # For 'vrf default' rd is configured in bgpConfig as well as in
            # ip/vrf/routeDistinguisherInputDir/config/bgp.
            # show run fetches from bgpConfig for 'vrf default' and from
            # ip/vrf/routeDistinguisherInputDir/config/bgp for non default vrf
            # Since we are mimicing running config, mirroring the same logic
            rd = bgpConfig.routeDistinguisher
            rdRemoteDomain = rdRemoteDomainConfig.routeDistinguisher.get(
                  DEFAULT_VRF, None )
            if bgpConfig.rdAll:
               assert rd != 'INVALID'
               assert rd == rdRemoteDomain
               defVrfCmds[ None ].append( 'rd evpn domain all %s' %
                     formatRd( rd, self.asdotConfigured ) )
            else:
               if rd != 'INVALID':
                  defVrfCmds[ None ].append( 'rd %s' %
                        formatRd( rd, self.asdotConfigured ) )
               if rdRemoteDomain is not None:
                  cmds.append( 'rd evpn domain remote %s' %
                        formatRd( rdRemoteDomain, self.asdotConfigured ) )
            defVrfCmds[ None ] += saveBgpVpnConfig( DEFAULT_VRF, bgpConfig,
                                       bgpConfig, self.asdotConfigured, False )
            defVrfCmds[ None ] += \
                  dreCallBack( bgpConfig, None ) if dreCallBack else []

            for af in [ 'ipv4', 'ipv6' ]:
               defVrfCmds[ af ] = []
               defVrfCmds[ af ] = saveBgpVpnAfConfig( bgpConfig, bgpConfig, af,
                                          self.asdotConfigured, False )
               defVrfCmds[ af ] += dreCallBack(
                                    bgpConfig, af ) if dreCallBack else []

            cmds_present = not all(
                              skipAfMode( k, v ) for k, v in defVrfCmds.items() )
            if cmds_present:
               with Mode(
                     routerBgpMode, 'vrf %s' % DEFAULT_VRF ) as routerBgpVrfMode:
                  for cmd in defVrfCmds[ None ]:
                     routerBgpVrfMode.addCommand( cmd )
                  for af in [ 'ipv4', 'ipv6' ]:
                     if skipAfMode( af, defVrfCmds[ af ] ):
                        continue
                     with Mode( routerBgpVrfMode, 'address-family %s' % af ) \
                           as routerBgpVrfAfMode:
                        for cmd in defVrfCmds[ af ]:
                           routerBgpVrfAfMode.addCommand( cmd )

            # Handle instance level config submodes that are implemented in other
            # packages (rpki is one).
            for callback in showBgpConfigInstanceCallbacks:
               cmdDict = callback( bgpConfig ) if callback else {}
               for submodeCmd, cmds in cmdDict.items():
                  if submodeCmd is not None:
                     with Mode( routerBgpMode, submodeCmd ) as submode:
                        for cmd in cmds:
                           submode.addCommand( cmd )
            # Handle non default vrf modes
            rdConfig = rdConfigDir[ 'bgp' ]
            for vrfName in sorted( bgpVrfConfigDir.vrfConfig ):
               bgpVrfConfig = bgpVrfConfigDir.vrfConfig[ vrfName ]
               vrfUcmpConfig = ucmpVrfConfigDir.vrfConfig[ vrfName ]
               rd = rdConfig.routeDistinguisher.get( vrfName, None )
               rdRemoteDomain = rdRemoteDomainConfig.routeDistinguisher.get(
                     vrfName, None )

               with Mode( routerBgpMode, 'vrf %s' % vrfName ) as routerBgpVrfMode:
                  cmds = []
                  if bgpVrfConfig.rdAll:
                     assert rd is not None
                     assert rd == rdRemoteDomain
                     cmds.append( 'rd evpn domain all %s' %
                           formatRd( rd, self.asdotConfigured ) )
                  else:
                     if rd is not None:
                        cmds.append( 'rd %s' % formatRd( rd, self.asdotConfigured ) )
                     if rdRemoteDomain is not None:
                        cmds.append( 'rd evpn domain remote %s' %
                              formatRd( rdRemoteDomain, self.asdotConfigured ) )
                  cmds += saveBgpConfig( bgpVrfConfig, bgpConfig,
                                         routingHwStatus, vrfName,
                                         securityConfig, aclCpConfig,
                                         commandTagIdState,
                                         asdotConfigured=self.asdotConfigured )
                  cmds += dreCallBack( bgpVrfConfig, None ) if dreCallBack else []
                  for cmd in cmds:
                     routerBgpVrfMode.addCommand( cmd )

                  # UCMP
                  if vrfUcmpConfig:
                     cmds = saveUcmpConfig( routingHwStatus, ucmpConfig,
                                            vrfUcmpConfig )
                     for cmd in cmds:
                        routerBgpVrfMode.addCommand( cmd )

                  for af in bgpConfigAttrAfList( vrfName ):
                     cmds = saveAfConfig( bgpVrfConfig, bgpConfig, vrfName,
                                          af, self.asdotConfigured )
                     cmds += dreCallBack( bgpVrfConfig, af ) if dreCallBack else []
                     if skipAfMode( af, cmds ):
                        continue
                     with Mode( routerBgpVrfMode, 'address-family %s' % af ) \
                           as routerBgpVrfAfMode:
                        for cmd in cmds:
                           routerBgpVrfAfMode.addCommand( cmd )

            def natsortMacVrfs( config ):
               vlan = []
               vni = []
               vpws = []
               bundle = []
               for name in config:
                  macVrfConfig = config[ name ]
                  if macVrfConfig.isVniMacVrf():
                     vni.append( name )
                  elif macVrfConfig.isBundle:
                     bundle.append( name )
                  elif macVrfConfig.isVpwsMacVrf():
                     vpws.append( name )
                  else:
                     vlan.append( name )
               return natsorted( vlan ) + natsorted( vni ) \
                  + natsorted( vpws ) + natsorted( bundle )
            redistributeCache = {}
            for macVrfConfigName in natsortMacVrfs( macVrfConfigDir.config ):
               macVrfConfig = macVrfConfigDir.config[ macVrfConfigName ]
               macVrfModeCmd = BgpMacVrfConfigMode.BgpMacVrfMode( macVrfConfigName,
                  macVrfConfig.isBundle, macVrfConfig.isVniMacVrf(),
                  macVrfConfig.isVpwsMacVrf() ).enterCmd()
               with Mode( routerBgpMode, macVrfModeCmd ) as macVrfConfigMode:
                  saveMacVrfConfig( macVrfConfig, macVrfConfigMode, bgpConfig,
                     self.asdotConfigured, options, redistributeCache )

   activeConfigMode = Mode()
   activeConfigMode.addRouteMapConfigCommands( arBgpMapConfig )
   activeConfigMode.addBgpConfigCommands( arBgpConfig, arBgpVrfConfigDir,
                                          arUcmpConfig, arUcmpVrfConfigDir )

   if option == 'active':
      output = activeConfigMode.output_
   else:
      runningConfigMode = Mode()
      runningConfigMode.addRouteMapConfigCommands( mapConfig, arBgpMapConfig )
      runningConfigMode.addBgpConfigCommands( config, vrfConfigDir,
                                              ucmpConfig, ucmpVrfConfigDir )
      ndiff = difflib.ndiff( runningConfigMode.output_, activeConfigMode.output_ )
      diff = list( ndiff )
      if option == 'detail':
         if len( diff ) != len( activeConfigMode.output_ ):
            output = diff
         else:
            output = activeConfigMode.output_
      else:
         assert option == 'unsupported'
         output = []
         if any( line.startswith( '-' ) for line in diff ):
            def strip( line ):
               return line[ 2 : ]

            def indent( line ):
               return len( strip( line ) ) - len( strip( line ).lstrip() )

            lastLine = ''
            modeCmds = []
            for line in diff: # pylint: disable-msg=too-many-nested-blocks
               if line.lstrip() == "" or \
                     line.startswith( '+' ) or \
                     line.startswith( '?' ):
                  # Ignore empty lines and starting with '+' or '?'.
                  continue
               if line.find( '!' ) >= 0:
                  # Keep the '!' separator only if we previously appended a
                  # line with greater indent. Otherwise, ignore it.
                  if output and indent( output[ -1 ] ) > indent( line ):
                     output.append( line )
               else:
                  if lastLine:
                     # Track the list of mode commands up to this line.
                     if indent( lastLine ) < indent( line ):
                        # If this line has greater indent than the lastLine,
                        # then we've entered a submode and the lastLine should
                        # be appended to the list of mode commands.
                        # pylint: disable=use-a-generator
                        if modeCmds and \
                              indent( modeCmds[ -1 ] ) == indent( lastLine ):
                           modeCmds[ -1 ] = lastLine
                        else:
                           if modeCmds:
                              # Sanity check to assert that the last mode
                              # command has greater indent than all other mode
                              # commands up to this point.
                              assert all( indent( cmd ) < indent( lastLine )
                                          for cmd in modeCmds )
                           modeCmds.append( lastLine )
                        # pylint: enable=use-a-generator
                     else:
                        # Otherwise, remove any previously appended mode
                        # commands with greater indent in case we've instead
                        # exited a submode.
                        modeCmds = [ cmd for cmd in modeCmds
                                       if indent( cmd ) < indent( line ) ]
                  if line.startswith( '-' ):
                     # Append lines starting with '-' to the output, while also
                     # extending the output with the list of mode commands up
                     # to this line.
                     lastLine = ''
                     if modeCmds:
                        output.extend( modeCmds )
                        modeCmds = []
                     output.append( line )
                  else:
                     lastLine = line

   print( "\n".join( output ) )
# pylint: enable=consider-using-f-string

def handlerShowBgpConfiguration( mode, args ):
   option = ( args.get( "active" ) or args.get( "unsupported" ) or
              args.get( "detail" ) )
   return doShowBgpConfig( mode, option=option )

# -------------------------------------------------------------------------------
# "show tech-support bgp debug [ipUrib]
# -------------------------------------------------------------------------------
def showArBgpDebug( mode, forwardingTarget=None ):
   # Run this command only if the protocol model is multiAgent
   if not mode.session_.shouldPrint() or \
         l3Config.protocolAgentModel != ProtoAgentModel.multiAgent:
      # shouldPrint is set to False when displaying json and we are
      # not a converted command.
      return
   cmd = ArBgpCliCommand( mode, 'show support' )
   cmd.addParam( 'debug' )
   # We default to ipUrib ForwardingTarget when no explicit forwardingTarget is
   # specified is for compatibility with gated - which has 'show tech ribd
   # debug' command
   if not forwardingTarget:
      forwardingTarget = 'ipUrib'
   cmd.addParam( 'forwardingTarget', forwardingTarget )
   cmd.run()

def CommonHistoryParser( mode, args, cmdClass, cmdStr ):
   '''
   Parses the common syntax objects and instantiate the command.
   params:
      mode, args: same as in handler
      cmdClass: class object that will run the ACR command
      cmdStr: command string registered in ArBgp
   returns the command object
   '''
   vrfName = args.get( 'VRF' )
   kwargs = { 'peerAddr' : args.get( 'PEER' ) }
   convertPeerAddr( kwargs )
   prefix = args.get( 'PREFIX' ) or args.get( 'PREFIX6' )

   cmd = cmdClass( mode, cmdStr, vrfName=vrfName, prefix=prefix, **kwargs )
   if 'PEERGROUP_NAME' in args:
      cmd.addParam( 'PEERGROUP_NAME', args[ 'PEERGROUP_NAME' ] )
   if 'HISTORY_SCOPE' in args and args[ 'HISTORY_SCOPE' ]:
      cmd.addParam( 'HISTORY_SCOPE', args[ 'HISTORY_SCOPE' ] )
   return cmd

@ArBgpShowOutput( 'doShowBgpNeighborsHistory', arBgpModeOnly=True )
def handlerShowBgpNeighborsHistoryCmd( mode, args ):
   cmd = CommonHistoryParser( mode, args, ArBgpAsyncCliCommand,
                                    'show neighbors history' )
   cmd.run()
   return cliPrinted( ShowNeighborsHistoryModel )

@ArBgpShowOutput( 'doShowBgpNeighborsHistorySocket', arBgpModeOnly=True )
def handlerShowBgpNeighborsHistorySocketCmd( mode, args ):
   cmd = CommonHistoryParser( mode, args, ArBgpAsyncCliCommand,
                                    'show neighbors history socket' )
   cmd.run()
   return cliPrinted( ShowNeighborsHistorySocketModel )

def handlerClearBgpNeighborsHistoryCmd( mode, args ):
   cmd = CommonHistoryParser( mode, args, ArBgpCliCommand,
                                    'clear neighbors history' )
   cmd.run()

@ArBgpShowOutput( 'doShowBribMedConsistency', arBgpModeOnly=True )
def handlerShowBgpMedConsistency( mode, args ):
   vrfName = args.get( 'VRF' )
   cmd = ArBgpAsyncCliCommand( mode, 'show bgp med consistency',
                               vrfName=vrfName )
   cmd.run()

def handlerSetThreadConfigCmd( mode, args ):
   threadConfig = agentBgpThreadConfig
   if 'single' in args:
      threadConfig.config = Tac.Value( 'Routing::Bgp::ThreadConfig::Config',
                                       True,
                                       threadConfig.numInputThreadsDefault,
                                       threadConfig.numBestpathThreadsDefault )
   else:
      threadConfig.config = Tac.Value( 'Routing::Bgp::ThreadConfig::Config',
                                       False,
                                       args[ 'INPUT_THREADS' ],
                                       args[ 'BESTPATH_THREADS' ] )
   mode.addWarning(
      'Please restart the Bgp and Bmp agents for these changes to take effect.' )

def noOrDefaultHandlerSetThreadConfigCmd( mode, args ):
   threadConfig = agentBgpThreadConfig
   threadConfig.config = Tac.Value( 'Routing::Bgp::ThreadConfig::Config',
                                    False,
                                    threadConfig.numInputThreadsDefault,
                                    threadConfig.numBestpathThreadsDefault )

@ArBgpShowOutput( 'ShowBgpSessionTracker', arBgpModeOnly=True )
def handlerShowBgpSessionTracker( mode, args ):
   trackerName = args.get( 'TRACKER_NAME' )
   cmd = ArBgpAsyncCliCommand( mode, 'show bgp session tracker',
                               trackerName=trackerName )
   cmd.run()
   return cliPrinted( ArBgpCliModels.BgpSessionTrackerAllModel )

def doArBgpShowDynPfxList( mode, model, args ):
   ret = model()

   def populateDynamicPrefixList():
      def contributingRoutesGenerator():
         contributingRoute = (
               DynamicPrefixListModel.DynamicPrefixListContributingRoutes() )
         # We need to print maximum 10 matched routes
         count = 0
         for matchedRoute in matchedRoutes:
            count += 1
            if count > 10:
               break
            contributingRoute.prefix = matchedRoute
            yield contributingRoute
      for sessionBgpDplName, sessionBgpDpl in (
            list( dynPfxListConfigDir.sessionBgpDplList.items() ) ):
         if not pfxListName or pfxListName == sessionBgpDplName:
            # Session Bgp activated dynamic prefix-list state
            # can be one of the following:
            # 1. active
            # 2. inactive
            # 3. notApplicableToRoutingInstance
            #
            # If dynamic prefix-list is not used by any VRFs,
            # OR if the show command is issued for a non-existent VRF,
            # then the state would be 'notApplicableToRoutingInstance'.
            # If there is at least one peer's bgpState is `Established` in the VRF,
            # and this VRF is using current dynamic prefix-list,
            # then the state would be 'active' for this given VRF.
            # If current dynamic prefix-list is used by a VRF that has NO peer
            # with `Established` bgpState, OR no peer in this VRF
            # with `Established` bgpState is used by current dynamic prefix-list,
            # then the state would be 'inactive' for this given VRF.
            dynPfxListInst = DynamicPrefixListModel.DynamicPrefixListModel()
            matchedRoutes = []
            bgpPeers = {}
            isActive = False
            peerEntry = None
            # If vrfName is NOT specified, and the DPL is used by multiple VRFs,
            # show command only displays DPLs that are under VRF default.
            vrfConfig = rtrGeneralConfigDir.vrfConfig.get( vrfName, None )
            if not vrfConfig or \
                  not vrfConfig.dynamicRoute.get( sessionBgpDplName, None ):
               dynPfxListInst.state = 'notApplicableToRoutingInstance'
               yield sessionBgpDplName, dynPfxListInst
               continue
            sessionBgpAddr = sessionBgpDpl.sessionBgpAddr.keys()
            # If vrfName is NOT specified, vrfName is 'default' by default.
            peerInfoStatusTable = peerInfoTable.entity.get( vrfName )
            if peerInfoStatusTable:
               peerEntry = peerInfoStatusTable.bgpPeerInfoStatusEntry
            for addr in sessionBgpAddr:
               peerSpec = DynamicPrefixListModel.PeerSpecification()
               if peerEntry and addr in peerEntry:
                  peerSpec.sessionState = peerEntry[ addr ].bgpState
                  if peerSpec.sessionState == BgpPeerState.Established:
                     matchedRoutes.append( Arnet.IpGenPrefix( str( addr ) ) )
                     isActive = True
               else:
                  peerSpec.sessionState = BgpPeerState.Unknown
               peerSpec.originatePeerAddress = True
               bgpPeers[ str( addr ) ] = peerSpec
            dynPfxListInst.bgpSession = DynamicPrefixListModel.BgpSession(
                  peers=bgpPeers )
            if isActive:
               dynPfxListInst.state = 'active'
            else:
               dynPfxListInst.state = 'inactive'

            dynPfxListInst.numContributingRoutes = len( matchedRoutes )
            dynPfxListInst.contributingRoutes = itertools.chain(
                  dynPfxListInst.contributingRoutes,
                  contributingRoutesGenerator() )
            yield sessionBgpDplName, dynPfxListInst

      for dynPfxListName, dynPfxList in (
            list( dynPfxListConfigDir.dynamicPrefixList.items() ) ):
         dynPfxUndefinedPolicy = False
         if not pfxListName or pfxListName == dynPfxListName:
            matchedRoutes = []

            # Dynamic prefix-list state can be one of the following:
            # 1. active
            # 2. inactive
            # 3. notApplicableToRoutingInstance
            #
            # If dynamic prefix-list configuration does not have any match map
            # configured, then state would be 'inactive'.
            # If the configuration is complete, then we check the dplState for the
            # VRF in Sysdb. Sysdb only stores the state as 'active' or 'inactive'.
            # If the state for the VRF is not present, that would mean that the DPL
            # is not applicable for that VRF, so state would be
            # 'notApplicableToRoutingInstance'
            # If the show command is issued for a non-existent VRF, then the state
            # would be 'notApplicableToRoutingInstance'
            # The state could be 'inactive' if there are no matching routes or if
            # the match route-map configuration is missing.
            if vrfNameStatus.nameToIdMap is None:
               continue
            vrfId = vrfNameStatus.nameToIdMap.vrfNameToId.get( vrfName )

            if ( dynPfxList.matchMap == '' ) and ( dynPfxList.matchRcf == '' ):
               dynPfxListState = 'inactive'
               dynPfxUndefinedPolicy = True

            else:
               routeMapPolicyType = PolicyType.routeMap
               rcfPolicyType = PolicyType.rcfFunction
               if dynPfxList.matchRcf != '':
                  policyStr = dynPfxList.matchRcf
                  policyName = GenPolicyName( dynPfxList.matchRcf,
                                              rcfPolicyType )
                  policyConfig = rcfStatus.functionNames
               else:
                  policyStr = dynPfxList.matchMap
                  policyName = GenPolicyName( dynPfxList.matchMap,
                                              routeMapPolicyType )
                  policyConfig = mapConfig.routeMap

               dynPfxListState = 'notApplicableToRoutingInstance'
               # Get the dynamic prefix-list state from sysdb
               dynPfxListStatus = dynPfxListStatusDir.dynamicPrefixList.get(
                  dynPfxListName, None )
               if dynPfxListStatus and vrfId is not None:
                  dynPfxListState = dynPfxListStatus.activeInVrf.get( vrfId,
                     'notApplicableToRoutingInstance' )

               # If dynamic prefix list state is inactive, then check for match route
               # map configuration
               if ( dynPfxListState == 'inactive' and
                    policyStr not in policyConfig ):
                  dynPfxUndefinedPolicy = True

               # We need to populate matched routes only if state is active
               vrfPolicyStateSet = None
               if vrfId is not None:
                  vrfPolicyStateSet = (
                        allVrfPolicyStateSet.vrfPolicyStateSet.get( vrfId, None ) )
               if dynPfxListState == 'active' and vrfPolicyStateSet:
                  # Avoid transient failures in lookups that could happen immediately
                  # after changing the config (like changing match-map name)
                  policyEvalState = None
                  policyEvalState = vrfPolicyStateSet.policyEvalState.get(
                     policyName, None )
                  if not policyEvalState:
                     continue
                  matchedRoutes = policyEvalState.matchedRoute

            dynPfxListInst = DynamicPrefixListModel.DynamicPrefixListModel()
            dynPfxListInst.ipv4PrefixList = dynPfxList.ipv4PrefixList
            dynPfxListInst.ipv6PrefixList = dynPfxList.ipv6PrefixList
            dynPfxListInst.matchRouteMap = dynPfxList.matchMap
            dynPfxListInst.matchRcfFunction = dynPfxList.matchRcf
            dynPfxListInst.state = dynPfxListState
            # pylint: disable=protected-access
            dynPfxListInst._undefinedPolicy = dynPfxUndefinedPolicy
            dynPfxListInst.numContributingRoutes = len( matchedRoutes )
            dynPfxListInst.contributingRoutes = itertools.chain(
                  dynPfxListInst.contributingRoutes,
                  contributingRoutesGenerator() )
            yield dynPfxListName, dynPfxListInst

      if ( pfxListName and pfxListName not in
           dynPfxListConfigDir.dynamicPrefixList
           and pfxListName not in dynPfxListConfigDir.sessionBgpDplList ):
         dynPfxListInst = DynamicPrefixListModel.DynamicPrefixListModel()
         dynPfxListInst.state = 'notApplicableToRoutingInstance'
         yield pfxListName, dynPfxListInst

   pfxListName = None
   vrfName = args[ 'vrfName' ]
   if 'prefix-list' in args:
      pfxListName = args[ 'prefix-list' ]
   # pylint: disable-msg=W0212
   ret._vrf = vrfName
   ret.dynamicPrefixLists = itertools.chain( ret.dynamicPrefixLists,
                                             populateDynamicPrefixList() )
   return ret

def removeUnsupportedMatchers( log, rmName, unsupportedMatchers ):
   for seqLog in log[ 'seqs' ].values():
      newMatchers = []
      for matcher in seqLog[ 'passedMatchers' ]:
         if matcher not in unsupportedMatchers:
            newMatchers.append( matcher )
      seqLog[ 'passedMatchers' ] = newMatchers
      if 'subRouteMap' in seqLog:
         subRmName = seqLog[ 'subRouteMap' ][ 'rmname' ]
         removeUnsupportedMatchers( seqLog[ 'subRouteMap' ],
               subRmName, unsupportedMatchers )

def removePartiallySupportedMatchers( results, application ):
   """Remove matchers that are not supported across all applications.
   In such case we may find matcher present in passedMatchers list. so
   we need to remove them from the list.
   """
   # match ip resolved-next-hop is supported only in Redistribute policy case and not
   # in Inbound and Outbound policy application. Storing unsupported matchers as set
   # for faster searching
   # pylint: disable=consider-using-in
   if application == 'inbound' or application == 'outbound':
      unsupportedMatchers = { 'matchResolvedNextHopPrefixList',
                              'matchIpv6ResolvedNextHopPrefixList' }
      if application == 'inbound':
         # match interface is not supported in inbound case
         unsupportedMatchers.add( 'matchInterface' )
   else:
      return
   # pylint: enable=consider-using-in

   for peer in results:
      if 'log' not in peer:
         continue
      if peer[ 'policyType' ] != 'routemap':
         continue
      rmName = peer[ 'log' ][ 'rmname' ]
      removeUnsupportedMatchers( peer[ 'log' ], rmName, unsupportedMatchers )
   return

def modifyResultsWithMissingPolicy( results, vrf, nlriType, application ):
   # check the results for peers that didn't evaluate a route map and determine
   # if it was due to a non existent route map at the point of application or
   # no route map being applied.

   vrfConfig = config
   if vrf != 'default':
      vrfConfig = vrfConfigDir.vrfConfig[ vrf ]

   if application in [ 'inbound', 'outbound' ]:
      for peer in results:
         if 'log' not in peer and peer.get( 'missingPolicyApplies' ):
            # the route map was not found for the peer
            peerAddr = peer[ 'peerAddr' ]
            key = createKey( peerAddr )
            if key in vrfConfig.neighborConfig:
               peerConfig = vrfConfig.neighborConfig[ key ]
            else:
               peerGroup = peer[ 'peerGroup' ]
               pgKey = createKey( peerGroup )
               peerConfig = config.neighborConfig[ pgKey ]
            rmName = getConfiguredRouteMap( peerConfig, nlriType, application )
            if rmName is not None:
               peer[ 'missingPolicy' ] = rmName
   elif application == 'redistribute':
      for source in results:
         if 'log' not in source:
            # the route map was not found for the source
            proto = source.pop( 'redistProto' )
            redistConfig = vrfConfig.redistributeConfig.get( proto )
            if redistConfig:
               if redistConfig.routeMap:
                  source[ 'missingPolicy' ] = redistConfig.routeMap
   else:
      assert 0, "application not supported"
   return results

def routeMapDebugMissingPolicyApplies( nlriType, application ):
   # Missing policy modifications only apply to AfiSafis where route-map
   # debugging is supported (e.g. ipv4Unicast, ipv6Unicast) and where the application
   # is also supported (e.g. inbound, outbound)
   return ( nlriType in [ 'ipv4Unicast', 'ipv6Unicast' ] and
            application in [ 'inbound', 'outbound' ] )

def getConfiguredRouteMap( peerConf, nlriType, application ):
   # Get the route-map configured based on nlriType and application
   # or the route-map configured outside an address family given
   # a Routing::Bgp::PeerConfig object, this function works for AfiSafis
   # and applications where route-map debugging is supported.
   lut = {
      "ipv4Unicast" : {
         "inbound" : ( peerConf.routeMapV4UniInPresent, peerConf.routeMapV4UniIn ),
         "outbound" : ( peerConf.routeMapV4UniOutPresent, peerConf.routeMapV4UniOut )
      },
      "ipv6Unicast" : {
         "inbound" : ( peerConf.routeMapV6UniInPresent, peerConf.routeMapV6UniIn ),
         "outbound" : ( peerConf.routeMapV6UniOutPresent, peerConf.routeMapV6UniOut )
      },
      None : {
         "inbound" : ( peerConf.routeMapInPresent, peerConf.routeMapIn ),
         "outbound" : ( peerConf.routeMapOutPresent, peerConf.routeMapOut )
      }
   }
   afRmPresent, afRmName = lut[ nlriType ][ application ]
   if afRmPresent:
      return afRmName
   globalRmPresent, globalRmName = lut[ None ][ application ]
   if globalRmPresent:
      return globalRmName
   return None

# pylint: disable=raise-missing-from
def policyDebugHandlerCommon( mode, args, nlriType, nlriAttrs,
                              additionalParams=None ):
   # Get the routing context vrf if none was specified
   if 'VRF' not in args:
      vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, None )
   else:
      vrfName = args[ 'VRF' ]

   if vrfName not in VrfCli.getAllVrfNames():
      if vrfName.lower() == 'all':
         mode.addError( "Policy debug for all VRFs is not supported" )
      else:
         mode.addError( f"Failed to find VRF {vrfName}" )
      return None

   peerAddr = args.get( 'ADDR6', args.get( 'ADDR' ) )
   assert nlriType in Tac.Type( 'Routing::Bgp::NlriType::NlriTypeEnum' ).attributes
   cmd = ArBgpAsyncCliCommand( mode, 'show debug policy',
                               vrfName=vrfName,
                               nlriType=nlriType,
                               prefix=nlriAttrs.get( 'ipPrefix', None ),
                               peerAddr=peerAddr,
                               pathId=nlriAttrs.get( 'pathId', None ) )
   rd = args.get( 'RD' )
   if rd is not None:
      cmd.addParam( 'rdValue', rd )

   application = ""
   if 'inbound' in args:
      application = "inbound"
   elif 'outbound' in args:
      application = "outbound"
   elif 'redistribute' in args:
      application = "redistribute"
   elif 'import' in args:
      application = "import"
   elif 'export' in args:
      application = "export"
   else:
      mode.addError( "Not supported" )
      return None
   cmd.addParam( 'application', application )

   if 'RCF_FUNC_NAME' in args:
      cmd.addParam( 'rcfSpecified', 1 )
      # Trim the last two characters as they are '()'
      cmd.addParam( 'routeMapOrRcfFuncName', args[ 'RCF_FUNC_NAME' ][ : -2 ] )
   elif 'RMAP_NAME' in args:
      cmd.addParam( 'routeMapOrRcfFuncName', args[ 'RMAP_NAME' ] )

   if additionalParams:
      for k, v in additionalParams.items():
         cmd.addParam( k, v )

   buff = StringIO()
   cmd.run( stringBuff=buff, forceOutputFormat="json" )

   if not buff.getvalue():
      # The command returned nothing which means it is likely that BGP crashed.
      mode.addError( "" )
      return None

   resultJson = buff.getvalue()
   results = json.loads( resultJson )

   if isinstance( results, str ):
      mode.addError( results )
   elif "error" in results:
      mode.addError( results[ "error" ] )
   else:
      if routeMapDebugMissingPolicyApplies( nlriType, application ):
         results = modifyResultsWithMissingPolicy( results[ "results" ], vrfName,
                                                   nlriType, application )
         removePartiallySupportedMatchers( results, application )
      else:
         # Missing policy modifications don't apply to AfiSafis where route-map
         # debugging is unsupported (e.g. EVPN, MPLSVPN)
         results = results[ "results" ]
      try:
         m = createPolicyEvalModel( results, nlriType, application, nlriAttrs )
      except RcfDebugStateOutOfSync as e:
         mode.addError( "RCF compilation in progress" )
         t1( "RcfDebugStateOutOfSync: " + str( e ) )
         raise AlreadyHandledError
      return m

   return None
# pylint: enable=raise-missing-from

# Common helper function that is also used in MplsVpnShowCli
def getUnicastNlriHelper( args ):
   nlriAttrs = {}
   if args.get( 'ipv6' ):
      nlriType = 'ipv6Unicast'
      # The IPv6 prefix is stored as an IpGenAddr so we need to check if the attr
      # "is not None" to correctly process the default prefix ::/0
      assert args.get( 'PREFIX6' ) is not None
      nlriAttrs[ 'ipPrefix' ] = args[ 'PREFIX6' ].stringValue()
   elif args.get( 'ipv4' ):
      nlriType = 'ipv4Unicast'
      # The IPv4 prefix is stored as a string so processing the default prefix
      # works as intended
      assert args.get( 'PREFIX' )
      nlriAttrs[ 'ipPrefix' ] = args[ 'PREFIX' ]
   else:
      assert False, "Unexpected afi for unicast safi"
   if 'PATH_ID' in args:
      nlriAttrs[ 'pathId' ] = args[ 'PATH_ID' ]
   return nlriType, nlriAttrs

@ArBgpShowOutput( 'doShowBgpDebugPolicy', arBgpModeOnly=True )
def policyDebugHandler( mode, args ):
   # Get the NLRI type
   # The 'nlriType' value should correspond to the enum values in BgpNlriType.tac
   if args.get( 'SAFI', '' ) == 'unicast':
      nlriType, nlriAttrs = getUnicastNlriHelper( args )
   else:
      assert False, "Unexpected afisafi"
   return policyDebugHandlerCommon( mode, args, nlriType, nlriAttrs )

# -------------------------------------------------------------------------------
# show bgp route-flap-damping policy [ POLICY_NAME ]
# -------------------------------------------------------------------------------
def getRfdPolicy( policyName ):
   def _buildRfdPolicy( rfdPolicy ):
      rfdPolicyCliModel = BgpRfdPolicy()
      suppThld = rfdPolicy.rfdPolicyParams.suppressThreshold
      reuseThld = rfdPolicy.rfdPolicyParams.reuseThreshold
      maxThld = rfdPolicy.rfdPolicyParams.ceilingThreshold
      halfLife = rfdPolicy.rfdPolicyParams.halfLife
      v4UniPfxLst = rfdPolicy.suppressionIgnoredPrefixListV4Uni
      v4UniPfxLstDef = rfdPolicy.suppressionIgnoredPrefixListV4UniDefault
      v6UniPfxLst = rfdPolicy.suppressionIgnoredPrefixListV6Uni
      v6UniPfxLstDef = rfdPolicy.suppressionIgnoredPrefixListV6UniDefault
      rfdPolicyCliModel.suppressionPenaltyThreshold = suppThld
      rfdPolicyCliModel.reusePenaltyThreshold = reuseThld
      rfdPolicyCliModel.maximumPenaltyThreshold = maxThld
      rfdPolicyCliModel.withdrawalPenalty = \
            rfdPolicy.rfdPolicyParams.penaltyUnreach
      rfdPolicyCliModel.readvertisementAfterWithdrawalPenalty = \
            rfdPolicy.rfdPolicyParams.penaltyReach
      rfdPolicyCliModel.attributeChangePenalty = \
            rfdPolicy.rfdPolicyParams.penaltyAttrUpdate
      rfdPolicyCliModel.penaltyDecayHalfLife = halfLife
      rfdPolicyCliModel.maximumSuppressionTime = \
            int( math.log( maxThld / reuseThld, 2 ) * halfLife )
      rfdPolicyCliModel.minimumSuppressionTime = \
            int( math.log( suppThld / reuseThld, 2 ) * halfLife )
      if v4UniPfxLst != v4UniPfxLstDef:
         rfdPolicyCliModel.suppressionIgnorePrefixListV4Unicast = v4UniPfxLst
      if v6UniPfxLst != v6UniPfxLstDef:
         rfdPolicyCliModel.suppressionIgnorePrefixListV6Unicast = v6UniPfxLst
      return rfdPolicyCliModel

   if policyName:
      if policyName in config.rfdPolicy:
         rfdPolicy = config.rfdPolicy[ policyName ]
         yield policyName, _buildRfdPolicy( rfdPolicy )
      else:
         return
   else:
      for name, rfdPolicy in config.rfdPolicy.items():
         yield name, _buildRfdPolicy( rfdPolicy )

@ArBgpShowOutput( 'ShowBgpRfdPolicy', arBgpModeOnly=True, requiresAgent=False )
def doShowBgpRfdPolicy( mode, args ):
   policyName = args.get( 'POLICY_NAME', None )
   model = BgpRfdPolicies()
   model.policies = getRfdPolicy( policyName )
   return model

# -------------------------------------------------------------------------------
# "show (ip | ipv6) bgp summary [<prefix>] [vrf <vrfName>]
# -------------------------------------------------------------------------------
def doShowIpBgpSummaryAcrImpl( mode, prefix=None, vrfName=None, lldp=None, afi=None,
                               nlriAfiSafi=None, extraNlriAfiSafis=None,
                               nlriType=None ):
   cmd = ArBgpAsyncCliCommand( mode, 'show summary', vrfName=vrfName,
                               lldp=lldp,
                               nlriAfiSafi=nlriAfiSafi,
                               extraNlriAfiSafis=extraNlriAfiSafis,
                               transportAfi=afi, nlriType=nlriType,
                               prefix=prefix )
   cmd.run()
   return cliPrinted( summaryVrfModel )

def doShowIpBgpSummary( mode, prefix=None, vrfName=None, lldp=None ):
   return doShowIpBgpSummaryAcrImpl( mode, prefix=prefix,
                                     vrfName=vrfName,
                                     lldp=lldp,
                                     nlriAfiSafi='ipv4Unicast',
                                     extraNlriAfiSafis=[ 'ipv4Lu' ] )

def doShowIpv6BgpSummary( mode, prefix=None, vrfName=None, lldp=None ):
   return doShowIpBgpSummaryAcrImpl( mode, prefix=prefix,
                                     vrfName=vrfName,
                                     lldp=lldp,
                                     nlriAfiSafi='ipv6Unicast',
                                     extraNlriAfiSafis=[ 'ipv6Lu' ] )

# -------------------------------------------------------------------------------
# "show (ip | ipv6) bgp [labeled-unicast] [<ip> | <prefix>] [longer-prefixes]
#  [detail] [vrf <vrfName>]"
# -------------------------------------------------------------------------------
def doShowAfBgpInternal( mode, **kwargs ):
   vrfName = kwargs.pop( "vrfName", None )
   prefix = kwargs.pop( "addr", None )
   afi = kwargs.pop( 'afi', '' )
   safi = kwargs.pop( 'safi', '' )
   extraNlriAfiSafis = []
   if safi == 'mplslabel':
      # Just do labeled-unicast
      nlriAfiSafi = afi + 'Lu'
   elif safi == 'multicast':
      # Just do multicast
      nlriAfiSafi = afi + 'Multicast'
   else:
      # Do both unicast and labeled-unicast
      nlriAfiSafi = afi + 'Unicast'
      extraNlriAfiSafis = [ afi + 'Lu' ]
   ArBgpAsyncCliCommand( mode, "show bgp <af> internal",
                         vrfName=vrfName,
                         prefix=prefix,
                         nlriAfiSafi=nlriAfiSafi,
                         extraNlriAfiSafis=extraNlriAfiSafis ).run()
   return cliPrinted( routeSummaryVrfModel )

def addressFamilyAndTypeToNlriAfiSafi( addressFamilyAndType ):
   if addressFamilyAndType is None:
      return None
   nlriAf = addressFamilyAndType.get( 'addressFamily' )
   if addressFamilyAndType.get( 'multicast' ):
      nlriAf = ( 'ipv6Multicast' if addressFamilyAndType.get( 'addressFamily' )
            == 'ipv6' else 'ipv4Multicast' )
   if addressFamilyAndType.get( 'mplslabel' ):
      nlriAf += 'Lu'
   if nlriAf == 'ip':
      nlriAf = 'ipv4'
   if nlriAf in ( 'ipv4', 'ipv6' ):
      # show only IP unicast safi
      nlriAf += 'Unicast'
   return nlriAf

# -------------------------------------------------------------------------------
# "show bgp (ipv4 | ipv6)
#       (unicast | multicast | labeled-unicast) summary [vrf <vrfName>]"
# -------------------------------------------------------------------------------
def doShowBgpNlriAfAcrSummary( mode, addressFamilyAndType, vrfName=None, lldp=None ):
   nlriAf = addressFamilyAndTypeToNlriAfiSafi( addressFamilyAndType )
   return doShowIpBgpSummaryAcrImpl( mode, vrfName=vrfName,
                                     nlriAfiSafi=nlriAf,
                                     lldp=lldp )

def doShowBgpNlriAfSummary( mode, addressFamilyAndType, vrfName=None, lldp=None ):
   return doShowBgpNlriAfAcrSummary( mode, addressFamilyAndType,
                                     vrfName=vrfName,
                                     lldp=lldp )

@ArBgpShowOutput( 'doShowBgpNlriAfSummaryLldp', arBgpModeOnly=True )
def handlerShowBgpNlriAfSummaryLldp( mode, args ):
   return doShowBgpNlriAfSummary( mode, args.get( "addressFamilyAndType" ),
                                 vrfName=args.get( "VRF" ),
                                 lldp=True )

# -------------------------------------------------------------------------------
# "show ip[v6] bgp neighbors [<peerip>] update errors [vrf <vrfName>]"
# -------------------------------------------------------------------------------
def doShowIpBgpUpdateErrors( mode, addr=None, vrfName=None ):
   cmd = ArBgpAsyncCliCommand( mode, 'show peer update errors', vrfName=vrfName,
      peerAddr=addr )
   cmd.run()
   return cliPrinted( RoutingBgpShowCliHandler.updateErrorsVrfModel )

def doShowIpBgpPeerGroup( mode, pgAndVrfName=( None, None ) ):
   cmd = ArBgpAsyncCliCommand( mode, 'show peer-group',
                               vrfName=pgAndVrfName[ 1 ] )
   cmd.addParam( "peerGroup", pgAndVrfName[ 0 ] )
   cmd.run()
   return cliPrinted( AllBgpPeerGroups )

# -------------------------------------------------------------------------------
# "show ip bgp neighbors <ip> [(ipv4 unicast) | (ipv6 unicast)]
#       advertised-routes queued-withdrawals"
# "show ipv6 bgp peers <ip> [(ipv4 unicast) | (ipv6 unicast)]
#       advertised-routes queued-withdrawals"
# -------------------------------------------------------------------------------
def doShowIpBgpNeighborsQueuedWithdrawals( mode, peerAddr, queuedWithdrawals,
                                           routeFamilyAndType=None, vrfName=None ):

   nlriAfSafi = addressFamilyAndTypeToNlriAfiSafi( routeFamilyAndType )
   if not nlriAfSafi:
      nlriAfSafi = 'ipv4Unicast'
      extraNlriAfiSafis = [ 'ipv4Lu', 'ipv6Unicast', 'ipv6Lu' ]
   else:
      extraNlriAfiSafis = None
   return doShowBgpNeighborsAcr( mode, peerAddr=peerAddr, af='ip', vrfName=vrfName,
                                 nlriAfiSafi=nlriAfSafi,
                                 queuedWithdrawals=queuedWithdrawals,
                                 extraNlriAfiSafis=extraNlriAfiSafis,
                                 cliModel=queuedWithdrawalsVrfModel )

def doShowIpv6BgpNeighborsQueuedWithdrawals( mode, peerAddr, queuedWithdrawals,
                                             routeFamilyAndType=None, vrfName=None ):

   nlriAfSafi = addressFamilyAndTypeToNlriAfiSafi( routeFamilyAndType )
   if not nlriAfSafi:
      nlriAfSafi = 'ipv6Unicast'
      extraNlriAfiSafis = [ 'ipv4Unicast', 'ipv4Lu', 'ipv6Lu' ]
   else:
      extraNlriAfiSafis = None
   return doShowBgpNeighborsAcr( mode, peerAddr=peerAddr, af='ipv6', vrfName=vrfName,
                                 nlriAfiSafi=nlriAfSafi,
                                 queuedWithdrawals=queuedWithdrawals,
                                 extraNlriAfiSafis=extraNlriAfiSafis,
                                 cliModel=queuedWithdrawalsVrfModel )

# -------------------------------------------------------------------------------
# "clear ip bgp [<ip> | *] (([vrf <vrfName>] [reason MESSAGE]) |
#                          ([soft [in | out]] [vrf <vrfName>]))"
# -------------------------------------------------------------------------------
@VrfCli.VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames )
def doClearIpBgp( mode, transportAddrFamily=None, peer=None, soft=False,
                  direction=None, vrfName=None, routerId=None,
                  errorsOrCounters=None, resetMsg='' ):

   if ( not peer and config.clearAllDisabled and not errorsOrCounters ):
      mode.addError( "Cannot clear all peering sessions because " +
                     "clear all commands are disabled" )
      return
   #
   # Log what we are attempting to do
   #
   doLogClearIpBgp( mode, transportAddrFamily, peer, soft, direction,
                    vrfName,
                    errorsOrCounters == 'errors',
                    errorsOrCounters == 'counters',
                    resetMsg,
                    routerId )

   cmd = ArBgpCliCommand( mode, 'clear', vrfName=vrfName,
                          transportAfi=transportAddrFamily )
   if peer:
      cmd.addParam( 'peer-address', peer.stringValue )
   if routerId:
      cmd.addParam( 'routerId', routerId )
   if soft or direction:
      cmd.addParam( 'soft', direction )
   if errorsOrCounters == 'errors':
      cmd.addParam( 'errors' )
   if errorsOrCounters == 'counters':
      cmd.addParam( 'counters' )

   # transportBasedClear is True in "clear ip(v6) bgp neighbor *" command
   # we derive the transportBasedClear from transportAddrFamily which is set only
   # in 'clear ip(v6) bgp neighbor *' command
   transportBasedClear = transportAddrFamily is not None
   if transportBasedClear:
      cmd.addParam( 'neighbor' )

   if resetMsg:
      cmd.addParam( 'reason', resetMsg )

   cmd.run()

# -------------------------------------------------------------------------------
# "clear (ip | ipv6) bgp neighbor [*] [vrf <vrfName>] [reason MESSAGE]"
# -------------------------------------------------------------------------------
@VrfCli.VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames )
def doClearIpGenBgpTransport( mode, transportAddrFamily, vrfName=None, resetMsg='' ):
   if config.clearAllDisabled:
      mode.addError( "Cannot clear all peering sessions because "
                     "clear all commands are disabled" )
      return
   cmd = ArBgpCliCommand( mode, 'clear', vrfName=vrfName,
                          transportAfi=transportAddrFamily )
   cmd.addParam( 'neighbor' )
   if resetMsg:
      cmd.addParam( 'reason', resetMsg )
   cmd.run()

# -------------------------------------------------------------------------------
# "clear bgp route-flap-damping [PEER] [UNICAST_AFI_SAFI [PREFIX]] [VRF]"
# -------------------------------------------------------------------------------
@VrfCli.VrfExecCmdDec( getVrfsFunc=VrfCli.getVrfNames )
def doClearBgpRouteFlapDamping( mode, peer=None, ipv4=None, ipv4Prefix=None,
                                 ipv6=None, ipv6Prefix=None, vrfName=None ):
   cmd = ArBgpCliCommand( mode, 'clear', vrfName=vrfName )
   cmd.addParam( 'routeFlapDamping' )
   if peer is not None:
      cmd.addParam( 'peer-address', peer.stringValue )
   if ipv4 or ipv6:
      AfiSafi = Tac.Type( "Routing::Bgp::AfiSafi" )
      if ipv4:
         cmdAfiSafiU32 = AfiSafi.afiSafiToU32( AfiSafi.ipv4Uni() )
      else:
         cmdAfiSafiU32 = AfiSafi.afiSafiToU32( AfiSafi.ipv6Uni() )
      cmd.addParam( 'afiSafiU32', cmdAfiSafiU32 )
      if ipv4Prefix or ipv6Prefix:
         cmd.addParam( 'prefix', ipv4Prefix if ipv4Prefix else ipv6Prefix )
   cmd.run()

def handlerdoClearBgpRouteFlapDamping( mode, args ):
   doClearBgpRouteFlapDamping(
      mode, peer=args.get( 'PEER' ), ipv4=args.get( 'ipv4' ),
      ipv4Prefix=args.get( 'IPPREFIX' ), ipv6=args.get( 'ipv6' ),
      ipv6Prefix=args.get( 'IP6PREFIX' ), vrfName=args.get( 'VRF' ) )

# -------------------------------------------------------------------------------
# clear bgp route priority ( <AfiSafi/VRF> | all )
# -------------------------------------------------------------------------------
def clearBgpRoutePriority( mode, args ):
   doClearBgpRoutePriority( mode, args )

def doClearBgpRoutePriority( mode, args ):
   cmd = createClearBgpRoutePriorityCliCommand( mode, args )
   cmd.run()

def createClearBgpRoutePriorityCliCommand( mode, args ):
   cmd = ArBgpCliCommand( mode, 'clear route priority' )
   afiSafiVrf = args.get( 'afiSafiVrf' )
   if afiSafiVrf:
      afi, safi, vrfName = afiSafiVrf
      cmd.addParam( 'vrf', vrfName )
      cmd.addNlriAfiSafi( Tac.Type( 'Routing::Bgp::AfiSafi' )( afi, safi ) )
   return cmd

# -------------------------------------------------------------------------------
# "show maintenance bgp ip all [vrf <vrfName>]"
# -------------------------------------------------------------------------------
def _doShowMaintenanceBgpAll( mode, afi, vrfName ):

   cmd = ArBgpAsyncCliCommand( mode, 'show maint bgp',
      vrfName=vrfName, transportAfi=afi )
   # The output of the AgentCommandRequest will be json format output. Collect
   # it in a string buffer and then load it into a python object, which we then
   # use to fill in the CAPI model.
   strBuff = StringIO()
   cmd.run( stringBuff=strBuff, forceOutputFormat='json' )
   try:
      data = ""
      if vrfName == "all" or (
         vrfNameStatus.nameToIdMap is not None and
         vrfNameStatus.nameToIdMap.vrfNameToId.get( vrfName ) is not None ):
         data = json.loads( strBuff.getvalue() )
   except ValueError:
      return None
   else:
      def bgpMaintenanceVrfModelGenerator( data ):
         for vrfName in data:
            model = BgpMaintenanceModels.BgpMaintenanceStatus()
            vrfData = data[ vrfName ]
            model.setAsdot( asnConfig.isAsdotConfigured() )
            model.asn = vrfData[ 'asn' ]
            model.routerId = vrfData[ 'routerId' ]
            model.vrf = vrfData[ 'vrf' ]
            model.peers = peerModelGenerator( vrfData[ 'peers' ] )
            yield vrfName, model

      def peerModelGenerator( peers ):
         for peerAddr in peers:
            peerModel = BgpMaintenanceModels.BgpPeerMaintenanceStatus()
            peer = peers[ peerAddr ]
            peerModel.underMaintenance = peer[ 'underMaintenance' ] == 'true'
            if peerModel.underMaintenance:
               if 'appliedRouteMapIn' in peer:
                  peerModel.appliedRouteMapIn = peer[ 'appliedRouteMapIn' ]
               if 'appliedRouteMapOut' in peer:
                  peerModel.appliedRouteMapOut = peer[ 'appliedRouteMapOut' ]
               # for backward compatability
               if 'appliedRouteMapIn' in peer and 'appliedRouteMapOut' in peer and \
                  peer[ 'appliedRouteMapIn' ] == peer[ 'appliedRouteMapOut' ]:
                  peerModel.appliedRouteMap = peer[ 'appliedRouteMapIn' ]
            yield peerAddr, peerModel

      model = BgpMaintenanceShowCliHandler.BgpMaintenanceStatusVrf()
      vrfs = bgpMaintenanceVrfModelGenerator( data )
      if vrfs is not None:
         model.vrfs = vrfs
      return model

def doShowMaintenanceBgpIpAll( mode, vrfName=None ):
   return _doShowMaintenanceBgpAll( mode, afi='ip', vrfName=vrfName )

# -------------------------------------------------------------------------------
# "show maintenance bgp ipv6 all [vrf <vrfName>]"
# -------------------------------------------------------------------------------
def doShowMaintenanceBgpIp6All( mode, vrfName=None ):
   return _doShowMaintenanceBgpAll( mode, afi='ipv6', vrfName=vrfName )

# -------------------------------------------------------------------------------
# show maintenance bgp receiver route-map
# -------------------------------------------------------------------------------
def doShowMaintenanceBgpReceiverRm( mode, args ):
   if defaultRmName not in defaultReceiverRmDir.routeMap:
      mode.addWarning( "Maintenance receiver route-map is disabled." )
   return routeMapContainerModel( defaultReceiverRmDir, defaultRmName )

# -------------------------------------------------------------------------------
# "show bgp statistics [vrf <vrfName>]
# -------------------------------------------------------------------------------
def doShowBgpStatistics( mode, vrfName=None ):
   cmd = ArBgpAsyncCliCommand( mode, 'show statistics', vrfName=vrfName )
   cmd.run()
   return cliPrinted( statisticsVrfModel )

# -------------------------------------------------------------------------------
# "show bgp convergence [vrf <vrfName>]
# -------------------------------------------------------------------------------
def doShowBgpConvergence( mode, vrfName=None ):
   cmd = ArBgpAsyncCliCommand( mode, 'show instance', vrfName=vrfName )
   cmd.addParam( 'convergence' )
   cmd.run()
   return cliPrinted( ShowBgpConvergenceModel )

#-------------------------------------------------------------------------------
# "show maintenance bgp [<addr>] [vrf <vrfName>]"
#-------------------------------------------------------------------------------

def doShowMaintenanceBgp( mode, peer, vrfName=None ):
   if not vrfName:
      vrfName = 'default'
   llIntf = ""
   afi = 'ip'
   peerAddr = peer.stringValue
   if peer.type in [ 'peerIpv6', 'peerIpv6Ll' ]:
      afi = 'ipv6'
      llIntf = Arnet.IntfId( peer.llIntf ).stringValue
      peerAddr = peer.v6Addr.stringValue
   cmd = ArBgpAsyncCliCommand( mode, 'show maint bgp', vrfName=vrfName,
                               transportAfi=afi, peerAddr=peerAddr )
   cmd.addParam( 'llIntf', llIntf )
   # The output of the AgentCommandRequest will be json format output. Collect
   # it in a string buffer and then load it into a python object, which we then
   # use to fill in the CAPI model.
   strBuff = StringIO()
   cmd.run( stringBuff=strBuff, forceOutputFormat='json' )
   try:
      data = ""
      if vrfName == "all" or (
         vrfNameStatus.nameToIdMap is not None and
         vrfNameStatus.nameToIdMap.vrfNameToId.get( vrfName ) is not None ):
         data = json.loads( strBuff.getvalue() )
   except ValueError:
      return None
   else:
      def getPeerModel( peers, maintPeer ):
         peer = peers[ maintPeer ]
         peerModel = BgpMaintenanceModels.BgpPeerMaintenanceStatus()
         peerModel.underMaintenance = peer[ 'underMaintenance' ] == 'true'
         if peerModel.underMaintenance:
            if 'appliedRouteMapIn' in peer:
               peerModel.appliedRouteMapIn = peer[ 'appliedRouteMapIn' ]
            if 'appliedRouteMapOut' in peer:
               peerModel.appliedRouteMapOut = peer[ 'appliedRouteMapOut' ]
            # for backward compatability
            if 'appliedRouteMapIn' in peer and 'appliedRouteMapOut' in peer and \
               peer[ 'appliedRouteMapIn' ] == peer[ 'appliedRouteMapOut' ]:
               peerModel.appliedRouteMap = peer[ 'appliedRouteMapIn' ]
         return peerModel

      def getRouteMapInfo( rmName ):
         if rmName.lower() == 'systemgenerated':
            routeMapInfo = routeMapContainerModel(
               defaultInitiatorRmDir, 'SystemGenerated' )
         else:
            routeMapInfo = routeMapContainerModel( mapConfig, rmName )
         return routeMapInfo

      def generatorVrfModel():
         if vrfName == VrfCli.ALL_VRF_NAME:
            vrfs = VrfCli.getAllVrfNames()
         else:
            vrfs = [ vrfName ]
         for vrf in vrfs:
            maintBgp = BgpMaintenanceModels.BgpPeerMaintenanceInfo()
            maintBgp.setAsdot( asnConfig.isAsdotConfigured() )
            maintBgp.peer = peer.stringValue
            if data and vrf in data:
               vrfData = data[ vrf ]
               peers = vrfData[ 'peers' ]
               bgpPeerStatus = None
               if maintBgp.peer in peers:
                  bgpPeerStatus = BgpMaintenanceModels.BgpMaintenanceStatusPerPeer()
                  bgpPeerStatus.asn = vrfData[ 'asn' ]
                  bgpPeerStatus.routerId = vrfData[ 'routerId' ]
                  bgpPeerStatus.vrf = vrfData[ 'vrf' ]
                  bgpPeerStatus.peers.__setitem__(
                     str( maintBgp.peer ), getPeerModel(
                        vrfData[ 'peers' ], maintBgp.peer ) )
               rmName = None
               rmNameIn = None
               rmNameOut = None
               if bgpPeerStatus and bgpPeerStatus.peers.get( maintBgp.peer ):
                  maintBgp.status = bgpPeerStatus
                  rmName = bgpPeerStatus.peers[ maintBgp.peer ].appliedRouteMap
                  rmNameIn = bgpPeerStatus.peers[
                     maintBgp.peer ].appliedRouteMapIn
                  rmNameOut = bgpPeerStatus.peers[
                      maintBgp.peer ].appliedRouteMapOut
                  if not rmName and ( rmNameIn == rmNameOut ):
                     rmName = rmNameIn
               if rmName:
                  maintBgp.routeMapInfo = getRouteMapInfo( rmName )
               if rmNameIn:
                  maintBgp.routeMapInfoIn = getRouteMapInfo( rmNameIn )
               if rmNameOut:
                  maintBgp.routeMapInfoOut = getRouteMapInfo( rmNameOut )

               BgpMaintenanceShowCliHandler.getBgpGroupProfileInfo(
                  mode, peer, vrf, maintBgp )
               yield vrf, maintBgp
            else:
               yield vrf, maintBgp

      vrfModel = BgpMaintenanceShowCliHandler.BgpPeerMaintenanceInfoVrf()
      vrfModel.vrfs = generatorVrfModel()
      return vrfModel

# -------------------------------------------------------------------------------
# CliHook for "show maintenance interface <intf> detail" command
# -------------------------------------------------------------------------------
def bgpShowMaintenanceIntf( mode, intfId, status ):
   intfStatus = allIntfStatusDir.intfStatus.get( intfId )
   if not intfStatus or intfStatus.deviceName == "":
      return

   vrfName = BgpMaintenanceShowCliHandler.getVrf( intfId )
   if not vrfName:
      return

   cmd = ArBgpAsyncCliCommand( mode, 'maint intf', vrfName=vrfName )
   # the str repr of Arnet::IntfId has quotes around the name. Since we need just
   # the name without quotes, use intfId.stringValue
   cmd.addParam( 'interface', intfId.stringValue )

   # need to get the output back from the cmd execution for further parsing
   # the JSON output to populate the MaintenanceInterfaceBgpStatus model,
   # so need a local string buffer to pass to the run method
   strBuff = StringIO()

   # need to force json output always, so set the output format explicitly
   # using the forceOutputFormat option
   cmd.run( stringBuff=strBuff, forceOutputFormat='json' )
   try:
      data = json.loads( strBuff.getvalue() )
   except ValueError:
      return
   else:
      model = MaintenanceInterfaceBgpStatus()
      model.maintState = data[ 'maintState' ]
      model.vrf = data[ 'vrf' ]

      for peer in data[ 'peers' ]:
         peerData = data[ 'peers' ][ peer ]
         peerIntfModel = MaintenanceInterfaceBgpPeerStatus()
         if peerData[ 'underMaintenance' ] == 'true':
            if 'appliedRouteMapIn' in peerData:
               peerIntfModel.appliedRouteMapIn = peerData[ 'appliedRouteMapIn' ]
            if 'appliedRouteMapOut' in peerData:
               peerIntfModel.appliedRouteMapOut = peerData[ 'appliedRouteMapOut' ]
            # for backward compatability
            if 'appliedRouteMapIn' in peerData and \
               'appliedRouteMapOut' in peerData and \
               peerData[ 'appliedRouteMapIn' ] == peerData[ 'appliedRouteMapOut' ]:
               peerIntfModel.appliedRouteMap = peerData[ 'appliedRouteMapIn' ]
         model.peers[ peer ] = peerIntfModel

      status.bgpStatus = model

def doShowBgpNeighborsImpl( mode, transportAfi=None, addr=None, bfd=None,
                            internal=None, taskStats=None, iar=None, vrfName=None,
                            routerId=None ):

   # No JSON for debug commands
   if ( internal or taskStats or iar ) and not bfd:
      if mode.session_.outputFormat_ == "json":
         mode.addError( "This is an unconverted command" )
         return None

   return doShowBgpNeighborsAcr( mode, peerAddr=addr, vrfName=vrfName,
                                    bfd=bfd, af=transportAfi, taskStats=taskStats,
                                    iar=iar, internal=internal, routerId=routerId )

# -------------------------------------------------------------------------------
# "show [ip | ipv6] bgp neighbors [<ip>] [bfd] [vrf <vrfName>]
#     (hidden options) [internal [verbose] [rib-out] [dump-duplicates]]
#                      [task-stats [reset]]
#
# Note: 'verbose' option prints AdjRibin paths
# -------------------------------------------------------------------------------
def doShowBgpNeighborsAcr( mode, peerAddr=None, vrfName=None, bfd=None, af=None,
                           taskStats=None, iar=None, internal=None, routerId=None,
                           nlriAfiSafi=None, queuedWithdrawals=None,
                           extraNlriAfiSafis=None,
                           cliModel=peerListVrfModel ):
   llIntf = ""
   if not peerAddr:
      peerAddr = ""
   else:
      key = peerAddr
      if key.type == 'peerIpv4':
         peerAddr = key.v4Addr
      elif key.type in [ 'peerIpv6', 'peerIpv6Ll' ]:
         peerAddr = key.v6Addr.stringValue
         llIntf = Arnet.IntfId( key.llIntf ).stringValue
   cmdType = 'show peer'
   if queuedWithdrawals:
      cmdType += '-ribout'
   elif bfd:
      cmdType += '-bfd'
   cmd = ArBgpAsyncCliCommand( mode, cmdType, vrfName=vrfName,
                               peerAddr=peerAddr, transportAfi=af,
                               nlriAfiSafi=nlriAfiSafi,
                               extraNlriAfiSafis=extraNlriAfiSafis,
                               routerId=routerId )
   cmd.addParam( 'llIntf', llIntf )
   if queuedWithdrawals:
      cmd.addParam( 'queuedWithdrawals' )
   else:
      if taskStats:
         cmd.addParam( 'task-stats' )
         if 'reset' in taskStats:
            cmd.addParam( 'reset' )
      if iar:
         cmd.addParam( 'iar' )
      if internal:
         cmd.addParam( 'internal' )
         if 'verbose' in internal:
            cmd.addParam( 'verbose' )
         if 'rib-out' in internal:
            cmd.addParam( 'rib-out' )
         if 'dump-duplicates' in internal:
            cmd.addParam( 'dump-duplicates' )
   cmd.run()
   return cliPrinted( cliModel )

desc = "Per VRF BGP summary"
summaryAllVrfModel = VrfCli.generateVrfCliModel( BgpSummaryAll, desc )

@ArBgpShowOutput( 'doShowBgpSummary', arBgpModeOnly=True )
def handlerShowBgpSummary( mode, args ):
   cmd = ArBgpAsyncCliCommand( mode, 'show summary all',
                                 vrfName=args.get( 'VRF' ) )
   cmd.run()
   return cliPrinted( summaryAllVrfModel )

# ----------------------------------------------------------------------------------
# (config-dyn-pfx-P1)# [ no ] match rcf <rcf-function-name>()
# ----------------------------------------------------------------------------------

def handlerDynamicPrefixListConfigModeMatchRcf( mode, args ):
   # Trim the () off the end of the RCF function
   function = args[ 'FUNCTION' ][ : -2 ]
   mode.dynPfxEntry.matchRcf = function

def noOrDefaultHandlerDynamicPrefixListConfigModeMatchRcf( mode, args ):
   # Trim the () off the end of the RCF function
   function = args[ 'FUNCTION' ][ : -2 ]
   if mode.dynPfxEntry.matchRcf == function:
      mode.dynPfxEntry.matchRcf = ''

# -------------------------------------------------------------------------------
# "show bgp update-group [index | A.B.C.D | A:B:C:D:E:F:G:H] [vrf <vrfName>]"
# -------------------------------------------------------------------------------
def doShowBgpUpdateGroup( mode, opt=None, vrfName=None ):
   cmd = ArBgpAsyncCliCommand( mode, 'show update-group', vrfName=vrfName )
   if opt is not None:
      if isinstance( opt, int ):
         cmd.addParam( 'index', opt )
      else:
         peerAddr = ""
         llIntf = ""
         if opt.type == 'peerIpv4':
            peerAddr = opt.v4Addr
         else:
            assert opt.type in [ 'peerIpv6', 'peerIpv6Ll' ]
            peerAddr = opt.v6Addr.stringValue
            llIntf = Arnet.IntfId( opt.llIntf ).stringValue
         cmd.addParam( 'peerAddr', peerAddr )
         cmd.addParam( 'llIntf', llIntf )
   cmd.run()
   return cliPrinted( RoutingBgpShowCliHandler.updateGroupVrfModel )

# -------------------------------------------------------------------------------
# "show bgp traffic-policy field-set mappings [vrf <vrfName>]"
# -------------------------------------------------------------------------------

def doShowBgpTrafficPolicyFieldSetMappings( mode, vrfName=None ):
   cmd = ArBgpAsyncCliCommand( mode, 'show field-set-mappings', vrfName=vrfName )
   cmd.addParam( "mappingsType", "tp" )
   cmd.run()
   return cliPrinted( ShowTrafficPolicyFieldSetCommunityMappingsModel )

# -------------------------------------------------------------------------------
# "show bgp vrf selection policy field-set mappings
# -------------------------------------------------------------------------------

def doShowBgpVrfSelectionPolicyFieldSetMappings( mode, vrfName=None ):
   cmd = ArBgpAsyncCliCommand( mode, 'show field-set-mappings', vrfName=vrfName )
   cmd.addParam( "mappingsType", "vsp" )
   cmd.run()
   return cliPrinted( ShowVrfSelectionPolicyFieldSetCommunityMappingsModel )

# -------------------------------------------------------------------------------
# "[ no | default ] next-hop resolution route igp-nexthop-cost protocol bgp"
# command, in "router-bgp" mode.
# -------------------------------------------------------------------------------
def handlerSetNhResRouteIgpNhCostBgpCmd( mode, args ):
   routerGeneralConfig.nhResRouteIgpNhCostBgp = True

def noOrDefaultHandlerSetNhResRouteIgpNhCostBgpCmd( mode, args ):
   routerGeneralConfig.nhResRouteIgpNhCostBgp = False

# -------------------------------------------------------------------------------
# "show bgp rcf detail [ <func-name> ]"
# -------------------------------------------------------------------------------
@ArBgpShowOutput( 'doShowBgpRcfDetail', arBgpModeOnly=True )
def handlerShowBgpRcfDetailCmd( mode, args ):
   if getEffectiveProtocolModel( mode ) != ProtoAgentModel.multiAgent:
      return doShowBgpOutputNotSupported( mode, *args )
   cmd = ArBgpAsyncCliCommand( mode, 'show bgp rcf detail' )
   if 'FUNC_NAME' in args:
      funcNameParam = CliPlugin.RcfCliLib.stripParens( args[ 'FUNC_NAME' ] )
      cmd.addParam( 'functionName', funcNameParam )
   ret = cmd.run( model=ShowAgentRcfDetailModel )
   if ret is None:
      return None
   return cliPrinted( ShowAgentRcfDetailModel )

# -------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
# -------------------------------------------------------------------------------
def Plugin( entityManager ):
   global routingHwStatusCommon
   global routingHwStatus
   global config, vrfConfigDir, mapConfig
   global asnConfig
   global allIntfStatusDir
   global vrfNameStatus, allVrfRmStatusSet, allVrfPolicyStateSet
   global dynPfxListConfigDir, dynPfxListStatusDir
   global arBgpConfig, arBgpVrfConfigDir, arBgpMapConfig, macVrfConfigDir
   global l3Config
   global securityConfig
   global ucmpConfig, ucmpVrfConfigDir
   global arUcmpConfig, arUcmpVrfConfigDir
   global aclCpConfig
   global rdConfigDir
   global lsImportConfig
   global lsRoleConfig
   global orrPositionStatusRoot
   global sessionTrackerConfigDir
   global sessionTrackerIntfConfigDir
   global orrPositionsSourceRoot
   global agentBgpThreadConfig
   global defaultReceiverRmDir
   global rtrGeneralConfigDir, peerInfoTable
   global rcfStatus
   global defaultInitiatorRmDir
   global commandTagIdState
   global routerGeneralConfig

   rtrGeneralConfigDir = LazyMount.mount( entityManager,
                                          'routing/general/config/dynPolicyRoutes',
                                          'Routing::DynPolicyRoutes::Config', 'r' )
   peerInfoTable = LazyMount.mount(
      entityManager,
      Cell.path( 'routing/bgp/export/vrfBgpPeerInfoStatusEntryTable' ),
      'Tac::Dir', 'ri' )
   routingHwStatusCommon = LazyMount.mount( entityManager,
                                            "routing/hardware/statuscommon",
                                            "Routing::Hardware::StatusCommon", "r" )
   routingHwStatus = LazyMount.mount( entityManager, "routing/hardware/status",
                                      "Routing::Hardware::Status", "r" )
   config = LazyMount.mount( entityManager, 'routing/bgp/config',
                             'Routing::Bgp::Config', 'r' )
   vrfConfigDir = LazyMount.mount( entityManager, 'routing/bgp/vrf/config',
                                   'Routing::Bgp::VrfConfigDir', 'r' )
   macVrfConfigDir = LazyMount.mount( entityManager, 'routing/bgp/macvrf',
                                      'Routing::Bgp::MacVrfConfigDir', 'r' )
   mapConfig = LazyMount.mount( entityManager, 'routing/routemap/config',
                                'Routing::RouteMap::Config', 'r' )
   asnConfig = LazyMount.mount( entityManager, 'routing/bgp/asn/config',
                                  'Routing::AsnConfig', 'r' )
   arBgpConfig = LazyMount.mount( entityManager,
                                  Cell.path( 'routing/arbgp/bgp/config' ),
                                  'Routing::Bgp::Config', 'r' )
   arBgpVrfConfigDir = LazyMount.mount( entityManager,
                                        Cell.path( 'routing/arbgp/vrf/config' ),
                                       'Routing::Bgp::VrfConfigDir', 'r' )
   arBgpMapConfig = LazyMount.mount( entityManager,
                                     Cell.path( 'routing/arbgp/routemap/config' ),
                                     'Routing::RouteMap::Config', 'r' )
   securityConfig = LazyMount.mount( entityManager, 'mgmt/security/config',
                                    'Mgmt::Security::Config', 'r' )
   l3Config = LazyMount.mount( entityManager, 'l3/config',
                               'L3::Config', 'r' )
   defaultReceiverRmDir = LazyMount.mount(
      entityManager, 'maintenance/profile/config/default/arbgpReceiverRM/',
      'Routing::RouteMap::Config', 'r' )

   # initialize the global ext comm parser state machinery, or get the existing one
   ArBgpCliCommon.getBgpExtCommParserContainerSm()

   # Mounts for dynamic prefix list, maintenance bgp show commands
   vrfNameStatus = LazyMount.mount( entityManager,
                                    Cell.path( 'vrf/vrfNameStatus' ),
                                    'Vrf::VrfIdMap::NameToIdMapWrapper', 'r' )
   allVrfRmStatusSet = LazyMount.mount( entityManager, 'routing/rmeval/status',
                                        'Routing::RmEval::AllVrfRmStatusSet', 'r' )
   allVrfPolicyStateSet = LazyMount.mount( entityManager, 'routing/rmeval/state',
                                          'Routing::RmEval::AllVrfPolicyStateSet',
                                          'r' )
   ucmpConfig = LazyMount.mount( entityManager, 'routing/ucmp/bgp/config',
                                 'Routing::Ucmp::UcmpConfig', 'r' )
   ucmpVrfConfigDir = LazyMount.mount( entityManager,
                                       'routing/ucmp/bgp/vrf/config',
                                       'Routing::Ucmp::VrfUcmpConfigDir',
                                       'r' )
   arUcmpConfig = LazyMount.mount( entityManager,
                                   Cell.path( 'routing/arbgp/ucmp/config' ),
                                   'Routing::Ucmp::UcmpConfig', 'r' )
   arUcmpVrfConfigDir = LazyMount.mount(
      entityManager,
      Cell.path( 'routing/arbgp/ucmp/vrf/config' ),
      'Routing::Ucmp::VrfUcmpConfigDir', 'r' )
   aclCpConfig = LazyMount.mount( entityManager, 'acl/cpconfig/cli',
                                  'Acl::Input::CpConfig', 'r' )
   rdConfigDir = LazyMount.mount( entityManager,
                              'ip/vrf/routeDistinguisherInputDir/config',
                              'Tac::Dir', 'wi' )
   lsImportConfig = LazyMount.mount( entityManager, 'routing/bgp/lsImportConfig',
                        'Routing::Bgp::LinkStateImportConfig', 'w' )
   lsRoleConfig = LazyMount.mount( entityManager, 'routing/bgp/lsRoleConfig',
                        'Routing::Bgp::LinkStateRoleConfig', 'w' )
   dynPfxListConfigDir = LazyMount.mount( entityManager, 'routing/dynPfxList/config',
                                          'Routing::DynamicPrefixList::Config', 'r' )
   dynPfxListStatusDir = LazyMount.mount( entityManager,
                                          Cell.path( 'routing/dynPfxList/status' ),
         'Routing::DynamicPrefixList::AllDynamicPrefixListVrfSet', 'r' )
   allIntfStatusDir = LazyMount.mount( entityManager, "interface/status/all",
                                       "Interface::AllIntfStatusDir", "r" )
   orrPositionStatusRoot = LazyMount.mount(
      entityManager,
      Cell.path( 'routing/bgp/orrPositionStatusRoot' ),
      'Routing::Bgp::OrrPositionStatusRoot', 'ri' )
   orrPositionsSourceRoot = LazyMount.mount(
      entityManager,
      Cell.path( 'routing/bgp/orrPositionsSourceRoot' ),
      'Cspf::OrrPositionsSourceRoot', 'ri' )
   sessionTrackerConfigDir = ConfigMount.mount(
      entityManager, 'routing/arbgp/sessionTracker/inst',
      'Routing::Bgp::SessionTrackerCliConfigDir', 'w' )
   sessionTrackerIntfConfigDir = ConfigMount.mount(
      entityManager, 'routing/arbgp/sessionTracker/intf',
      'Routing::Bgp::SessionTrackerIntfCliConfigDir', 'w' )
   IntfCli.Intf.registerDependentClass( SessionTrackerIntf )
   agentBgpThreadConfig = LazyMount.mount(
      entityManager,
      'agent/bgp/threadconfig',
      'Routing::Bgp::ThreadConfig', 'w' )
   rcfStatus = LazyMount.mount( entityManager,
                                Cell.path( 'routing/rcf/status' ),
                                'Rcf::Status', 'r' )
   defaultInitiatorRmDir = LazyMount.mount(
      entityManager, 'maintenance/profile/config/default/initiatorRM/',
      'Routing::RouteMap::Config', 'r' )
   commandTagIdState = LazyMount.mount(
      entityManager, "configTag/configTagIdState", "ConfigTag::ConfigTagIdState",
      "r" )
   routerGeneralConfig = LazyMount.mount( entityManager,
                                          'routing/general/config/global',
                                          'Routing::General::Config', 'w' )
