#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AgentCommandRequest
import ArnetLib
from CliDynamicSymbol import CliDynamicPlugin
from CliModel import cliPrinted
from CliPlugin.ArBgpCli import (
   ArBgpCliCommand,
   mplsLabelSafiArg,
)
from CliPlugin import BgpCliHelperCli
from CliPlugin.RoutingBgpShowCli import (
   ArBgpShowOutput,
   getCommunityValuesScalarList,
   routeSummaryVrfModel,
   validateAspRegex,
   validatePosixRegex,
)
from CliPlugin import VrfCli

ArBgpCliHandler = CliDynamicPlugin( "ArBgpCliHandler" )
BgpCliHelperCliModel = CliDynamicPlugin( "BgpCliHelperCliModel" )

class BgpCliHelperCommand( ArBgpCliCommand ):
   def __init__( self, mode, command, **kwargs ):
      assert 'keepalive' not in kwargs
      vrfName = kwargs.pop( 'vrfName', None )
      nlriAfiSafi = kwargs.pop( 'afi', '' )
      safi = kwargs.pop( 'safi', '' )
      if safi == 'unicast':
         nlriAfiSafi += "Unicast"
      elif safi == 'mplslabel':
         nlriAfiSafi += "Lu"
      elif safi == 'multicast':
         nlriAfiSafi += "Multicast"
      extraNlriAfiSafis = kwargs.pop( 'extraNlriAfiSafis', None )
      self.model = kwargs.pop( 'model', None )
      super().__init__( mode,
                        command,
                        vrfName=vrfName,
                        nlriAfiSafi=nlriAfiSafi,
                        extraNlriAfiSafis=extraNlriAfiSafis,
                        transportAfi=None,
                        disableFork=True )
      addr = kwargs.pop( 'addr', None )
      if addr is not None:
         self.addParam( 'addr', str( addr ) )

      self.mode = mode

      if self.mode.session_.outputFormat_ == 'json':
         self.addParam( 'json' )

      for k, v in kwargs.items():
         if v:
            self.addParam( k, v )

   def addNlriAfiSafi( self, nlriAfiSafi ):
      """Overloaded method from ArBgpCliCommand  base class"""
      if nlriAfiSafi:
         self.addParam( 'nlriAfiSafi', nlriAfiSafi )

   def run( self, **kwargs ):
      AgentCommandRequest.runCliPrintSocketCommand( self.entityManager,
                                                    'BgpCliHelper',
                                                    self.command,
                                                    self.paramString(),
                                                    self.mode,
                                                    keepalive=True,
                                                    **kwargs )
      # Return the correct capi model for converted commands and return
      # a "dummy" capi model for all unconverted commands.
      if self.command == 'show bgp segment-routing':
         return BgpCliHelperCliModel.PrefixSegments
      elif self.model:
         return cliPrinted( self.model )
      else:
         return cliPrinted( routeSummaryVrfModel )

def dictReplace( what, where ):
   for k, v in what.items():
      assert k in where, f"key {k} not found in {where}"
      where[ v ] = where[ k ]
      del where[ k ]

def validateRegex( mode, kwargs ):
   commReg = kwargs.get( 'commRegex', None )
   aspReg = kwargs.get( 'aspRegex', None )
   validatedComm = validatePosixRegex( mode, commReg ) if commReg else True
   if aspReg:
      # If we have a valid as path regex, set which mode we are in too
      validatedAsp = validateAspRegex( mode, aspReg )
   else:
      validatedAsp = True
   return validatedComm and validatedAsp

def convertCommunityValues( mode, kwargs ):
   key = 'commValues'
   value = kwargs.get( key, None )
   if not value:
      return
   commValues = getCommunityValuesScalarList( value )
   # Using a custom separator that is not intepreted by the ACR
   kwargs[ key ] = "|".join( [ str( c ) for c in commValues ] )
   exact = kwargs.pop( "exact", None )
   if exact:
      kwargs[ "standardCommunitiesExactMatch" ] = exact

def convertLargeCommunityValues( kwargs, fromKwargs=None, key=None ):
   def asplain( largeCommValue ):
      """ Return large community value in AS-plain format """
      tokens = largeCommValue.split( ':' )
      tokens[ 0 ] = str( ArnetLib.asnStrToNum( tokens[ 0 ] ) )
      return ":".join( tokens )
   if fromKwargs is None:
      fromKwargs = kwargs
   value = fromKwargs.get( key, None )
   if not value:
      return
   # Using a custom separator that is not intepreted by the ACR
   kwargs[ 'largeCommValues' ] = "|".join( [ asplain( c ) for c in value ] )
   exact = fromKwargs.pop( "exact", None )
   if exact:
      kwargs[ "largeCommunitiesExactMatch" ] = exact

def getRegisterShowBgpNlriAfCallback( cmdName, argReplace=None, argsAdd=None ):
   def callback( mode, *args, **kwargs ):
      if argsAdd:
         kwargs.update( argsAdd )
      if isinstance( argReplace, dict ):
         dictReplace( argReplace, kwargs )
      elif callable( argReplace ):
         argReplace( kwargs )
      elif argReplace is not None:
         raise Exception( f"Unexpected argReplace={argReplace}" )

      # BgpCliHelper expects pfxAddr as addr
      pfxAddr = kwargs.pop( "pfxAddr", None )
      if pfxAddr is not None:
         kwargs[ "addr" ] = pfxAddr

      mplsLabelSafiArg( kwargs )

      internal = kwargs.pop( "internal", False )
      if internal:
         return ArBgpCliHandler.doShowAfBgpInternal( mode, **kwargs )

      if kwargs.get( 'afi' ) is None:
         return False

      BgpCliHelperCli.convertPeerAddr( kwargs )
      if not validateRegex( mode, kwargs ):
         return False
      convertCommunityValues( mode, kwargs )
      convertLargeCommunityValues( kwargs, key="largeCommValues" )
      if 'routeType' not in kwargs:
         kwargs[ 'routeType' ] = 'routes'

      # Set ACR read timeout to 10m. In scale setup, BgpCliHelper can
      # iterate a huge number of paths without printing anything
      # and the default 120s timeout before ConfigAgent gives up
      # reading from the socket is not enough.
      bchCmd = BgpCliHelperCommand( mode, 'show bgp <af>', **kwargs )
      return bchCmd.run( timeout=600 )

   return callback

def ShowBgpSegmentRoutingCmd_handler( mode, args ):
   extraNlriAfiSafis = [ 'ipv6Lu' ]
   return BgpCliHelperCommand( mode,
                               'show bgp segment-routing',
                               afi='ipv4',
                               safi='mplslabel',
                               routeType='routes',
                               extraNlriAfiSafis=extraNlriAfiSafis ).run()

def AgentBgpSnapshotMrtCmd_handler( mode, args ):
   filename = args[ 'FILE' ].localFilename()
   vrfName = args.get( 'VRF', VrfCli.vrfMap.getCliSessVrf( mode.session ) )
   bchCmd = BgpCliHelperCommand( mode, "bgp mrt received-routes", vrfName=vrfName )
   bchCmd.addParam( 'filename', filename )
   # Set ACR read timeout to 10m (same as for the "show ip bgp" commands).
   # In scale setup, BgpCliHelper can iterate a huge number of paths without
   # printing anything and the default 120s timeout before ConfigAgent gives up
   # reading from the socket is not enough.
   return bchCmd.run( timeout=600 )

@ArBgpShowOutput( 'showBgpRfdCmdHandler', arBgpModeOnly=True )
def showBgpRfdCmdHandler( mode, args ):
   kwargs = {}
   afiSafi = args.get( "addressFamilyAndType" )
   # afi and safi is parsed by BgpCliHelperCommand to construct the param
   # expected by BgpCliHelper
   kwargs[ "afi" ] = afiSafi.get( "addressFamily" )
   kwargs[ "safi" ] = 'unicast' if afiSafi.get( "unicast" ) else None

   # Rest of the keywords in kwargs are as expected by BgpCliHelper. The expected
   # params are defined in parse rules in BgpCommandParamsBase.tac
   if "PEER" in args:
      kwargs[ "peerAddr" ] = args.get( "PEER" )
      BgpCliHelperCli.convertPeerAddr( kwargs )
      kwargs[ "routerId" ] = args.get( "ROUTER_ID" )
   if "IPV4_OR_IPV6_ADDR_MASK" in args:
      kwargs[ "addr" ] = args.get( "IPV4_OR_IPV6_ADDR_MASK" )
      kwargs[ "longerPrefixes" ] = args.get( "longer-prefixes" )
   kwargs[ "vrfName" ] = \
         args.get( 'VRF', VrfCli.vrfMap.getCliSessVrf( mode.session ) )
   kwargs[ "suppressed" ] = args.get( "suppressed" )
   kwargs[ "received" ] = args.get( "received" )
   kwargs[ "withdrawn" ] = args.get( "withdrawn" )
   kwargs[ "tracked" ] = args.get( "tracked" )
   kwargs[ "ignored" ] = args.get( "ignored" )
   kwargs[ "model" ] = BgpCliHelperCliModel.rfdRouteVrfModel
   bchCmd = BgpCliHelperCommand( mode, "show bgp route-flap-damping", **kwargs )
   return bchCmd.run( timeout=600 )

def getAfiSafiFromAddressFamilyAndType( kwargs ):
   addressFamilyAndType = kwargs.pop( "addressFamilyAndType", None )
   if addressFamilyAndType is None:
      addressFamilyAndType = kwargs.pop( "routeFamilyAndType", None )
   afi = None
   safi = None
   if addressFamilyAndType:
      afi = addressFamilyAndType.get( 'addressFamily' )
      if afi == 'ip':
         afi = 'ipv4'
      if addressFamilyAndType.get( 'multicast' ):
         safi = 'multicast'
      elif addressFamilyAndType.get( 'mplslabel' ) or \
            addressFamilyAndType.get( 'labeled-unicast' ):
         safi = 'mplslabel'
      elif addressFamilyAndType.get( 'unicast' ):
         safi = 'unicast'
   return afi, safi

def flattenShowBgpNlriAf( kwargs ):
   afi, safi = getAfiSafiFromAddressFamilyAndType( kwargs )
   # if we have contradicting afis here we remove it from
   # kwargs so show command will be empty in case like:
   # show ip bgp neighbors ipv6 unicast received-routes
   # ... except when we have "peerAddr" then we leave it e.g.:
   # show ip bgp neighbors 10.20.30.40 ipv6 unicast received-routes
   # should return ipv6 path from this peer even though we have
   # "show ip ..." here
   if afi is not None:
      if kwargs.get( "peerAddr" ) is None \
            and kwargs.get( "afi" ) is not None \
            and kwargs.get( "afi" ) != afi:
         kwargs.pop( "afi", None )
      else:
         kwargs[ "afi" ] = afi
         kwargs[ "safi" ] = safi
   elif "pfxAddr" in kwargs:
      kwargs[ "afi" ] = "ipv6" if ":" in str( kwargs[ "pfxAddr" ] ) else "ipv4"
   elif kwargs.get( "peerAddr" ) is not None:
      # "show (ip|ipv6) bgp <peer_addr> routes" case: we dont' have address family
      # in the arguments so we should add both ipv4 and ipv6
      kwargs[ "afi" ] = "ipv4"
      kwargs[ "extraNlriAfiSafis" ] = [ "ipv6" ]

   if not "vrfName" in kwargs:
      kwargs[ "vrfName" ] = None

   communityValuesAndExact = kwargs.pop( "communityValuesAndExact", None )
   if communityValuesAndExact:
      commValues = communityValuesAndExact.get( 'communityValues' )
      exact = communityValuesAndExact.get( 'exact' )
      if commValues:
         kwargs[ "commValues" ] = commValues
      if exact:
         kwargs[ "exact" ] = exact
   return kwargs

def handleShowIpBgpNeighborsRoutesArgs( kwargs ):
   if 'afi' not in kwargs:
      kwargs[ 'afi' ] = 'ipv4'
   if 'nbrAddr' in kwargs:
      dictReplace( { "nbrAddr" : "peerAddr" }, kwargs )
   addr = kwargs.pop( 'addr', None )
   if addr:
      kwargs[ 'peerAddr' ] = addr
   flattenShowBgpNlriAf( kwargs )
   if 'regex' in kwargs:
      dictReplace( { "regex" : "aspRegex" }, kwargs )
   values = kwargs.pop( 'values', None )
   if values:
      kwargs[ 'commValues' ] = values
