#!/usr/bin/env python3
# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from ArnetModel import (
   Ip4Address,
   IpGenericAddress,
   IpGenericPrefix,
)
from CliModel import (
   Bool,
   DeferredModel,
   Dict,
   Enum,
   Float,
   Int,
   List,
   Str,
   Submodel,
)
from CliPlugin.BgpCliModels import (
   BgpRouteASPathEntry,
   BgpRoutePeerEntry,
)
from CliPlugin.VrfCli import generateVrfCliModel
import Tac

class BgpDpsRouteState( DeferredModel ):
   stale = Bool( default=False, help='Route is stale' )
   valid = Bool( default=False, help='Route is valid' )
   queued = Bool( default=False,
                  help='Route is queued for advertisement' )

class BgpDpsAvtNodeInfo( DeferredModel ):
   regionId = Int( help='Adaptive virtual topology region ID', optional=True )
   zoneId = Int( help='Adaptive virtual topology zone ID', optional=True )
   siteId = Int( help='Adaptive virtual topology site ID', optional=True )
   role = Enum( values=( 'unknown', 'edge', 'zone transit', 'region transit',
                'pathfinder' ), help='Adaptive virtual topology role',
                optional=True )

class IpSecurityKeyExchangeInfo( DeferredModel ):
   dhGroup = Int( help='Diffie-Hellman group number', optional=True )
   dhKeyLength = Int( help='Length of Diffie-Hellman public value in bytes',
                      optional=True )

class IpSecurityAlgorithmInfo( DeferredModel ):
   encryptionAlgo = Enum( values=Tac.Type( 'Ipsec::IpsecEspAlgorithm' ).attributes,
                          help='Encryption algorithm', optional=True )
   authenticationAlgo = Enum(
      values=Tac.Type( 'Ipsec::IpsecHmacAlgorithm' ).attributes,
      help='Authentication algorithm', optional=True )

class IpSecurityInfo( DeferredModel ):
   initialContact = Bool( help='A flag to indicate whether this is an initial '
                          'contact', optional=True )
   rekeyCounter = Int( help='Rekey counter', optional=True )
   nonceLength = Int( help='Length of nonce data in bytes', optional=True )
   keyExchanges = List( valueType=IpSecurityKeyExchangeInfo,
                        help='Key exchange information' )
   algorithms = List( valueType=IpSecurityAlgorithmInfo,
                      help='Algorithm information' )

class BgpDpsRoutePath( DeferredModel ):
   endpointAddr = IpGenericAddress( help='DPS endpoint address', optional=True )
   wanId = Int( help='DPS endpoint WAN ID', optional=True )
   udpPort = Int( help='DPS endpoint UDP port', optional=True )
   pathGroupName = Str( help='DPS path group name', optional=True )
   routeState = Submodel( valueType=BgpDpsRouteState, help='Route state' )

   # The following fields are present when the command is run with the detail keyword
   asPathEntry = Submodel( valueType=BgpRouteASPathEntry,
                           help='AS path information', optional=True )
   peerEntry = Submodel( valueType=BgpRoutePeerEntry, help='Peer information',
                         optional=True )
   nextHop = Str( help='Next hop address', optional=True )
   med = Int( help='Multi Exit Discriminator', optional=True )
   weight = Int( help='Weight for the path', optional=True )
   timestamp = Float( help='UTC seconds since epoch when the path was received.',
                      optional=True )
   localPreference = Int( help='I-BGP Local preference indicator', optional=True )
   avtNodeInfo = Submodel( valueType=BgpDpsAvtNodeInfo,
                           help='Adaptive virtual topology information',
                           optional=True )
   ipSecurityInfo = Submodel( valueType=IpSecurityInfo,
                              help='IP security tunnel encapsulation information',
                              optional=True )

class BgpDpsRouteEntry( DeferredModel ):
   totalPaths = Int( help='Total number of paths for this route', optional=True )
   dpsRoutePaths = List( valueType=BgpDpsRoutePath,
                         help='List of BGP path selection route ECMP paths' )

class BgpDpsVrfNlriRoutes( DeferredModel ):
   vteps = Dict( keyType=IpGenericPrefix,
                 valueType=BgpDpsRouteEntry,
                 help='BGP path selection routes keyed by VTEP address' )

class BgpDpsVrfRoutes( DeferredModel ):
   routerId = Ip4Address( help='BGP router identity' )
   asn = Str( help='Autonomous system number' )
   nlris = Dict( keyType=str, valueType=BgpDpsVrfNlriRoutes,
                 help='BGP path selection routes keyed by NLRI' )

BgpDpsRoutesVrfModel = generateVrfCliModel(
   BgpDpsVrfRoutes,
   "BGP path selection routes keyed by VRF" )
