#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliPlugin.RoutingBgpCli import (
   afModeExtensionHook,
   configForVrf,
)
from CliPlugin.RoutingBgpInstanceCli import (
   resetBgpAfModeConfig,
)
from CliPlugin.IraServiceCli import getEffectiveProtocolModel
from IpLibConsts import DEFAULT_VRF
from IpLibTypes import ProtocolAgentModelType
from TypeFuture import TacLazyType

RedistributeFlowspecType = TacLazyType( "Routing::Bgp::RedistributeFlowspec" )

def EnterBgpFlowspecAfMode_handler( mode, args ):
   # pylint: disable=consider-using-f-string
   if getEffectiveProtocolModel( mode ) != ProtocolAgentModelType.multiAgent:
      mode.addWarning( "Routing protocols model multi-agent must be "
                       "configured for the BGP Flow Specification "
                       "address-family" )
   addrFamily = 'flow-spec %s' % args[ 'AF' ]
   if mode.vrfName == DEFAULT_VRF:
      assert addrFamily in afModeExtensionHook.afModeExtensions()
      childMode = mode.childMode(
         afModeExtensionHook.afModeExtension[ addrFamily ],
         addrFamily=addrFamily )
   else:
      assert addrFamily in afModeExtensionHook.vrfAfModeExtensions()
      childMode = mode.childMode(
         afModeExtensionHook.vrfAfModeExtension[ addrFamily ],
         addrFamily=addrFamily, vrfName=mode.vrfName )
   mode.session_.gotoChildMode( childMode )

def EnterBgpFlowspecAfMode_noOrDefaultHandler( mode, args ):
   # pylint: disable=consider-using-f-string
   addrFamily = 'flow-spec %s' % args[ 'AF' ]
   bgpConfig = configForVrf( mode.vrfName )
   resetBgpAfModeConfig( bgpConfig, addrFamily, mode.vrfName )

def EnterBgpFlowspecVpnAfMode_handler( mode, args ):
   # pylint: disable=consider-using-f-string
   if getEffectiveProtocolModel( mode ) != ProtocolAgentModelType.multiAgent:
      mode.addWarning( "Routing protocols model multi-agent must be "
                       "configured for the BGP Flow Specification "
                       "address-family" )
   addrFamily = 'flow-spec %s' % args[ 'VPN_AF' ]
   assert mode.vrfName == DEFAULT_VRF
   assert addrFamily in afModeExtensionHook.afModeExtensions()
   childMode = mode.childMode(
      afModeExtensionHook.afModeExtension[ addrFamily ],
      addrFamily=addrFamily )
   mode.session_.gotoChildMode( childMode )

def EnterBgpFlowspecVpnAfMode_noOrDefaultHandler( mode, args ):
   # pylint: disable=consider-using-f-string
   addrFamily = 'flow-spec %s' % args[ 'VPN_AF' ]
   assert mode.vrfName == DEFAULT_VRF
   bgpConfig = configForVrf( mode.vrfName )
   resetBgpAfModeConfig( bgpConfig, addrFamily, mode.vrfName )

def redistributeFlowspecCollection( vrfName, addrFamily ):
   config = configForVrf( vrfName )
   if addrFamily == 'flow-spec ipv4':
      return config.redistributeAfV4Flowspec
   assert addrFamily == 'flow-spec ipv6'
   return config.redistributeAfV6Flowspec

def redistributeFlowspecPolicy_handler( mode, args ):
   redistColl = redistributeFlowspecCollection( mode.vrfName, mode.addrFamily )
   policyName = args[ 'POLICY_NAME' ]
   redistConfig = RedistributeFlowspecType( policyName )
   # Strip the () off of the RCF function name (if provided)
   rcfFunction = args.get( 'FUNCTION', "" )
   redistConfig.rcf = rcfFunction.replace( '()', '' )
   redistColl.addMember( redistConfig )

def redistributeFlowspecPolicy_noOrDefaultHandler( mode, args ):
   redistColl = redistributeFlowspecCollection( mode.vrfName, mode.addrFamily )
   policyName = args[ 'POLICY_NAME' ]
   del redistColl[ policyName ]

def routeTargetRedirectVrfLpm_handler( mode, args ):
   config = configForVrf( mode.vrfName )
   addrFamily = mode.addrFamily
   if addrFamily == 'flow-spec ipv4':
      config.rtRedirectVrfLpmV4Flowspec = True
      return
   assert addrFamily == 'flow-spec ipv6'
   config.rtRedirectVrfLpmV6Flowspec = True

def routeTargetRedirectVrfLpm_noOrDefaultHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   addrFamily = mode.addrFamily
   if addrFamily == 'flow-spec ipv4':
      config.rtRedirectVrfLpmV4Flowspec = False
      return
   assert addrFamily == 'flow-spec ipv6'
   config.rtRedirectVrfLpmV6Flowspec = False
