#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import ( DeferredModel, Submodel, Str, Int, List, Enum,
                       GeneratorDict, Bool )
from ArnetModel import Ip4Address
from CliPlugin.BgpCliModels import ( BgpRouteASPathEntry, BgpRouteTypeEntry,
                                     BgpRouteDetailEntry, BgpRoutePeerEntry,
                                     REASON_NOT_BEST_LIST )
from CliPlugin.VrfCli import generateVrfCliModel

ACTION_TYPES = [ 'drop', 'police', 'trafficAction', 'redirectVrf', 'markDscp',
                 'redirectIp', 'mirrorIp' ]

class BgpFlowspecAction( DeferredModel ):
   action = Enum( values=( ACTION_TYPES ),
                  help='Type of action that is performed' )
   value = Str( optional=True,
                help='Value associated with the action' )

class BgpFlowspecRedirectVrf( DeferredModel ):
   vrf = Str( help='Name of the VRF traffic is being redirected to' )
   routeTarget = Str( help='Route target that was used to select a VRF' )

class BgpFlowspecRulePath( DeferredModel ):
   asPathEntry = Submodel( valueType=BgpRouteASPathEntry,
                           help='AS path information' )
   med = Int( optional=True, help='Multi Exit Discriminator for the route' )
   localPreference = Int( optional=True,
                          help='I-BGP Local preference indicator' )
   routeType = Submodel( valueType=BgpRouteTypeEntry, help='Route type' )
   weight = Int( optional=True, help='Weight for the route' )
   routeDetail = Submodel( valueType=BgpRouteDetailEntry, optional=True,
                           help='Route details' )
   peerEntry = Submodel( valueType=BgpRoutePeerEntry, optional=True,
                         help='Peer information for the route' )
   reasonNotBestpath = Enum( values=REASON_NOT_BEST_LIST,
                             help='Reason route was not selected as BGP best path' )
   redistPolicy = Str( help='Name of the redistributed flowspec policy',
                       optional=True )
   actions = List( valueType=BgpFlowspecAction,
                   help='Actions that are performed for this rule' )
   redirectVrf = Submodel( valueType=BgpFlowspecRedirectVrf, optional=True,
                           help='VRF being used by a traffic redirect action' )
   unsupportedReason = List( optional=True, valueType=str,
                             help='Reasons why this rule path is unsupported' )

class BgpFlowspecRuleEntry( DeferredModel ):
   # Report the total number of paths as a separate attribute because the path list
   # may be filtered by peer.
   totalPaths = Int( help='Total number of paths for this rule' )
   ruleId = Int( help='Unique identifier of this rule' )
   matchDetails = List( optional=True, valueType=str,
                        help='Descriptions of matching conditions' )
   paths = List( valueType=BgpFlowspecRulePath,
                 help='BGP flowspec rule paths' )
   _routeKey = Str( help='Route Key, this is a string representation of the flowspec'
                    ' nlri, which is the set of matching conditions for the rule' )
   unsupportedReason = List( optional=True, valueType=str,
                             help='Reasons why the flowspec nlri for the rule is'
                             ' unsupported' )

class BgpFlowspecRules( DeferredModel ):
   vrf = Str( help='VRF name' )
   routerId = Ip4Address( help='BGP Router Identity' )
   asn = Int( help='Autonomous System Number' )
   _detail = Bool( optional=True, help='Detailed output is requested' )
   rules = GeneratorDict( keyType=str, valueType=BgpFlowspecRuleEntry,
                          help='Dictionary of BGP Flowspec rule entries indexed by '
                          'the route key' )

# Wrap BgpFlowspecRules model under "vrfs" key
bgpFlowspecRulesVrfModel = generateVrfCliModel( BgpFlowspecRules,
                                                "Bgp Flowspec rule summary" )
