#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import cliPrinted
from CliDynamicSymbol import CliDynamicPlugin
from CliPlugin.ArBgpCli import (
   ArBgpAsyncCliCommand,
)
from CliPlugin.BgpCliHelperCli import convertPeerAddr
from CliPlugin.RoutingBgpShowCli import ArBgpShowOutput

BgpFlowspecCliModels = CliDynamicPlugin( "BgpFlowspecCliModels" )
ArBgpCliHandler = CliDynamicPlugin( "ArBgpCliHandler" )

#-------------------------------------------------------------------------------
# Helper Methods
#-------------------------------------------------------------------------------

@ArBgpShowOutput( 'doShowBgpFlowspec', arBgpModeOnly=True, vrfLookup=True )
def doShowBgpFlowspec( mode, **kwargs ):
   convertPeerAddr( kwargs )
   kwargs = { k : v for k, v in kwargs.items() if v is not None }
   ArBgpAsyncCliCommand( mode, 'show flowspec', **kwargs ).run()
   return cliPrinted( BgpFlowspecCliModels.bgpFlowspecRulesVrfModel )

# The output of "show bgp flow-spec ipv4|ipv6 summary" and "show ip bgp summary"
# are the same, so we can use the existing doShowIpBgpSummaryAcrImpl.
@ArBgpShowOutput( 'doShowBgpFlowspecSummary', arBgpModeOnly=True,
                  vrfLookup=True )
def doShowBgpFlowspecSummary( mode, nlriAfiSafi, vrfName=None ):
   return ArBgpCliHandler.doShowIpBgpSummaryAcrImpl( mode, nlriAfiSafi=nlriAfiSafi,
                                                    vrfName=vrfName )

#-------------------------------------------------------------------------------

nlriTypeMap = {
   'ipv4' : 'ipv4Flowspec',
   'ipv6' : 'ipv6Flowspec',
   'vpn-ipv4' : 'ipv4FlowspecVpn',
   'vpn-ipv6' : 'ipv6FlowspecVpn',
}

def getNlriType( args ):
   for af, nlriType in nlriTypeMap.items():
      if af in args:
         return nlriType
   return None

def ShowBgpFlowspecBaseClass_handler( mode, args ):
   return doShowBgpFlowspec( mode,
                             nlriType=getNlriType( args ),
                             peerAddr=args.get( 'PEER' ),
                             routeType=args.get( 'ROUTE_TYPE' ),
                             rule=args.get( 'STRING' ),
                             ruleId=args.get( 'RULEID' ),
                             destPrefix=args.get( 'DSTPREFIX' ),
                             srcPrefix=args.get( 'SRCPREFIX' ),
                             detail=args.get( 'detail' ),
                             vrfName=args.get( 'VRF' ),
                             routeDistinguisher=None,
                             interfaceSet=args.get( 'SETID' ),
                             policy=args.get( 'POLICY_NAME' ) )

def ShowBgpFlowspecVpnBaseClass_handler( mode, args ):
   return doShowBgpFlowspec( mode,
                             nlriType=getNlriType( args ),
                             peerAddr=args.get( 'PEER' ),
                             routeType=args.get( 'ROUTE_TYPE' ),
                             rule=args.get( 'STRING' ),
                             ruleId=args.get( 'RULEID' ),
                             destPrefix=args.get( 'DSTPREFIX' ),
                             srcPrefix=args.get( 'SRCPREFIX' ),
                             detail=args.get( 'detail' ),
                             vrfName=args.get( 'VRF' ),
                             routeDistinguisher=args.get( 'RD_VAL' ),
                             interfaceSet=args.get( 'SETID' ),
                             policy=None )


def ShowBgpFlowspecSummaryCmd_handler( mode, args ):
   nlriAfiSafi = getNlriType( args )
   vrfName = args.get( 'VRF' )
   return doShowBgpFlowspecSummary( mode, nlriAfiSafi, vrfName=vrfName )

@ArBgpShowOutput( 'doShowBgpDebugPolicyFlowspec', arBgpModeOnly=True )
def policyDebugFlowspecHandler( mode, args ):
   if 'ipv4' in args:
      nlriType = 'ipv4Flowspec'
   elif 'ipv6' in args:
      nlriType = 'ipv6Flowspec'
   elif 'vpn-ipv4' in args:
      nlriType = 'ipv4FlowspecVpn'
   elif 'vpn-ipv6' in args:
      nlriType = 'ipv6FlowspecVpn'
   else:
      assert False, "Unexpected afi"

   nlriAttrs = { 'flowspecRule' : args[ 'FLOWSPEC_RULE' ] }

   additionalParams = {}
   if args.pop( 'rule', None ):
      additionalParams.update( { 'flowspecRule' : args.pop( 'FLOWSPEC_RULE', '' ) } )

   return ArBgpCliHandler.policyDebugHandlerCommon( mode, args, nlriType, nlriAttrs,
                                    additionalParams )
