#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import ConfigMount
import LazyMount
from CliPlugin.BgpGroupCli import (
   BgpGroupConfigMode,
   BgpBuiltinGroupConfigMode
)
from CliPlugin import IraIpCli
from CliPlugin.GroupLib import Group
from CliPlugin.MaintenanceGroupCli import MaintenanceGroup
from CliPlugin.MaintenanceCliLib import (
   bgpGroupType,
   bgpNeighVrfBuiltinGroupPrefix,
   builtinGroupType,
)
from CliPlugin.MaintenanceModels import bgpMaintenanceGroupCleanupHook
from BgpLib import createKey
from IpLibConsts import VRFNAMES_RESERVED

globalBgpConfig = None
globalBgpGroupStatusDir = None
globalBgpGroupConfigDir = None

#
# BgpGroup class: Holds cli state for each configured peer group
#
class BgpGroup( Group ):
   def __init__( self, groupName ):
      self.name_ = groupName
      groupKey = Tac.Value( 'Group::GroupKey', bgpGroupType, groupName )
      Group.__init__( self, groupKey )

   def addGroup( self ):
      globalBgpGroupConfigDir.newConfig( self.name_ )
      Group.addGroup( self )

   def remGroup( self ):
      maintenanceGroup = MaintenanceGroup( self.key_ )
      maintenanceGroup.remGroup()
      if self.name_ in globalBgpGroupConfigDir.config:
         del globalBgpGroupConfigDir.config[ self.name_ ]
         Group.remGroup( self )

   def name( self ):
      return self.name_

   def neighborKey( self, peerOrAddr ):
      return createKey( peerOrAddr )

   def addPeer( self, peerOrAddr ):
      key = self.neighborKey( peerOrAddr )
      config = globalBgpGroupConfigDir.config.get( self.name_ )
      if not config:
         return
      config.neighbor[ key ] = True

   def remPeer( self, peerOrAddr ):
      key = self.neighborKey( peerOrAddr )
      config = globalBgpGroupConfigDir.config.get( self.name_ )
      if not config:
         return
      del config.neighbor[ key ]

   def addProfile( self, profileKey ):
      groupConfig = globalBgpGroupConfigDir.config.get( self.name_ )
      if not groupConfig:
         return
      groupConfig.profile[ profileKey ] = True

   def remProfile( self, profileType ):
      groupConfig = globalBgpGroupConfigDir.config.get( self.name_ )
      if not groupConfig:
         return
      for profile in groupConfig.profile:
         if profile.type == profileType:
            profileKey = Tac.Value( "Group::ProfileKey", type=profileType,
                                    name=profile.name )
            del groupConfig.profile[ profileKey ]

   def addVrf( self, vrfName ):
      groupConfig = globalBgpGroupConfigDir.config.get( self.name_ )
      if groupConfig:
         groupConfig.vrfName = vrfName

   def vrfName( self ):
      groupConfig = globalBgpGroupConfigDir.config.get( self.name_ )
      if groupConfig:
         return groupConfig.vrfName
      else:
         return None

#
# BgpBuiltinGroup class: Holds cli state for each builtin peer group
#
class BgpBuiltinGroup( Group ):
   def __init__( self, groupName ):
      self.name_ = groupName
      groupKey = Tac.Value( 'Group::GroupKey', bgpGroupType, groupName )
      Group.__init__( self, groupKey )
      globalBgpGroupConfigDir.newConfig( self.name_ )
      group = globalBgpGroupConfigDir.config.get( self.name_ )
      group.origin = builtinGroupType
      Group.addGroup( self )

   def name( self ):
      return self.name_

   def remGroup( self ):
      maintenanceGroup = MaintenanceGroup( self.key_ )
      maintenanceGroup.remGroup()
      if self.name_ in globalBgpGroupConfigDir.config:
         del globalBgpGroupConfigDir.config[ self.name_ ]
         Group.remGroup( self )

def BgpGroupNeighborCmd_handler( mode, args ):
   mode.group().addPeer( args[ 'PEER' ] )

def BgpGroupNeighborCmd_noOrDefaultHandler( mode, args ):
   mode.group().remPeer( args[ 'PEER' ] )

def BgpGroupVrfCmd_changeVrf( mode, vrfName ):
   groupConfig = globalBgpGroupConfigDir.config.get( mode.group().name() )
   if not groupConfig:
      return
   if len( vrfName ) > Tac.Type( "L3::VrfName" ).maxLength:
      # pylint: disable-next=consider-using-f-string
      mode.addError( "'%s' too long: must be no more than %d characters"
                     % ( vrfName, Tac.Type( "L3::VrfName" ).maxLength ) )
      return
   IraIpCli.warnIfRoutingDisabled( mode, vrfName )
   if groupConfig.vrfName == vrfName:
      return
   mode.group().addVrf( vrfName )

def BgpGroupVrfCmd_handler( mode, args ):
   vrfName = args[ 'VRF_NAME' ]
   if vrfName in VRFNAMES_RESERVED:
      # pylint: disable-next=consider-using-f-string
      mode.addError( "vrf name %s is reserved." % vrfName )
      return
   BgpGroupVrfCmd_changeVrf( mode, vrfName )

def BgpGroupVrfCmd_noOrDefaultHandler( mode, args ):
   vrfName = args.get( 'VRF_NAME' )
   if mode.group().vrfName() == vrfName or not vrfName:
      BgpGroupVrfCmd_changeVrf( mode, 'default' )

def GroupBgpCmd_handler( mode, args ):
   groupName = args[ 'BGP_GROUP_NAME' ]
   if groupName.startswith( bgpNeighVrfBuiltinGroupPrefix ):
      bgpBuiltinGroup = BgpBuiltinGroup( groupName )
      bgpBuiltinGroup.addGroup()
      childMode = mode.childMode( BgpBuiltinGroupConfigMode,
                                  bgpBuiltinGroup=bgpBuiltinGroup )
      mode.session_.gotoChildMode( childMode )
      return
   bgpGroup = BgpGroup( groupName )
   bgpGroup.addGroup()
   childMode = mode.childMode( BgpGroupConfigMode, bgpGroup=bgpGroup )
   mode.session_.gotoChildMode( childMode )

def GroupBgpCmd_noOrDefaultHandler( mode, args ):
   GroupBgpCmd_delBgpGroupConfig( mode, args[ 'BGP_GROUP_NAME' ] )

def GroupBgpCmd_delBgpGroupConfig( mode, groupName ):
   if groupName.startswith( bgpNeighVrfBuiltinGroupPrefix ):
      bgpBuiltinGroup = BgpBuiltinGroup( groupName )
      bgpBuiltinGroup.remGroup()
      return
   bgpGroup = BgpGroup( groupName )
   bgpGroup.remGroup()

def MaintenanceUnitGroupBgpCmd_handler( mode, args ):
   groupName = args[ 'BGP_GROUP_NAME' ]
   groupKey = Tac.Value( "Group::GroupKey", type=bgpGroupType,
                         name=groupName )
   mode.maintenanceUnit_.addGroup( groupKey )

def MaintenanceUnitGroupBgpCmd_noOrDefaultHandler( mode, args ):
   groupName = args[ 'BGP_GROUP_NAME' ]
   groupKey = Tac.Value( "Group::GroupKey", type=bgpGroupType,
                         name=groupName )
   mode.maintenanceUnit_.remGroup( groupKey )

def Plugin( entityManager ):
   global globalBgpGroupConfigDir, globalBgpConfig
   global globalBgpGroupStatusDir
   bgpMaintenanceGroupCleanupHook.addExtension( GroupBgpCmd_delBgpGroupConfig )
   globalBgpGroupConfigDir = ConfigMount.mount( entityManager,
      'group/config/bgp', 'BgpGroup::ConfigDir', 'w' )
   globalBgpConfig = LazyMount.mount( entityManager, 'routing/bgp/config',
      'Routing::Bgp::Config', 'r' )
   globalBgpGroupStatusDir = LazyMount.mount( entityManager,
      'group/status/bgp', 'BgpGroup::StatusDir', 'r' )
