# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AgentCommandRequest
from CliDynamicSymbol import CliDynamicPlugin
import CliModel
from CliPlugin.ArBgpCli import (
   ArBgpCliCommand,
)
from CliPlugin.RoutingBgpShowCli import (
   getCommunityValuesScalarList,
)
from CliPlugin.BgpCliHelperCli import convertPeerAddr
from CliPlugin.RoutingBgpShowCli import ArBgpShowOutput
from CliPlugin.TechSupportCli import showTechSupportExec
from IpLibConsts import DEFAULT_VRF
from Toggles import RoutingLibToggleLib

ArBgpCliHandler = CliDynamicPlugin( "ArBgpCliHandler" )
ArBgpCliLabelBindingsHelper = CliDynamicPlugin( "ArBgpCliLabelBindingsHelper" )
BgpLsCliModels = CliDynamicPlugin( "BgpLsCliModels" )

labelBindingsCliHelper = None

@ArBgpShowOutput( 'ShowBgpLsSummary', arBgpModeOnly=True )
def handlerShowBgpLsSummary( mode, args ):
   return ArBgpCliHandler.doShowIpBgpSummaryAcrImpl( mode, vrfName=DEFAULT_VRF,
                                                    nlriAfiSafi='linkState' )

class BgpLsCliHelperCommand( ArBgpCliCommand ):
   def __init__( self, mode, command, **kwargs ):
      ArBgpCliCommand.__init__( self, mode, command, vrfName=DEFAULT_VRF,
            nlriAfiSafi='linkState', disableFork=True )
      self.mode = mode
      if self.mode.session_.outputFormat_ == 'json':
         self.addParam( 'json' )
      if 'routeType' not in kwargs:
         kwargs[ 'routeType' ] = 'routes'
      for k, v in kwargs.items():
         if v:
            self.addParam( k, v )

   def run( self, **kwargs ):
      AgentCommandRequest.runCliPrintSocketCommand(
            self.entityManager, 'BgpCliHelper', self.command, self.paramString(),
            self.mode, keepalive=True )
      return CliModel.cliPrinted( BgpLsCliModels.BgpLsRoutesVrfModel )

def doShowBgpLs( mode, **kwargs ):
   return BgpLsCliHelperCommand( mode, 'show bgp link-state', **kwargs ).run()

@ArBgpShowOutput( 'ShowBgpLsNlri', arBgpModeOnly=True )
def handlerShowBgpLsNlri( mode, args ):
   return doShowBgpLs( mode,
                        nlriType=args.get( 'NLRI_TYPE' ),
                        detail=args.get( "detail" ) )

@ArBgpShowOutput( 'ShowBgpLsRoutes', arBgpModeOnly=True )
def handlerShowBgpLsRoutes( mode, args ):
   kwargs = {}
   kwargs[ 'routeType' ] = args.get( "route-type" )
   kwargs[ 'filtered' ] = args.get( "filtered" )
   kwargs[ 'peerAddr' ] = args.get( "PEER" )
   kwargs[ 'nlriType' ] = args.get( 'NLRI_TYPE' )
   kwargs[ 'detail' ] = args.get( "detail" )
   # Flatten the communityValues args
   commValuesAndExact = args.get( "communityValuesAndExact" )
   if commValuesAndExact:
      values = commValuesAndExact.get( "communityValues" )
      commValues = getCommunityValuesScalarList( values )
      kwargs[ 'commValues' ] = "|".join( [ str( c ) for c in commValues ] )
      kwargs[ 'standardCommunitiesExactMatch' ] = commValuesAndExact.get( "exact" )
   convertPeerAddr( kwargs )
   return doShowBgpLs( mode, **kwargs )

# List of show commands to run with 'show tech extended bgp link-state'
def _showTechLsCmds():
   return [
            'show isis neighbors detail',
            'show isis database traffic-engineering',
            'show bgp link-state summary',
            'show bgp link-state detail',
          ]

def handlerShowTechExtBgpLsSupport( mode, args ):
   cmd = 'show tech-support extended bgp link-state'
   # inCmdCallbacks is expected to be a list of tuples containing ( timestamp,
   # commandList function ). The timestamp is used to order between the
   # multiple show command list. In our case, there will just be one entry. So
   # pass 0 for the timestamp.
   return showTechSupportExec(
      mode, inputCmd=cmd, inCmdCallbacks=[ ( 0, _showTechLsCmds ) ] )

def showMplsBgpLsBindingsCmdHandler( mode, args ):
   assert labelBindingsCliHelper, 'Missing label bindings CLI helper'
   labelBindingsCliHelper.showConfigWarnings( mode )

   matchPrefix = args.get( 'PREFIX' )
   summary = 'summary' in args
   detail = summary or 'detail' in args

   # Render the bindings. For EPE, we're not interested in the peer bindings as they
   # will only contain dummy entries (for peer 0.0.0.0 with an imp-null label). Also,
   # we wish to hide the "Advertisements Received" column as we will only have
   # locally-originated bindings.
   fmt = mode.session_.outputFormat()
   return labelBindingsCliHelper.renderBindings(
      fmt, detail=detail, summary=summary, matchPrefix=matchPrefix,
      showPeerBindings=False, showRecvCount=False )

# -------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state
# -------------------------------------------------------------------------------
def Plugin( entityManager ):
   if RoutingLibToggleLib.toggleBgpLsProducerBgpEpeEnabled():
      global labelBindingsCliHelper
      labelBindingsCliHelper = ArBgpCliLabelBindingsHelper.BgpLabelBindingsCliHelper(
         entityManager, "mpls/labelBindingTables/bgpLs" )
