#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
from BgpLib import PeerConfigKey
import ConfigMount
from CliPlugin.IntfGroupCli import IntfBuiltinGroup
from CliPlugin.MaintenanceGroupCli import MaintenanceGroup
from CliPlugin.MaintenanceCliLib import (
   bgpGroupType,
   defaultProfile,
   linecardBuiltinGroupPrefix,
   bgpProfileType,
   dynamicGroupName,
   dynamicUnitName,
   dynamicUnitType,
   DynamicComponentRe,
   reservedProfileName
)
from CliPlugin.BgpMaintenanceCli import (
   ProfileBgpCmd,
   BgpProfile,
   BgpMaintenanceConfigMode,
   BgpMaintenanceProfileConfigMode
)
from TypeFuture import TacLazyType
import LazyMount
import re

globalBgpGroupConfigDir = None
globalBgpProfileConfigDir = None
globalMaintenanceUnitConfigDir = None
globalMaintenanceUnitStatusDir = None
globalMaintenanceUnitInputDir = None
globalDefaultBgpProfile = None

GroupOrigin = Tac.Type( "Group::GroupOrigin" )
GenPolicyName = TacLazyType( 'Routing::Policy::GenPolicyName' )
PolicyType = TacLazyType( 'Routing::Policy::PolicyType' )

def MaintenanceProfileBgpCmd_handleLineCardProfile( mode, profileName, add ):
   """
   Linecard Interface group has list of builtin Interface groups.
   Add/Remove profile for all maintenance groups in the list.
   """
   if add:
      profileKey = Tac.Value( 'Maintenance::Profile::ProfileKey',
                              type='profileTypeBgp', name=profileName )
   for _group in mode.group():
      assert( isinstance( _group, IntfBuiltinGroup ) and
              _group.name().startswith( linecardBuiltinGroupPrefix ) )
      maintenanceGroup = MaintenanceGroup( _group.key() )
      if add:
         maintenanceGroup.addProfile( profileKey )
      else:
         maintenanceGroup.remProfile( bgpProfileType, profileName )

def MaintenanceProfileBgpCmd_addProfile( mode, profileName ):
   if profileName.lower() == reservedProfileName:
      # pylint: disable-next=consider-using-f-string
      mode.addError( "The profile name '%s' is reserved." % profileName )
      return
   if isinstance( mode.group(), list ):
      MaintenanceProfileBgpCmd_handleLineCardProfile( mode,
                                                      profileName,
                                                      True )
   else:
      profileKey = Tac.Value( 'Maintenance::Profile::ProfileKey',
                              type='profileTypeBgp', name=profileName )
      maintenanceGroup = MaintenanceGroup( mode.group().key() )
      maintenanceGroup.addProfile( profileKey )

def MaintenanceProfileBgpCmd_remProfile( mode, profileName ):
   if isinstance( mode.group(), list ):
      MaintenanceProfileBgpCmd_handleLineCardProfile( mode,
                                                      profileName,
                                                      False )
   else:
      maintenanceGroup = MaintenanceGroup( mode.group().key() )
      maintenanceGroup.remProfile( bgpProfileType, profileName )

def MaintenanceProfileBgpCmd_handler( mode, args ):
   profileName = args[ 'PROFILE_NAME' ]
   MaintenanceProfileBgpCmd_addProfile( mode, profileName )

def MaintenanceProfileBgpCmd_noOrDefaultHandler( mode, args ):
   profileName = args.get( 'PROFILE_NAME' )
   MaintenanceProfileBgpCmd_remProfile( mode, profileName )

def ProfileBgpCmd_handler( mode, args ):
   profileName = args[ 'PROFILE_NAME' ]
   if profileName.lower() == reservedProfileName:
      # pylint: disable-next=consider-using-f-string
      mode.addError( "The profile name '%s' is reserved." % profileName )
      return
   bgpProfile = BgpProfile( profileName )
   bgpProfile.addProfile( profileName )
   childMode = mode.childMode( BgpMaintenanceProfileConfigMode,
                               bgpProfile=bgpProfile )
   mode.session_.gotoChildMode( childMode )

def ProfileBgpCmd_noOrDefaultHandler( mode, args ):
   profileName = args[ 'PROFILE_NAME' ]
   ProfileBgpCmd.delBgpProfileConfig_( mode, profileName )

def InitiatorPolicyCmd_setMaintenanceModePolicy( mode, policyName=None,
                                                 policyDirectionAttr=None,
                                                 isRcf=False ):
   profileName = mode.bgpProfile_.name_
   profileConfig = globalBgpProfileConfigDir.config.get( profileName )
   if not profileConfig:
      return

   policyType = PolicyType.rcfFunction if isRcf else PolicyType.routeMap
   policyObj = GenPolicyName( policyName, policyType )
   setattr( profileConfig, policyDirectionAttr, policyObj )

def InitiatorPolicyCmd_noMaintenanceModePolicy( mode, policyName=None,
                                                policyDirectionAttr=None,
                                                isRcf=None ):
   profileName = mode.bgpProfile_.name_
   profileConfig = globalBgpProfileConfigDir.config.get( profileName )
   if not profileConfig:
      return

   if ( getattr( profileConfig, policyDirectionAttr ).policyName == policyName and
        getattr( profileConfig, policyDirectionAttr ).isRcfFunction() == isRcf ):
      setattr( profileConfig, policyDirectionAttr, profileConfig.policyDefault )

def InitiatorPolicyCmd_delAllMaintenanceModePolicies( mode, isRcf=None ):
   profileName = mode.bgpProfile_.name_
   profileConfig = globalBgpProfileConfigDir.config.get( profileName )
   if not profileConfig:
      return

   if isRcf is None or ( profileConfig.policyInOut and
                         profileConfig.policyInOut.isRcfFunction() == isRcf ):
      profileConfig.policyInOut = profileConfig.policyDefault
   if isRcf is None or ( profileConfig.policyIn and
                         profileConfig.policyIn.isRcfFunction() == isRcf ):
      profileConfig.policyIn = profileConfig.policyDefault
   if isRcf is None or ( profileConfig.policyOut and
                         profileConfig.policyOut.isRcfFunction() == isRcf ):
      profileConfig.policyOut = profileConfig.policyDefault

def getPolicyNameAndPolicyType( args ):
   policyName = None
   isRcf = None
   if args.get( "rcf" ):
      isRcf = True
      if args.get( "FUNCTION" ):
         policyName = args.get( "FUNCTION" )[ : -2 ]
   elif args.get( "route-map" ):
      isRcf = False
      if args.get( 'MAPNAME' ):
         policyName = args.get( 'MAPNAME' )
   else:   
      assert False

   return policyName, isRcf

def InitiatorPolicyCmd_handler( mode, args ):
   policyName, isRcf = getPolicyNameAndPolicyType( args )
   InitiatorPolicyCmd_setMaintenanceModePolicy( mode,
      policyName=policyName,
      policyDirectionAttr=args[ 'directionAttr' ], isRcf=isRcf )

def InitiatorPolicyCmd_noOrDefaultHandler( mode, args ):
   policyName, isRcf = getPolicyNameAndPolicyType( args )
   if policyName:
      InitiatorPolicyCmd_noMaintenanceModePolicy(
         mode, policyName=policyName,
         policyDirectionAttr=args[ 'directionAttr' ], isRcf=isRcf )
   else:
      InitiatorPolicyCmd_delAllMaintenanceModePolicies( mode, isRcf=isRcf )

def BgpMaintenanceConfigModeCmd_handler( mode, args ):
   peer = args[ 'PEER' ]
   vrfName = args.get( 'VRFNAME', 'default' )
   peerKey = PeerConfigKey( peer )
   peerString = peerKey.stringValue
   # Create a new dynamic unit and group and add it to input
   # Not having any profile in the dynamic group would mean default profile

   unitName = dynamicUnitName( peerString, vrfName=vrfName )
   groupName = dynamicGroupName( peerString, vrfName=vrfName )

   dynamicUnit = globalMaintenanceUnitConfigDir.newConfig( unitName )
   dynamicUnit.unitType = dynamicUnitType
   dynamicGroup = globalBgpGroupConfigDir.newConfig( groupName )
   dynamicGroup.origin = GroupOrigin.dynamic
   dynamicGroup.vrfName = vrfName

   dynamicGroup.neighbor[ peerKey ] = True
   groupKey = Tac.Value( 'Group::GroupKey', bgpGroupType, groupName )
   dynamicUnit.group[ groupKey ] = True

   if not isinstance( peer, str ):
      peer = peer.stringValue

      childMode = mode.childMode( BgpMaintenanceConfigMode,
                                  bgpPeer=peer, vrfName=vrfName )
      mode.session_.gotoChildMode( childMode )

def BgpMaintenanceConfigModeCmd_noOrDefaultHandler( mode, args ):
   peer = args[ 'PEER' ]
   vrfName = args.get( 'VRFNAME', 'default' )
   peerString = PeerConfigKey( peer ).stringValue
   unitName = dynamicUnitName( peerString, vrfName=vrfName )
   groupName = dynamicGroupName( peerString, vrfName=vrfName )
   del globalMaintenanceUnitConfigDir.config[ unitName ]
   del globalBgpGroupConfigDir.config[ groupName ]
   del globalMaintenanceUnitInputDir.unit[ unitName ]

def QuiesceCmd_enterMaintenanceModeBgp( mode, peer, vrfName=None ):
   if not vrfName:
      vrfName = 'default'
   peerString = PeerConfigKey( peer ).stringValue
   unitName = dynamicUnitName( peerString, vrfName=vrfName )
   globalMaintenanceUnitInputDir.unit[ unitName ] = True

def QuiesceCmd_exitMaintenanceModeBgp( mode, peer, vrfName=None ):
   if not vrfName:
      vrfName = 'default'
   peerString = PeerConfigKey( peer ).stringValue
   unitName = dynamicUnitName( peerString, vrfName=vrfName )
   del globalMaintenanceUnitInputDir.unit[ unitName ]

def QuiesceCmd_handler( mode, args ):
   bgpVrfRe = "<Dynamic.+><.+><vrf-(.+)>"
   vrf = re.search( bgpVrfRe, mode.bgpPeer_ ).group( 1 )
   peer = re.search( DynamicComponentRe, mode.bgpPeer_ ).group( 1 )
   if vrf != 'default':
      QuiesceCmd_enterMaintenanceModeBgp( mode, peer, vrf )
   else:
      QuiesceCmd_enterMaintenanceModeBgp( mode, peer )

def QuiesceCmd_noOrDefaultHandler( mode, args ):
   bgpVrfRe = "<Dynamic.+><.+><vrf-(.+)>"
   vrf = re.search( bgpVrfRe, mode.bgpPeer_ ).group( 1 )
   peer = re.search( DynamicComponentRe, mode.bgpPeer_ ).group( 1 )
   if vrf != 'default':
      QuiesceCmd_exitMaintenanceModeBgp( mode, peer, vrf )
   else:
      QuiesceCmd_exitMaintenanceModeBgp( mode, peer )

def ProfileBgpDefaultCmd_handler( mode, args ):
   profileName = args[ 'PROFILE_NAME' ]
   if profileName.lower() == reservedProfileName:
      # pylint: disable-next=consider-using-f-string
      mode.addError( "The profile name '%s' is reserved." % profileName )
      return
   globalDefaultBgpProfile.profileName = profileName

def ProfileBgpDefaultCmd_noOrDefaultHandler( mode, args ):
   profileName = args[ 'PROFILE_NAME' ]
   if profileName == globalDefaultBgpProfile.profileName:
      globalDefaultBgpProfile.profileName = defaultProfile

def Plugin( entityManager ):
   global globalBgpProfileConfigDir
   global globalBgpGroupConfigDir
   global globalDefaultBgpProfile
   global globalMaintenanceUnitConfigDir
   global globalMaintenanceUnitStatusDir
   global globalMaintenanceUnitInputDir
   globalBgpProfileConfigDir = ConfigMount.mount(
      entityManager, 'maintenance/profile/config/bgp',
      'BgpMaintenanceProfile::ConfigDir', 'w' )
   globalDefaultBgpProfile = ConfigMount.mount(
      entityManager, 'maintenance/profile/config/default/bgp',
      'Maintenance::Profile::DefaultProfile', 'w' )
   globalBgpGroupConfigDir = ConfigMount.mount( entityManager,
                                                'group/config/bgp',
                                                'BgpGroup::ConfigDir', 'w' )
   globalMaintenanceUnitConfigDir = ConfigMount.mount(
      entityManager, 'maintenance/unit/config',
      'Maintenance::Unit::ConfigDir', 'w' )
   globalMaintenanceUnitStatusDir = LazyMount.mount(
      entityManager, 'maintenance/unit/status',
      'Maintenance::Unit::StatusDir', 'r' )
   globalMaintenanceUnitInputDir = ConfigMount.mount(
      entityManager, 'maintenance/unit/input/cli',
      'Maintenance::Unit::Input', 'w' )
