#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AgentCommandRequest
import Arnet
from CliDynamicSymbol import CliDynamicPlugin
import CliModel
from CliPlugin.ArBgpCli import (
   ArBgpCliCommand,
)
from CliPlugin.BgpCliHelperCli import convertPeerAddr
from CliPlugin.RoutingBgpShowCli import (
   routeSummaryVrfModel,
   doShowBgpOutputNotSupported,
)
from CliPlugin.WanTEShowCli import getVniAvtId
from IpLibConsts import (
   ALL_VRF_NAME,
   DEFAULT_VRF,
)
import Tac

ArBgpCliHandler = CliDynamicPlugin( "ArBgpCliHandler" )
BgpCliHelperCliHandler = CliDynamicPlugin( "BgpCliHelperCliHandler" )

# --------------------------------------------------------------------------------
# "show bgp sr-te [ ipv4 | ipv6 ] summary [ vrf <vrfName> ]
# --------------------------------------------------------------------------------
# pylint: disable-msg=inconsistent-return-statements
def doShowBgpSrTeSummary( mode, addressFamily=None, vrfName=DEFAULT_VRF ):
   if vrfName not in ( DEFAULT_VRF, ALL_VRF_NAME ):
      return doShowBgpOutputNotSupported( mode )
   if addressFamily is not None:
      nlriAfiSafi = 'ipv4SrTe' if addressFamily == 'ipv4' else 'ipv6SrTe'
      return ArBgpCliHandler.doShowIpBgpSummaryAcrImpl( mode,
                                        vrfName=vrfName,
                                        nlriAfiSafi=nlriAfiSafi )
   else:
      return ArBgpCliHandler.doShowIpBgpSummaryAcrImpl( mode,
                                        vrfName=vrfName,
                                        nlriAfiSafi='ipv4SrTe',
                                        extraNlriAfiSafis=[ 'ipv6SrTe' ] )

def showCmdSupported( **kwargs ):
   vrfName = kwargs.get( 'vrfName', DEFAULT_VRF )
   if vrfName not in ( DEFAULT_VRF, ALL_VRF_NAME ):
      return False
   return True

class SrTePolicyCliHelperCommand( ArBgpCliCommand ):
   def __init__( self, mode, command, **kwargs ):
      vrfName = kwargs.pop( 'vrfName', DEFAULT_VRF )
      addressFamily = kwargs.pop( 'addressFamily', None )
      if addressFamily is None:
         # we need to show both AFI=1 and AFI=2 paths
         nlriAfiSafi = 'ipv4SrTe'
         extraNlriAfiSafis = [ 'ipv6SrTe' ]
      else:
         nlriAfiSafi = 'ipv4SrTe' if addressFamily == 'ipv4' else 'ipv6SrTe'
         extraNlriAfiSafis = None

      super().__init__(
         mode, command, vrfName=vrfName, nlriAfiSafi=nlriAfiSafi,
         extraNlriAfiSafis=extraNlriAfiSafis, transportAfi=None, disableFork=True )

      self.mode = mode

      if self.mode.session_.outputFormat_ == 'json':
         self.addParam( 'json' )

      BgpCliHelperCliHandler.convertCommunityValues( mode, kwargs )

      if 'routeType' not in kwargs:
         kwargs[ 'routeType' ] = 'routes'

      for k, v in kwargs.items():
         if v:
            self.addParam( k, v )

   def run( self, **kwargs ):
      AgentCommandRequest.runCliPrintSocketCommand(
         self.entityManager, 'BgpCliHelper', self.command, self.paramString(),
         self.mode, keepalive=True )

      return CliModel.cliPrinted( routeSummaryVrfModel )

# --------------------------------------------------------------------------------
# "show bgp sr-te [ ipv4 | ipv6 ] [ detail ] [ vrf <vrfName> ]
# "show bgp sr-te endpoint <> color <> distinguisher <> [detail] [ vrf <vrfName> ]"
# --------------------------------------------------------------------------------
def doShowBgpSrTeRoutes( mode, **kwargs ):
   if not showCmdSupported( **kwargs ):
      return doShowBgpOutputNotSupported( mode )
   ep = kwargs.pop( 'endpoint', None )
   color = kwargs.pop( 'color', None )
   distinguisher = kwargs.pop( 'distinguisher', None )
   if ep is not None and color is not None and distinguisher is not None:
      kwargs[ 'addr' ] = str( distinguisher ) + '|' + str( color ) + '|' + str( ep )
   return SrTePolicyCliHelperCommand( mode, 'show bgp sr-te', **kwargs ).run()

# ---------------------------------------------------------------------------------
# "show bgp sr-te [ ipv4 | ipv6 ] community [<com>] [exact] [detail] [vrf <name>]"
# ---------------------------------------------------------------------------------
def doShowBgpSrTeRoutesCommunities( mode, **kwargs ):
   if not showCmdSupported( **kwargs ):
      return doShowBgpOutputNotSupported( mode )
   commValues = kwargs.pop( 'values', None )
   kwargs[ 'commValues' ] = commValues
   return SrTePolicyCliHelperCommand( mode, 'show bgp sr-te', **kwargs ).run()

# -------------------------------------------------------------------------------
# "show bgp sr-te [ ipv4 | ipv6 ] community-list [<community-list name>]
#       [exact] [detail] [vrf <vrfName>]"
# -------------------------------------------------------------------------------
def doShowBgpSrTeRoutesCommunityList( mode, **kwargs ):
   if not showCmdSupported( **kwargs ):
      return doShowBgpOutputNotSupported( mode )
   return SrTePolicyCliHelperCommand( mode, 'show bgp sr-te', **kwargs ).run()

# -----------------------------------------------------------------------------
# "show bgp neighbors <neigh_addr> [ ipv4 | ipv6 ] sr-te
# <policies | received-policies | advertised-policies> [ detail ] [vrf <vrfName>]"
# -----------------------------------------------------------------------------
def doShowBgpSrTeNeighborRoutes( mode, **kwargs ):
   if not showCmdSupported( **kwargs ):
      return doShowBgpOutputNotSupported( mode )
   policyFilter = kwargs.pop( 'policyFilter', 'policies' )
   # policyFilter in ( 'policies', 'received-policies', 'advertised-policies' )
   # routeType in ( 'routes', 'received-routes', 'advertised-routes' )
   # Replace 'policies' with 'routes' to get routeType for simplicity
   kwargs[ 'routeType' ] = policyFilter.replace( 'policies', 'routes' )
   convertPeerAddr( kwargs )
   return SrTePolicyCliHelperCommand( mode, 'show bgp sr-te', **kwargs ).run()

# ---------------------------------------------------------------------------------
# "show bgp sr-te endpoint <> path-selection color-vrf <colorVrfName> avt <avtName>
#    router <IPv4 address> [ detail ] [ vrf <vrfName> ]"
# ---------------------------------------------------------------------------------
def handlerShowBgpSrTeDpsPolicyEndpoint( mode, args ):
   # Get VNI and AVT ID
   vni, avtId = getVniAvtId( mode, args.get( "COLOR_VRF" ), args.get( "AVT" ) )
   if not vni or not avtId:
      return

   # Get headend router ID
   routerId = args[ "ROUTER_ID" ]

   return doShowBgpSrTeRoutes(
         mode,
         endpoint=args[ "ENDPOINT_VALUE" ],
         color=Tac.Type( 'Avt::VniAndAvtId' ).makeVniAndAvtId( vni, avtId ),
         distinguisher=Arnet.IpAddr( routerId ).value,
         detail=args.get( "detail" ),
         vrfName=args.get( "VRF", DEFAULT_VRF ) )
