#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from BgpLib import NoOrDefault
from CliDynamicSymbol import CliDynamicPlugin
import CliPlugin.BmpCli as Globals
from CliPlugin.IraServiceCli import getEffectiveProtocolModel
from CliPlugin.RoutingBgpCli import (
   BgpCmdBaseClass,
   bgpNeighborConfig,
)
from CliPlugin.RoutingBgpNeighborCli import validatePeer
from IpLibConsts import DEFAULT_VRF
from IpLibTypes import ProtocolAgentModelType
import Tac

RoutingBgpCliHandler = CliDynamicPlugin( "RoutingBgpCliHandler" )

def cleanupBmpConfig( vrfName ):
   # Determine which attrs to automatically reset to default
   noCleanupAttrs = { 'parent', 'bmpStation', 'localPort', 'afiSafiExport',
                      'afiSafiExportFlag', 'statsInterval',
                      'locRibAfiSafiExport', 'locRibAfiSafiExportFlag',
                      'locRibMonitoringEnabled', 'ribinExportVrfFilter' }
   cleanupAttrs = { i.name for i in Globals.bmpConfig.tacType.attributeQ
                    if i.writable and i.name not in noCleanupAttrs }
   defaultBmpConfig = \
      Tac.root.newEntity( 'Routing::Bmp::BmpConfig', 'defaultBmp' )

   del Globals.bmpConfig.locRibMonitoringEnabled[ vrfName ]

   if vrfName == DEFAULT_VRF:
      Globals.bmpConfig.bmpStation.clear()
      if vrfName in Globals.bmpConfig.localPort:
         del Globals.bmpConfig.localPort[ vrfName ]
      # Reset remaining config attrs to default values
      for attr in cleanupAttrs:
         setattr( Globals.bmpConfig, attr, getattr( defaultBmpConfig, attr ) )
      Globals.bmpConfig.bmpActivate = False

def handlerRouterBgpMonitoring( mode, args ):
   Globals.bmpConfig.bmpActivate = True

def noOrDefaultHandlerRouterBgpMonitoring( mode, args ):
   Globals.bmpConfig.bmpActivate = Globals.bmpConfig.bmpActivateDefault

def handleNormalRouterBgpRibBestPathMonitoringBase( mode, args ):
   if getEffectiveProtocolModel( mode ) != ProtocolAgentModelType.multiAgent:
      mode.addWarning( "BGP RIB bestpaths monitoring is only supported in "
                        "multi-agent mode" )
   # change the default value as well: all inherithing VRF will be impacted
   Globals.bmpConfig.locRibMonitoringEnabledAllVrfDefault = True

def handleNoOrDefaultRouterBgpRibBestPathMonitoringBase( mode, args, noOrDefault ):
   if noOrDefault == NoOrDefault.DEFAULT:
      # By default, locRibMonitoring is None.
      #
      # Effectively (agent code): all vrf that don't have explicit config
      # will be reevaluated, their "monitoring enalbed" flag will be
      # set to true.
      Globals.bmpConfig.locRibMonitoringEnabledAllVrfDefault = None
   else: # noOrDefault == NoOrDefault.NO:
      # Effectively (agent code): all vrf that don't have explicit config
      # will be reevaluated, their "monitoring enalbed" flag will be
      # set to false.
      Globals.bmpConfig.locRibMonitoringEnabledAllVrfDefault = False

def handlerRouterBgpRibBestPathMonitoringBase( mode, args ):
   BgpCmdBaseClass.callHandler(
      handleNormalRouterBgpRibBestPathMonitoringBase,
      handleNoOrDefaultRouterBgpRibBestPathMonitoringBase,
      mode,
      args )

def noOrDefaultHandlerRouterBgpRibBestPathMonitoringBase( mode, args ):
   BgpCmdBaseClass.callNoOrDefaultHandler(
      handleNoOrDefaultRouterBgpRibBestPathMonitoringBase,
      mode,
      args )

def handleNormalRouterBgpRibBestPathMonitoringVrf( mode, args ):
   if getEffectiveProtocolModel( mode ) != ProtocolAgentModelType.multiAgent:
      mode.addWarning( "BGP RIB bestpaths monitoring is only supported in "
                        "multi-agent mode" )
   Globals.bmpConfig.locRibMonitoringEnabled[ mode.vrfName ] = True

def handleNoOrDefaultRouterBgpRibBestPathMonitoringVrf( mode, args, noOrDefault ):
   if noOrDefault == NoOrDefault.DEFAULT:
      # Set original behavior back: remove config.
      # Effectively (agent code): inherit this vrf configuration from
      # locRibMonitoringEnabledAllVrfDefault.
      del Globals.bmpConfig.locRibMonitoringEnabled[ mode.vrfName ]
   else: # noOrDefault == NoOrDefault.NO:
      # explicitly make this vrf "NO"
      Globals.bmpConfig.locRibMonitoringEnabled[ mode.vrfName ] = False

def handlerRouterBgpRibBestPathMonitoringVrf( mode, args ):
   BgpCmdBaseClass.callHandler(
      handleNormalRouterBgpRibBestPathMonitoringVrf,
      handleNoOrDefaultRouterBgpRibBestPathMonitoringVrf,
      mode,
      args )

def noOrDefaultHandlerRouterBgpRibBestPathMonitoringVrf( mode, args ):
   BgpCmdBaseClass.callNoOrDefaultHandler(
      handleNoOrDefaultRouterBgpRibBestPathMonitoringVrf,
      mode,
      args )

def handlerRouterBgpVrfAdjRibinMonitorDisabled( mode, args ):
   Globals.bmpConfig.ribinExportVrfFilter.add( mode.vrfName )

def noOrDefaultHandlerRouterBgpVrfAdjRibinMonitorDisabled( mode, args ):
   del Globals.bmpConfig.ribinExportVrfFilter[ mode.vrfName ]

def handlerRouterBgpNeighborBmp( mode, args ):
   peer = args[ 'PEER' ]
   validatePeer( mode, peer )
   config = bgpNeighborConfig( peer, vrfName=mode.vrfName )
   config.bmpActivatePresent = True
   config.bmpActivate = True
   RoutingBgpCliHandler.delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def noHandlerRouterBgpNeighborBmp( mode, args ):
   peer = args[ 'PEER' ]
   config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
   if config:
      config.bmpActivatePresent = True
      config.bmpActivate = not config.bmpActivatePresent
      RoutingBgpCliHandler.delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )

def defaultHandlerRouterBgpNeighborBmp( mode, args ):
   peer = args[ 'PEER' ]
   config = bgpNeighborConfig( peer, create=False, vrfName=mode.vrfName )
   if config:
      config.bmpActivatePresent = False
      config.bmpActivate = not config.bmpActivatePresent
      RoutingBgpCliHandler.delNeighborConfigIfDefault( peer, vrfName=mode.vrfName )
