#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AgentCommandRequest
import AgentDirectory
from Ark import switchTimeToUtc
import BmpAgent
import Cell
from CliDynamicSymbol import CliDynamicPlugin
from CliPlugin import AgentLibShowCommands
from CliPlugin import BmpShowCli as Globals
import LazyMount
import Tac
import Toggles.BmpToggleLib
from io import StringIO

BmpCliModels = CliDynamicPlugin( "BmpCliModels" )

bmpStatus = None
bgpClientConfig = None

AddrType = Tac.Type( 'Routing::Bmp::IpAddrOrHostname::AddrType' )
AfType = Tac.Type( 'Arnet::AddressFamily' )

adjRibOutExportToggle = Toggles.BmpToggleLib.toggleBmpAdjRibOutExportEnabled()

def isBmpRunning( mode ):
   return AgentDirectory.agentIsRunning( mode.entityManager.sysname(), 'Bmp' )

def runSocketCommand( mode, *args, **kwargs ):
   # The bmp agent will not be running in the case of a cohab test
   # In this case the agentCommandRequestSm will be started by createBmpRootSm
   if isBmpRunning( mode ) or mode.entityManager.isLocalEm():
      AgentCommandRequest.runSocketCommand( mode.entityManager, *args, **kwargs )

def getMonitoringStatus():
   # BGP Monitoring Status is set in
   # routing/bgp/export/clientconfig/adjribin/Bmp/bmpEnabled
   bgpExportBmpClientConfig = bgpClientConfig.entityPtr.get( 'Bmp' )
   return False if bgpExportBmpClientConfig is None \
         else bgpExportBmpClientConfig.bmpEnabled

def checkBmpRunning( func ):
   def checkBmpRunning_wrapper( mode, args ):
      if not isBmpRunning( mode ) and not mode.entityManager.isLocalEm():
         mode.addError( "Bmp agent is not running" )
         return None
      else:
         return func( mode, args )

   return checkBmpRunning_wrapper

def afiSafiStr( afiSafi ):
   if afiSafi == Tac.Value( 'Routing::Bgp::AfiSafi', 'afiIpv4', 'safiUnicast' ):
      afiStr = 'IPv4 Unicast'
   elif afiSafi == Tac.Value( 'Routing::Bgp::AfiSafi', 'afiIpv6', 'safiUnicast' ):
      afiStr = 'IPv6 Unicast'
   elif afiSafi == Tac.Value( 'Routing::Bgp::AfiSafi', 'afiIpv6', 'safiMplsLabels' ):
      afiStr = 'IPv6 Labeled Unicast'
   elif afiSafi == Tac.Value( 'Routing::Bgp::AfiSafi', 'afiIpv4', 'safiMplsVpn' ):
      afiStr = 'VPN IPv4'
   elif afiSafi == Tac.Value( 'Routing::Bgp::AfiSafi', 'afiIpv6', 'safiMplsVpn' ):
      afiStr = 'VPN IPv6'
   elif afiSafi == Tac.Value( 'Routing::Bgp::AfiSafi', 'afiIpv4', 'safiFlowspec' ):
      afiStr = 'Flow Specification IPv4'
   elif afiSafi == Tac.Value( 'Routing::Bgp::AfiSafi', 'afiIpv6', 'safiFlowspec' ):
      afiStr = 'Flow Specification IPv6'
   else:
      afiStr = 'Unknown'
   return afiStr

def afiSafiExportConfig():
   afiSafiExport = {}
   for afiSafi, enabled in Globals.bmpConfig.afiSafiExportFlag.items():
      afiSafiExport[ afiSafiStr( afiSafi ) ] = enabled
   return afiSafiExport

@checkBmpRunning
def handlerBgpMonitoringActiveCmd( mode, args ):
   station = args.get( 'STATION' )
   allActiveStatus = {}
   if station:
      status = bmpStatus.bmpActiveStatus.get( station )
      if status:
         allActiveStatus[ station ] = status
   else:
      allActiveStatus = bmpStatus.bmpActiveStatus

   model = BmpCliModels.AllBmpActive()
   model.bgpAdjRibInExportStatus = \
      BmpCliModels.BgpAdjRibInExportStatus(
         monitoringEnabled=getMonitoringStatus() )

   model.stations = {}
   for name, status in allActiveStatus.items():
      stationModel = BmpCliModels.BmpActive()
      stationModel.address = BmpCliModels.StationHostnameOrAddress()
      if status.ipAddr.af != AfType.ipunknown:
         stationModel.address.ip = status.ipAddr
      stationModel.vrfName = status.vrfName
      stationModel.remotePort = status.remotePort
      stationModel.retryInterval = status.retryInterval

      stationModel.connected = status.connected
      if not status.connected:
         reason = status.reason
         if reason:
            stationModel.reason = reason[ 0 ].lower() + reason[ 1 : ]
         if status.nextRetryTime and status.nextRetryTime != float( "inf" ):
            stationModel.retryTime = switchTimeToUtc( status.nextRetryTime )

      stationModel.connectionAttempts = status.connectionAttemptCount
      stationModel.connectionSuccesses = status.connectionSuccessCount
      stationModel.connectionErrors = status.connectionErrorCount

      model.stations[ name ] = stationModel
   return model

@checkBmpRunning
def handlerBgpMonitoringActiveSummaryCmd( mode, args ):
   model = BmpCliModels.AllBmpActiveSummary()
   model.bgpAdjRibInExportStatus = \
      BmpCliModels.BgpAdjRibInExportStatus(
         monitoringEnabled=getMonitoringStatus() )

   model.stations = {}

   activeStatus = bmpStatus.bmpActiveStatus
   for status in activeStatus.values():
      stationModel = BmpCliModels.BmpActiveSummary()
      stationModel.connected = status.connected
      if not status.connected and status.nextRetryTime and \
         status.nextRetryTime != float( "inf" ):
         stationModel.retryTime = switchTimeToUtc( status.nextRetryTime )
      model.stations[ status.station ] = stationModel
   return model

def stationToModel( config, status ):
   model = BmpCliModels.BmpStation()
   if config.description:
      model.description = config.description
   model.connectionMode = config.connectionMode
   model.vrfName = config.vrfName

   remoteHost = config.remoteHost
   model.address = BmpCliModels.StationHostnameOrAddress()
   if remoteHost != config.remoteHostDefault:
      if remoteHost.addrType == AddrType.ip:
         model.address.ip = remoteHost.ipAddr
      elif remoteHost.addrType == AddrType.hostname:
         model.address.hostName = remoteHost.host

   if config.connectionMode == 'active':
      model.remotePort = config.remotePort

   if config.ribInExportPolicyStationConfig or \
      config.ribOutExportPolicyStationConfig:
      kwargs = {
         'prePolicyExport': config.ribInExportPolicyStationConfig.prePolicyExport,
         'postPolicyExport': config.ribInExportPolicyStationConfig.postPolicyExport
      }
      if adjRibOutExportToggle:
         kwargs.update( { 'ribOutExport':
            config.ribOutExportPolicyStationConfig.postPolicyExport } )

      model.exportPolicyStationConfig = \
         BmpCliModels.BmpStationPolicies( **kwargs )

   model.connected = ( status.sessionSmStatus is not None and
                       status.sessionSmStatus.connected )
   model.state = status.stationSmStatus.stationState

   if model.state == 'up':
      model.upTime = switchTimeToUtc( status.stationSmStatus.lastUpTime )
   model.flapCount = status.stationSmStatus.flapCount
   return model

@checkBmpRunning
def handlerBgpMonitoringCmd( mode, args ):
   station = args.get( 'STATION' )
   if station:
      allStationStatus = {}
      status = bmpStatus.bmpStation.get( station )
      if status:
         allStationStatus[ station ] = status
   else:
      allStationStatus = bmpStatus.bmpStation

   model = BmpCliModels.AllBmpStation()
   model.bgpAdjRibInExportStatus = \
      BmpCliModels.BgpAdjRibInExportStatus(
         monitoringEnabled=getMonitoringStatus() )

   afiSafiExport = afiSafiExportConfig()
   kwargs = {
      'timestampMode': Globals.bmpConfig.timestampMode,
      'prePolicyExport': Globals.bmpConfig.ribInExportPolicyConfig.prePolicyExport,
      'postPolicyExport': Globals.bmpConfig.ribInExportPolicyConfig.postPolicyExport,
      'afiSafiExport': afiSafiExport,
      'exportSixPe': Globals.bmpConfig.exportSixPe,
      'exportIpv6LuTunnel': Globals.bmpConfig.exportIpv6LuTunnel }

   if adjRibOutExportToggle:
      kwargs.update( { 'ribOutExport':
         Globals.bmpConfig.ribOutExportPolicyConfig.postPolicyExport } )

   model.bmpGlobalConfig = BmpCliModels.BmpGlobalConfig( **kwargs )
   model.stations = {}

   for name, status in allStationStatus.items():
      config = Globals.bmpConfig.bmpStation.get( name )
      if not config:
         continue
      stationModel = stationToModel( config, status )

      if 'detail' in args:
         tcpModel = None
         buff = StringIO()
         command1 = 'bmpSocketStats' + status.station
         runSocketCommand( mode, BmpAgent.name, "state",
               command1, stringBuff=buff )
         try:
            dataDict = eval( buff.getvalue() ) # pylint: disable-msg=eval-used
            if 'tcpStats' in dataDict:
               tcpModel = BmpCliModels.BmpTcpStatistics()
               tcpStats = dataDict[ 'tcpStats' ]
               tcpModel.setAttrFromDict( tcpStats )
               stationModel.tcpSocketStatistics = tcpModel
         except SyntaxError:
            Globals.t0(
               "Error: Unable to retrieve TCP socket information correctly" )

      model.stations[ name ] = stationModel
   return model

@checkBmpRunning
def handlerBgpMonitoringInternalCmd( mode, args ):
   runSocketCommand( mode, BmpAgent.name, 'state', '' )

def getConvergenceModel( status ):
   stationModel = BmpCliModels.BmpStationConvergence()
   sessionStatus = status.sessionSmStatus
   stationModel.converged = sessionStatus.converged
   stationModel.eorStatistics = {}
   if stationModel.converged:
      stationModel.lastConvergedTimestamp = \
         switchTimeToUtc( sessionStatus.lastConvergedTimestamp )
      stationModel.convergenceTime = sessionStatus.convergenceTime

   eorStatCol = sessionStatus.eorStatCol
   for peerKey, eorStat in eorStatCol.items():
      eorStatModel = BmpCliModels.EorStatistic()
      eorStatModel.peerAddr = peerKey.addr
      eorStatModel.initialPeer = eorStat.initialPeer
      eorStatModel.pendingCount = len( eorStat.pendingEor )
      stationModel.eorStatistics[ peerKey.addr ] = eorStatModel

   return stationModel

@checkBmpRunning
def handlerBgpMonitoringInternalConvergenceCmd( mode, args ):
   allStationStatus = bmpStatus.bmpStation
   model = BmpCliModels.AllBmpConvergence()
   model.stations = {}

   for status in allStationStatus.values():
      stationModel = getConvergenceModel( status )
      model.stations[ status.station ] = stationModel
   return model

@checkBmpRunning
def handlerBgpMonitoringInternalSchedulerResetCmd( mode, args ):
   # Since it's deprecated, this should never get called, but
   # if it is, let's make sure that the user knows that it didn't
   # work.
   mode.addError( 'Use "clear agent Bmp task scheduler"' )

@checkBmpRunning
def handlerBgpMonitoringInternalSchedulerCmd( mode, args ):
   # Translate options to those used in AgentCli
   translation = { 'verbose': 'detail',
                   'history': 'history' }
   agentArgs = { 'AGENT': 'Bmp',
                 'SCHEDULER_OPTS':
                 [ j for i, j in translation.items() if i in args ] }
   return AgentLibShowCommands.showTaskSchedulerHandler( mode, agentArgs )

@checkBmpRunning
def handlerBgpMonitoringPassiveCmd( mode, args ):
   vrfName = args.get( 'VRF' )
   if vrfName:
      listenStatus = {}
      for af in [ AfType.ipv4, AfType.ipv6 ]:
         passiveKey = Tac.ValueConst( 'Routing::Bmp::PassiveKey', vrfName, af )
         vrfStatus = bmpStatus.bmpListenStatus.get( passiveKey )
         if vrfStatus:
            listenStatus[ passiveKey ] = vrfStatus
   else:
      listenStatus = bmpStatus.bmpListenStatus

   model = BmpCliModels.AllBmpPassive()
   model.bgpAdjRibInExportStatus = \
      BmpCliModels.BgpAdjRibInExportStatus(
         monitoringEnabled=getMonitoringStatus() )

   model.listenersV4 = {}
   model.listenersV6 = {}

   for passiveKey, vrfStatus in listenStatus.items():
      # Populate BmpPassive model
      passiveModel = BmpCliModels.BmpPassive()
      passiveModel.port = vrfStatus.port
      passiveModel.listening = vrfStatus.listening
      reason = vrfStatus.reason
      if not vrfStatus.listening and reason:
         passiveModel.reason = reason[ 0 ].lower() + reason[ 1 : ]
      passiveModel.connectionsAccepted = vrfStatus.connectionsAccepted
      passiveModel.connectionsRejected = vrfStatus.connectionsRejected
      passiveModel.stations = {
         name: BmpCliModels.StationHostnameOrAddress( ip=str( ip ) )
         for ip,
         name in vrfStatus.stationHost.items() }

      # Add BmpPassive model into AllBmpPassive model
      if passiveKey.addrFamily == AfType.ipv4:
         model.listenersV4[ passiveKey.vrfName ] = passiveModel
      elif passiveKey.addrFamily == AfType.ipv6:
         model.listenersV6[ passiveKey.vrfName ] = passiveModel
   return model

@checkBmpRunning
def handlerBgpMonitoringPassiveSummaryCmd( mode, args ):
   model = BmpCliModels.AllBmpPassiveSummary()
   listenStatus = bmpStatus.bmpListenStatus
   model.bgpAdjRibInExportStatus = \
      BmpCliModels.BgpAdjRibInExportStatus(
         monitoringEnabled=getMonitoringStatus() )

   model.listenersV4 = {}
   model.listenersV6 = {}

   for passiveKey, vrfStatus in listenStatus.items():
      # Populate the BmpPassiveSummary model
      passiveModel = BmpCliModels.BmpPassiveSummary()
      passiveModel.port = vrfStatus.port
      passiveModel.stationCount = len( vrfStatus.stationHost )
      passiveModel.listening = vrfStatus.listening

      # Add BmpPassiveSummary model into AllBmpPassiveSummary model
      if passiveKey.addrFamily == AfType.ipv4:
         model.listenersV4[ passiveKey.vrfName ] = passiveModel
      elif passiveKey.addrFamily == AfType.ipv6:
         model.listenersV6[ passiveKey.vrfName ] = passiveModel

   return model

@checkBmpRunning
def handlerBgpMonitoringSummaryCmd( mode, args ):
   model = BmpCliModels.AllBmpStationSummary()
   model.bgpAdjRibInExportStatus = \
      BmpCliModels.BgpAdjRibInExportStatus(
         monitoringEnabled=getMonitoringStatus() )

   afiSafiExport = afiSafiExportConfig()
   kwargs = {
      'timestampMode': Globals.bmpConfig.timestampMode,
      'prePolicyExport': Globals.bmpConfig.ribInExportPolicyConfig.prePolicyExport,
      'postPolicyExport': Globals.bmpConfig.ribInExportPolicyConfig.postPolicyExport,
      'afiSafiExport': afiSafiExport,
      'exportSixPe': Globals.bmpConfig.exportSixPe,
      'exportIpv6LuTunnel': Globals.bmpConfig.exportIpv6LuTunnel }

   if adjRibOutExportToggle:
      kwargs.update( { 'ribOutExport':
         Globals.bmpConfig.ribOutExportPolicyConfig.postPolicyExport } )

   model.bmpGlobalConfig = BmpCliModels.BmpGlobalConfig( **kwargs )
   model.stations = {}

   stationStatus = bmpStatus.bmpStation
   for status in stationStatus.values():
      stationModel = BmpCliModels.BmpStationSummary()
      stationModel.state = status.stationSmStatus.stationState
      if stationModel.state == 'up':
         stationModel.upTime = switchTimeToUtc(
               status.stationSmStatus.lastUpTime )
      model.stations[ status.station ] = stationModel
   return model

# -------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
# -------------------------------------------------------------------------------
def Plugin( entityManager ):
   global bmpStatus
   global bgpClientConfig

   bmpStatus = LazyMount.mount(
      entityManager, 'routing/bmp/status', 'Routing::Bmp::BmpStatus', 'r' )
   bgpClientConfig = LazyMount.mount(
      entityManager, Cell.path( 'routing/bgp/export/clientconfig/adjribin' ),
      'Tac::Dir', 'ri' )
