# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCliUtil
import Cell
import CliGlobal
from CliPlugin import EthIntfCli
from CliPlugin import IntfCli
from CliPlugin import PhyCli
from CliPlugin.CableTestCliModel import (
   BaseTCableTestStatus,
   InterfaceBaseTCableTestStatuses
)
import EthIntfLib
import Intf
import LazyMount
import Tac

gv = CliGlobal.CliGlobal( dict( cableTestSliceDir=None ) )

def getCableTestStatusDir( intfName ):
   if Cell.cellType() == "fixed":
      return gv.cableTestSliceDir[ 'FixedSystem' ][ 'PhyIsland-FixedSystem' ]
   else:
      linecard = EthIntfLib.sliceName( intfName )
      if linecard in gv.cableTestSliceDir:
         # Linecard may not be managed by PhyIsland e.g. Tundra Linecard1
         return gv.cableTestSliceDir[ linecard ][ 'PhyIsland' ]
   return None

def getCableTestStatus( intfName ):
   cableTestStatusDir = getCableTestStatusDir( intfName )
   if not cableTestStatusDir:
      return None
   return cableTestStatusDir.cableTestStatus.get( intfName )

def filterIntfsWithCableTestSupport( intfRange ):
   supportedIntfs = []
   unsupportedIntfs = []
   for i in intfRange:
      if getCableTestStatus( i ):
         supportedIntfs.append( i )
      else:
         unsupportedIntfs.append( i )
   return [ supportedIntfs, unsupportedIntfs ]

# Returns BaseTCableTestStatus for specified intfs if they support cableTest
def getBaseTCableTestStatus( intfNameList ):
   intfsCableTestStatus = {}
   unsupportedIntfs = []
   for intfName in intfNameList:
      cableTestStatus = getCableTestStatus( intfName )
      if cableTestStatus:
         intfsCableTestStatus[ cableTestStatus.intfId ] = \
                              BaseTCableTestStatus().toModel( cableTestStatus )
      else:
         unsupportedIntfs.append( intfName )
   return [ intfsCableTestStatus, unsupportedIntfs ]

def getAllEthIntfs( mode ):
   intfNameList = []
   intfs = IntfCli.Intf.getAll( mode, intfType=EthIntfCli.EthPhyIntf )
   for intf in intfs:
      if not intf.name.startswith( 'Management' ):
         intfNameList.append( intf.name )
   return intfNameList

def baseTCableTestHandler( mode, args ):
   intfs = args[ 'INTFS' ]
   supportedIntfs, unsupportedIntfs = filterIntfsWithCableTestSupport( intfs )
   if not supportedIntfs:
      mode.addError( "Cable testing is not supported on specified interfaces" )
      return
   if unsupportedIntfs:
      unsupportedIntfStr = Intf.IntfRange.intfListToCanonical(
         unsupportedIntfs )[ 0 ]
      mode.addWarning( "Cable testing is not supported on " + unsupportedIntfStr )
   # Display warning that cable test will cause interface flaps
   supportedIntfStr = Intf.IntfRange.intfListToCanonical( supportedIntfs )[ 0 ]
   promptStr = f"! Testing cables will cause {supportedIntfStr}" \
               " to flap. Continue? [y/N]"
   if not BasicCliUtil.confirm( mode, prompt=promptStr, answerForReturn=False ):
      return
   # Trigger cable test for supported intfs
   now = Tac.now()
   for intfName in supportedIntfs:
      cfg = PhyCli.getPhyCliFeatureConfigForIntf( intfName )
      cfg.baseTCableTestGenId = now

def showIntfCableDetailHander( mode, args ):
   intfs = list( args.get( 'INTFS', () ) ) or getAllEthIntfs( mode )
   testStatuses, unsupportedIntfs = getBaseTCableTestStatus( intfs )
   if unsupportedIntfs:
      intfString = Intf.IntfRange.intfListToCanonical( unsupportedIntfs )[ 0 ]
      unsupportedStr = f"Cable testing is not supported on {intfString}"
      if len( unsupportedIntfs ) == len( intfs ):
         mode.addError( unsupportedStr )
         return None
      else:
         mode.addWarning( unsupportedStr )
   return InterfaceBaseTCableTestStatuses( interfaces=testStatuses )

def Plugin( entityManager ):
   gv.cableTestSliceDir = LazyMount.mount(
      entityManager,
      'hardware/phy/cabletest/status/slice',
      'Tac::Dir', 'ri' )
