# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliPlugin import AclCli
from CliPlugin.Capi import CapiConfigMode
from CliDynamicSymbol import CliDynamicPlugin
import CliGlobal
import BasicCliUtil
import ConfigMount
import LazyMount
from HttpServiceConstants import CapiCipherSuites
import Tac

HttpServiceHandler = CliDynamicPlugin( "HttpServiceHandler" )
DynamicMode = CliDynamicPlugin( "CapiVrf" )
CAPI_CONSTANTS = Tac.Type( "HttpService::Constants" )

# -----------------------------------------------------------------------------------
# Mount path holders ( Define all mount path holders here )
# -----------------------------------------------------------------------------------
gv = CliGlobal.CliGlobal(
   dict(
      capiConfig=None,
      cliConfig=None,
      capiExecRequest=None,
      serviceStatusDir=None,
      aclCheckpoint=None,
   )
)

# -------------------------------------------------------------------------------
# Command Helper Function
# -------------------------------------------------------------------------------
def shutdownCapi( mode ):
   gv.capiConfig.service[ 'http-commands' ].enabled = False

def capiNoVrf( mode, args ):
   serviceAclTypeVrfMap = gv.capiConfig.serviceAclTypeVrfMap
   vrfName = args.get( 'VRF' )
   if vrfName:
      for aclType in [ 'ip', 'ipv6' ]:
         key = Tac.Value( "Acl::AclTypeAndVrfName", aclType, vrfName )
         if key in serviceAclTypeVrfMap.aclName:
            del serviceAclTypeVrfMap.aclName[ key ]
   else:
      serviceAclTypeVrfMap.aclName.clear()

   HttpServiceHandler.removeVrfFromService( gv.capiConfig, 'http-commands', vrfName )

def capiClearHttpsCert( mode ):
   _updateHttpsCert( mode, '', '' )

def _updateHttpsCert( mode, certificate, key ):
   """ Updates the certificate and private key. An error is added to the mode
   if there is an error """
   import HttpServiceSsl # pylint: disable=import-outside-toplevel
   if certificate and key:
      error = HttpServiceSsl.validateCertificateAndKey( certificate, key )
      if error:
         mode.addError( error )
         return

      gv.capiConfig.httpsConfig.sslKey = key
      gv.capiConfig.httpsConfig.sslCertificate = certificate
   elif not certificate and not key:
      gv.capiConfig.httpsConfig.sslKey = ''
      gv.capiConfig.httpsConfig.sslCertificate = ''
   else:
      mode.addError( "Both a Certificate and Private Key must be configured" )

def capiClearHttpsKeyExchange( mode ):
   gv.capiConfig.httpsConfig.keyExchange.clear()
   gv.capiConfig.httpsConfig.hasKeyExchangeConfig = False

def capiClearHttpsCipher( mode ):
   gv.capiConfig.httpsConfig.ciphers.clear()
   gv.capiConfig.httpsConfig.hasCipherConfig = False

def capiClearHttpsMac( mode ):
   gv.capiConfig.httpsConfig.mac.clear()
   gv.capiConfig.httpsConfig.hasMacConfig = False

# -------------------------------------------------------------------------------
# BasicCli Command Handler Function
# -------------------------------------------------------------------------------
def enterCapiModeHandler( mode, args ):
   childMode = mode.childMode( CapiConfigMode )
   mode.session_.gotoChildMode( childMode )

def enterCapiModeNoOrDefaultHandler( mode, args ):
   shutdownCapi( mode )
   capiNoVrf( mode, {} )
   capiClearHttpsCert( mode )
   capiClearHttpsKeyExchange( mode )
   capiClearHttpsCipher( mode )
   capiClearHttpsMac( mode )
   if not gv.capiConfig.useHttpServiceCli:
      HttpServiceHandler.handleDefaultProtocol( "http" )
      HttpServiceHandler.handleDefaultProtocol( "https" )
      HttpServiceHandler.handleDefaultProtocol( "http localhost" )
      HttpServiceHandler.handleDefaultProtocol( "unix-socket" )
      gv.capiConfig.corsAllowedOrigins.clear()
      gv.capiConfig.httpsConfig.sslProfile = ''
      gv.capiConfig.qosDscp = 0
      gv.capiConfig.sessionTimeout = CAPI_CONSTANTS.defaultSessionTimeout
      gv.capiConfig.contentFrameAncestor = ""
      gv.capiConfig.defaultServicesEnabled = True
      gv.capiConfig.syslogLevel = gv.capiConfig.syslogLevelDefault
   if not HttpServiceHandler.serverConfigurationExists():
      gv.capiConfig.useHttpServiceCli = False
   gv.cliConfig.validateOutput = False

def resetDhParamsHandler( mode, args ):
   mode.addError( "Please use 'reset ssl diffie-hellman parameters' instead" )

def clearCountersHandler( mode, args ):
   gv.capiExecRequest.lastStatsResetTime = Tac.now()

def clearIpAclCountersHandler( mode, args ):
   if 'ip' in args:
      aclType = 'ip'
   elif 'ipv6' in args:
      aclType = 'ipv6'
   else:
      assert False

   capiStatus = gv.serviceStatusDir.get( 'http-commands' )
   aclStatus = capiStatus.aclStatus if capiStatus else None
   AclCli.clearServiceAclCounters( mode,
                                   aclStatus,
                                   gv.aclCheckpoint,
                                   aclType )

# -------------------------------------------------------------------------------
# CapiConfigMode Command Handler Function
# -------------------------------------------------------------------------------
def enterCapiVrfConfigMode( mode, args ):
   vrfName = args[ 'VRF' ]
   childMode = mode.childMode( DynamicMode.CapiVrfConfigMode, vrfName=vrfName )
   mode.session_.gotoChildMode( childMode )

def configureCapiHandler( mode, args ):
   shutdownCapi( mode )

def configureCapiNoHandler( mode, args ):
   if ( not gv.capiConfig.httpsConfig.enabled and
         not gv.capiConfig.httpConfig.enabled and
         not gv.capiConfig.localHttpConfig.enabled and
         not gv.capiConfig.unixConfig.enabled ):
      # No protocol is enabled
      mode.addWarning( "No protocols are enabled" )
   gv.capiConfig.service[ 'http-commands' ].enabled = True

def capiHttpsCertHandler( mode, args ):
   if not HttpServiceHandler.checkServerConfigNotSplit( mode ):
      return
   cmd = """protocol https certificate"""
   certificate = BasicCliUtil.getMultiLineInput( mode, cmd, lstrip=True,
                                                prompt="Enter TEXT certificate" )
   key = BasicCliUtil.getMultiLineInput( mode, cmd, lstrip=True,
                                        prompt="\nEnter TEXT private key" )
   _updateHttpsCert( mode, certificate, key )
   HttpServiceHandler.setUseHttpServiceCli( mode )

def capiHttpsCertNoOrDefaultHandler( mode, args ):
   capiClearHttpsCert( mode )
   HttpServiceHandler.setUseHttpServiceCli( mode )

def protocolHttpsCipherHandler( mode, args ):
   if not HttpServiceHandler.checkServerConfigNotSplit( mode ):
      return
   gv.capiConfig.httpsConfig.ciphers.clear()
   isDefault = True
   ciphers = [ key for key in CapiCipherSuites.CIPHERS if key in args ]
   if ciphers:
      gv.capiConfig.httpsConfig.hasCipherConfig = True
   for cipher in CapiCipherSuites.CIPHERS:
      isDefault &= cipher in ciphers

   if not isDefault:
      for c in ciphers:
         gv.capiConfig.httpsConfig.ciphers.enq( c )
   HttpServiceHandler.setUseHttpServiceCli( mode )

def protocolHttpsCipherNoOrDefaultHandler( mode, args ):
   capiClearHttpsCipher( mode )
   HttpServiceHandler.setUseHttpServiceCli( mode )

def protocolHttpsKeyExchangeHandler( mode, args ):
   if not HttpServiceHandler.checkServerConfigNotSplit( mode ):
      return
   gv.capiConfig.httpsConfig.keyExchange.clear()
   isDefault = True
   keyExchange = [ key for key in CapiCipherSuites.KEYEXCHANGE if key in args ]
   if keyExchange:
      gv.capiConfig.httpsConfig.hasKeyExchangeConfig = True
   for key in CapiCipherSuites.KEYEXCHANGE:
      isDefault &= key in keyExchange

   if not isDefault:
      for k in keyExchange:
         gv.capiConfig.httpsConfig.keyExchange.enq( k )
   HttpServiceHandler.setUseHttpServiceCli( mode )

def protocolHttpsKeyExchangeNoOrDefaultHandler( mode, args ):
   capiClearHttpsKeyExchange( mode )
   HttpServiceHandler.setUseHttpServiceCli( mode )

def protocolHttpsMacHandler( mode, args ):
   if not HttpServiceHandler.checkServerConfigNotSplit( mode ):
      return
   gv.capiConfig.httpsConfig.mac.clear()
   isDefault = True

   mac = [ key for key in CapiCipherSuites.MAC if key in args ]
   if mac:
      gv.capiConfig.httpsConfig.hasMacConfig = True
   for m in CapiCipherSuites.MAC:
      isDefault &= m in mac

   if not isDefault:
      for m in mac:
         gv.capiConfig.httpsConfig.mac.enq( m )
   HttpServiceHandler.setUseHttpServiceCli( mode )

def protocolHttpsMacNoOrDefaultHandler( mode, args ):
   capiClearHttpsMac( mode )
   HttpServiceHandler.setUseHttpServiceCli( mode )

def validateJsonOutputHandler( mode, args ):
   gv.cliConfig.validateOutput = True

def validateJsonOutputNoOrDefaultHandler( mode, args ):
   gv.cliConfig.validateOutput = False

def sessionTimeoutHandler( mode, args ):
   gv.capiConfig.sessionTimeout = args[ "TIMEOUT" ]

def sessionTimeoutNoOrDefaultHandler( mode, args ):
   gv.capiConfig.sessionTimeout = CAPI_CONSTANTS.defaultSessionTimeout

# -------------------------------------------------------------------------------
# CapiVrfConfigMode Command Handler Function
# -------------------------------------------------------------------------------
def vrfModeShutDownHandler( mode, args ):
   mode.shutdown()

def vrfModeShutDownNoHandler( mode, args ):
   mode.noShutdown()

def capiIpAclHandler( mode, args ):
   mode.setServiceAcl( 'ip', args[ 'ACL_NAME' ] )

def capiIpAclNoOrDefaultHandler( mode, args ):
   mode.deleteServiceAcl( 'ip' )

def capiIpv6AclHandler( mode, args ):
   mode.setServiceAcl( 'ipv6', args[ 'ACL_NAME' ] )

def capiIpv6AclNoOrDefaultHandler( mode, args ):
   mode.deleteServiceAcl( 'ipv6' )

# Plug-in definition:
def Plugin( entityManager ):
   gv.capiConfig = ConfigMount.mount( entityManager,
                                      "mgmt/capi/config",
                                      "HttpService::Config",
                                      "w" )
   gv.cliConfig = ConfigMount.mount( entityManager,
                                     "cli/config",
                                     "Cli::Config",
                                     "w" )
   gv.capiExecRequest = LazyMount.mount( entityManager,
                                         "mgmt/httpserver/execRequest",
                                         "HttpService::ExecRequest",
                                         "w" )
   gv.serviceStatusDir = LazyMount.mount( entityManager,
                                          "mgmt/httpserver/service",
                                          "Tac::Dir",
                                          "ri" )
   gv.aclCheckpoint = LazyMount.mount( entityManager,
                                       "mgmt/capi/aclCheckpoint",
                                       "Acl::CheckpointStatus",
                                       "w" )
