# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import errno

from CliPlugin import AclCli
from CliPlugin import CapiModel
import CliGlobal
import ConfigMount
from HttpServiceConstants import CapiCipherSuites
import LazyMount
import Tac

# -----------------------------------------------------------------------------------
# Mount path holders ( Define all mount path holders here )
# -----------------------------------------------------------------------------------
gv = CliGlobal.CliGlobal(
   dict(
      capiConfig=None,
      serverStatus=None,
      serviceStatusDir=None,
      sslStatus=None,
      aclCheckpoint=None,
   )
)

MGMT_SECURITY_SSL_CONSTANTS = Tac.Type( "Mgmt::Security::Ssl::Constants" )

# -------------------------------------------------------------------------------
# Show Command Helper Function
# -------------------------------------------------------------------------------
def readFile( path ):
   """ Return the contents of the path, or the empty string if it doesn't exist """
   try:
      with open( path ) as f:
         return f.read()
   except OSError as e:
      if e.errno != errno.ENOENT:
         raise
   return ""

# -------------------------------------------------------------------------------
# Capi Show Command Handler Function
# -------------------------------------------------------------------------------
def showCapiCertHandler( mode, args ):
   import HttpServiceSsl # pylint: disable=import-outside-toplevel
   profile = gv.capiConfig.httpsConfig.sslProfile
   status = gv.sslStatus.profileStatus.get( profile )
   certificate = ""

   # we don't have a valid capi cert
   if not gv.capiConfig.httpsConfig.sslCertificate:
      # if we have an ssl profile configured
      if profile:
         # we have valid ssl profile
         if status and status.state == "valid" and status.certKeyPath:
            certKey = readFile( status.certKeyPath )
            ( certificate, _ ) = HttpServiceSsl.parseCertKey( certKey )
         else: # we have an invalid ssl profile
            certificate = ""
      else: # we don't have an ssl profile configured
         certificate = readFile( HttpServiceSsl.getCertFilepath() )

   else: # we have a valid capi cert, always use it!
      certificate = readFile( HttpServiceSsl.getCertFilepath() )

   return CapiModel.CommandApiCertificate( certificate=certificate )

def showCapiCiphersHandler( mode, args ):
   cipherList = []
   if gv.serverStatus.cipherFilter:
      cipherList = gv.serverStatus.cipherFilter.split( ":" )
   else:
      cipherList = Tac.run(
         [
            'openssl',
            'ciphers',
            MGMT_SECURITY_SSL_CONSTANTS.defaultCipherSuite(),
         ],
         stdout=Tac.CAPTURE,
      ).strip().split( ':' )

   ciphers = []
   for cipher in cipherList:
      rfcName = CapiCipherSuites.cipherSuiteFilterToNameMap.get( cipher )
      if rfcName:
         ciphers.append( CapiModel.CipherSuite( cipherSuite=rfcName ) )
      else:
         # pylint: disable-next=consider-using-f-string
         mode.addWarning( "No conversion from OpenSSL cipher %s to RFC name" %
                          cipher )

   return CapiModel.CommandApiCipherSuites( enabledCipherSuites=ciphers )

def showCapiDhparamsHandler( mode, args ):
   constants = Tac.Type( "Mgmt::Security::Ssl::Constants" )
   dhparams = readFile( constants.dhParamPath() )
   return CapiModel.CommandApiDiffieHellman( diffieHellmanParameters=dhparams )

def showCapiAclHandler( mode, args ):
   aclType = 'ip' if 'ip' in args else 'ipv6'
   name = args[ '<aclNameExpr>' ]
   capiStatus = gv.serviceStatusDir.get( 'http-commands' )
   aclStatus = capiStatus.aclStatus if capiStatus else None
   return AclCli.showServiceAcl( mode,
                                 gv.capiConfig.serviceAclTypeVrfMap,
                                 aclStatus,
                                 gv.aclCheckpoint,
                                 aclType,
                                 name )

# Plug-in definition:
def Plugin( entityManager ):
   gv.capiConfig = ConfigMount.mount( entityManager,
                                      "mgmt/capi/config",
                                      "HttpService::Config",
                                      "w" )
   gv.serverStatus = LazyMount.mount( entityManager,
                                      "mgmt/httpserver/status",
                                      "HttpService::Status",
                                      "r" )
   gv.serviceStatusDir = LazyMount.mount( entityManager,
                                          "mgmt/httpserver/service",
                                          "Tac::Dir",
                                          "ri" )
   gv.sslStatus = LazyMount.mount( entityManager,
                                   "mgmt/security/ssl/status",
                                   "Mgmt::Security::Ssl::Status",
                                   "r" )
   gv.aclCheckpoint = LazyMount.mount( entityManager,
                                       "mgmt/capi/aclCheckpoint",
                                       "Acl::CheckpointStatus",
                                       "w" )
