# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCliUtil
from CliPlugin import CertRotationModel
from CliPlugin import Pki
from CliPlugin.CertRotation import getRotationMgr
import six
import SslCertKey
import Tac

rotationNotReadyError = 'certificate rotation not ready'
Constants = Tac.Type( 'Mgmt::Security::Ssl::Constants' )
rotationMgr = getRotationMgr()

def PkiCertGenerateSigningRequest_handler( mode, args ):
   if not rotationMgr.isReady():
      mode.addError( rotationNotReadyError )
      return None

   profileName = args[ 'PROFILE_NAME' ]
   keyAlgo = args[ 'KEY_ALGO' ]
   newKeyBits = args.get( 'RSA_BIT_LENGTH' )
   newEcCurve = args.get( 'EC_CURVE_NAME' )
   importTimeout = args.get( 'IMPORT_TIMEOUT', Constants.defImportTimeoutMins )
   digest = args.get( 'DIGEST', Constants.defaultDigest )
   signReqParams = args.get( 'SIGN_REQ_PARAMS' )
   try:
      signReqParams = Pki.getSignRequestParams( mode, signReqParams )
   except SslCertKey.SslCertKeyError as e:
      mode.addError( f'Error generating CSR ({str( e )})' )
      return None

   return rotationMgr.generateCsr( profileName, importTimeout,
                                   digest, signReqParams, keyAlgo,
                                   newKeyBits=newKeyBits,
                                   newEcCurve=newEcCurve )

def PkiCertificateRotationImport_handler( mode, args ):
   rotationId = args[ 'ROTATION_ID' ]
   if not rotationMgr.isReady():
      mode.addError( rotationNotReadyError )
      return None

   cmd = 'security pki certificate rotation import UUID'
   certPem = BasicCliUtil.getMultiLineInput( mode, cmd,
                                             prompt='Enter TEXT certificate' )
   return rotationMgr.importCertificate( rotationId, certPem )

def PkiCertificateRotationCommit_handler( mode, args ):
   rotationId = args[ 'ROTATION_ID' ]
   if not rotationMgr.isReady():
      mode.addError( rotationNotReadyError )
      return None

   return rotationMgr.commit( rotationId )

def PkiCertificateRotationCommitSslProfile_handler( mode, args ):
   if not rotationMgr.isReady():
      mode.addError( rotationNotReadyError )
      return None

   profileName = args.get( 'PROFILE_NAME' )
   dontWait = 'dont-wait' in args
   validateDates = 'ignore' not in args
   cmd = 'security pki certificate rotation commit ssl profile WORD'
   key = BasicCliUtil.getMultiLineInput( mode, cmd,
                                         prompt='Enter TEXT private key' )
   cert = BasicCliUtil.getMultiLineInput( mode, cmd,
                                          prompt='Enter TEXT certificate' )
   return rotationMgr.commitSslProfile( profileName, key, cert, validateDates,
                                        dontWait )

def PkiCertificateRotationClear_handler( mode, args ):
   rotationId = args[ 'ROTATION_ID' ]
   if not rotationMgr.isReady():
      mode.addError( rotationNotReadyError )
      return None

   return rotationMgr.clear( rotationId )

def ShowPkiCertRotations_handler( mode, args ):
   if not rotationMgr.isReady():
      mode.addError( rotationNotReadyError )
      return None
   detail = 'detail' in args
   result = CertRotationModel.CertificateRotations( _detail=detail )
   for rotationId, rr in six.iteritems( rotationMgr.getAllRotationReqs() ):
      rotationModel = rr.getCliRotationModel()
      if rotationModel is None:
         continue
      result.rotations[ rotationId ] = rotationModel
   return result

def ShowPkiCertRotation_handler( mode, args ):
   if not rotationMgr.isReady():
      mode.addError( rotationNotReadyError )
      return None

   rotationReq = rotationMgr.getRotationReq( args[ 'ROTATION_ID' ] )
   if rotationReq is None:
      mode.addError( f"Rotation {args[ 'ROTATION_ID' ]} not found" )
      return None

   result = rotationReq.getCliRotationModel()
   if result is None:
      mode.addError( f"Rotation {args[ 'ROTATION_ID' ]} not found" )
   return result

