#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliGlobal
import CliGuards
import CliMatcher
import CliMode.Clb
import CliParser
from CliPlugin.ClbConfig import ClbConfigMode
from CliPlugin.EthIntfCli import EthPhyAutoIntfType
from CliPlugin.SwitchIntfCli import SwitchAutoIntfType
import CliToken.LoadBalance
import ConfigMount
from Intf import IntfRange
import LazyMount
import Tac
from TypeFuture import TacLazyType

OffloadMethod = TacLazyType( "Clb::Hardware::OffloadMethod" )

gv = CliGlobal.CliGlobal( config=None, status=None, qosHwStatus=None,
                          hwCapabilities=None )

flowKw = CliMatcher.KeywordMatcher( "flow", "Configure flow related settings" )
portKw = CliMatcher.KeywordMatcher( "port", "Configure port related settings" )
groupKw = CliMatcher.KeywordMatcher( "group", "Configure port groups" )
exhaustionKw = CliMatcher.KeywordMatcher( "exhaustion",
                                     "Configure settings when flows reach limits" )
forwardingKw = CliMatcher.KeywordMatcher( "forwarding",
                                          "Configure forwarding settings" )

matchPolicyNameMatcher = CliMatcher.DynamicNameMatcher(
   lambda mode: gv.config.flowMatchPolicy,
   "Specify flow traffic policy name" )

def trafficClassRangeFn( mode, context ):
   return ( 0, gv.qosHwStatus.numTcSupported - 1 )

class ClbLearningConfigMode( CliMode.Clb.ClbFlowLearningMode,
                             BasicCli.ConfigModeBase ):
   def __init__( self, parent, session ):
      CliMode.Clb.ClbFlowLearningMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class ClbPortGroupConfigMode( CliMode.Clb.ClbPortGroupMode,
                              BasicCli.ConfigModeBase ):
   def __init__( self, parent, session, groupType, groupName ):
      CliMode.Clb.ClbPortGroupMode.__init__( self,
                                             groupType,
                                             groupName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def instanceRuleKey( self ):
      # share parse tree for each group type
      return self.groupType

   def intfGroup( self ):
      if self.groupType == "intfGroupTypeHost":
         coll = gv.config.hostIntfGroup
      elif self.groupType == "intfGroupTypeUplink":
         coll = gv.config.uplinkIntfGroup
      else:
         assert False

      return coll.newMember( self.groupName )

class HostPortGroupModelet( CliParser.Modelet ):
   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.groupType == "intfGroupTypeHost"

ClbPortGroupConfigMode.addModelet( HostPortGroupModelet )

def enterClbMode( mode, args ):
   childMode = mode.childMode( ClbConfigMode )
   mode.session.gotoChildMode( childMode )

def enterClbLearningMode( mode, args ):
   childMode = mode.childMode( ClbLearningConfigMode )
   mode.session.gotoChildMode( childMode )
   gv.config.flowSource = 'flowSourceLearning'

def clearClbLearningMode( mode, args ):
   gv.config.flowSource = 'flowSourceInvalid'
   gv.config.flowAgingTime = gv.config.flowAgingTimeDefault

def clearForwardingMode( mode, args ):
   gv.config.forwardingMode = 'disabled'

def clearLoadBalanceMethod( mode, args ):
   gv.config.loadBalanceMethod = "methodNone"

def clearFlowMatchCriteria( mode, args ):
   gv.config.flowMatchCriteria = gv.config.flowMatchCriteriaDefault

def clearFlowMonitor( mode, args ):
   gv.config.flowMonitor = False

def clearClbMode( mode, args ):
   clearForwardingMode( mode, args )
   clearLoadBalanceMethod( mode, args )
   clearFlowMonitor( mode, args )
   clearClbLearningMode( mode, args )
   clearFlowMatchCriteria( mode, args )
   gv.config.hostIntfGroup.clear()
   gv.config.uplinkIntfGroup.clear()
   gv.config.flowMatchPolicy.clear()
   gv.config.flowExhaustionMatchPolicy = ''
   gv.config.destinationGroupingType = 'destGroupTypeDefault'
   gv.config.destinationGroupingPrefixLen = 0

def enterClbPortGroupMode( mode, args ):
   groupName = args[ 'NAME' ]
   groupColl = getPortGroupCollectionFromArgs( mode, args )
   intfGroup = groupColl.newMember( groupName )
   childMode = mode.childMode( ClbPortGroupConfigMode,
                               groupType=intfGroup.groupType,
                               groupName=groupName )
   mode.session.gotoChildMode( childMode )

def clearClbPortGroupMode( mode, args ):
   groupName = args[ 'NAME' ]
   groupColl = getPortGroupCollectionFromArgs( mode, args )
   del groupColl[ groupName ]

class EnterClbLearningMode( CliCommand.CliCommandClass ):
   syntax = "flow source learning"
   noOrDefaultSyntax = "flow source ..."
   data = {
      "flow": flowKw,
      "source": "Configure flow sources",
      "learning": "Configure flow discovery by learning",
   }
   handler = enterClbLearningMode
   noOrDefaultHandler = clearClbLearningMode

ClbConfigMode.addCommandClass( EnterClbLearningMode )

def getPortGroupCollectionFromArgs( mode, args ):
   if "host" in args:
      return gv.config.hostIntfGroup
   else:
      return gv.config.uplinkIntfGroup

def getPortGroupNames( mode, context ):
   return getPortGroupCollectionFromArgs( mode, context.sharedResult )

class EnterClbPortGroupMode( CliCommand.CliCommandClass ):
   syntax = "port group host | uplink NAME"
   noOrDefaultSyntax = syntax
   data = {
      "port": portKw,
      "group": groupKw,
      "host": CliCommand.Node(
         CliMatcher.KeywordMatcher( "host",
                                    "Configure host ports" ),
         storeSharedResult=True ),
      "uplink": CliCommand.Node(
         CliMatcher.KeywordMatcher( "uplink",
                                    "Configure uplink ports" ),
         storeSharedResult=True,
         hidden=True ),
      "NAME": CliMatcher.DynamicNameMatcher( getPortGroupNames,
                                             'Specify port group name',
                                             passContext=True )
   }
   handler = enterClbPortGroupMode
   noOrDefaultHandler = clearClbPortGroupMode

ClbConfigMode.addCommandClass( EnterClbPortGroupMode )

def forwardingBridgedVxlanGuard( mode, token ):
   return ( None if gv.hwCapabilities.bridgedVxlanModeSupported
                 else CliGuards.guardNotThisPlatform )

def forwardingRoutedGuard( mode, token ):
   return ( None if gv.hwCapabilities.routedModeSupported
                 else CliGuards.guardNotThisPlatform )

def flowMatchEncapVxlanGuard( mode, token ):
   return ( None if gv.hwCapabilities.matchEncapVxlanSupported
               else CliGuards.guardNotThisPlatform )

def flowMatchEncapNoneIpv6Guard( mode, token ):
   return ( None if gv.hwCapabilities.matchEncapNoneIpv6Supported
            else CliGuards.guardNotThisPlatform )

class ForwardingModeCommand( CliCommand.CliCommandClass ):
   syntax = "forwarding type ( bridged encapsulation vxlan ) | routed [ ipv4 ]"
   noOrDefaultSyntax = "forwarding type ..."
   data = {
      "forwarding": forwardingKw,
      "type": "Specify forwarding type",
      "bridged": CliCommand.guardedKeyword(
            "bridged", "Configure bridged forwarding", forwardingBridgedVxlanGuard ),
      "encapsulation": "Configure encapsulation format of bridged packets",
      "vxlan": "Configure VXLAN encapsulation",
      "routed": CliCommand.guardedKeyword(
            "routed", "Configure routed forwarding", forwardingRoutedGuard ),
      # this keyword is no longer necessary as ipv4/ipv6 is configured through
      # the "flow match" command.
      "ipv4": CliCommand.hiddenKeyword( "ipv4" )
   }

   @staticmethod
   def handler( mode, args ):
      if 'routed' in args:
         mode = 'routed'
      else:
         mode = 'bridgedVxlan'
      gv.config.forwardingMode = mode
      # ipv4 isn't needed anymore, but remember this to keep compatibility
      gv.config.forwardingModeCliTokenIpv4 = 'ipv4' in args

   noOrDefaultHandler = clearForwardingMode

ClbConfigMode.addCommandClass( ForwardingModeCommand )

class LoadBalanceMethodCommand( CliCommand.CliCommandClass ):
   syntax = "load-balance method ( flow round-robin )"
   noOrDefaultSyntax = "load-balance method ..."
   data = {
      "load-balance": CliToken.LoadBalance.loadBalanceMatcherForConfig,
      "method": "Configure load-balance method",
      "flow": "Configure flow-based load-balance",
      "round-robin": "Configure flow round-robin load-balance",
      }

   @staticmethod
   def handler( mode, args ):
      assert "flow" in args
      assert 'round-robin' in args
      method = "methodFlowRoundRobin"
      gv.config.loadBalanceMethod = method

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      clearLoadBalanceMethod( mode, args )

ClbConfigMode.addCommandClass( LoadBalanceMethodCommand )

class FlowMatchEncapCommand( CliCommand.CliCommandClass ):
   syntax = "flow match encapsulation ( vxlan ipv4 ) | ( none ipv4 | ipv6 )"
   noOrDefaultSyntax = "flow match encapsulation ..."
   data = {
      "flow": flowKw,
      "match": "Configure flow match",
      "encapsulation": "Configure encapsulation format of packets",
      "vxlan": CliCommand.guardedKeyword(
            "vxlan", "Encapsulated vxlan packet type", flowMatchEncapVxlanGuard ),

      "none": "Non-encapsulated packet type",
      "ipv4": "Match IPv4 packets",
      "ipv6": CliCommand.guardedKeyword( "ipv6",
                                         "Match IPv6 packets",
                                         flowMatchEncapNoneIpv6Guard )
   }

   @staticmethod
   def handler( mode, args ):
      if "vxlan" in args:
         encapType = "encapTypeVxlan"
      else:
         assert "none" in args
         encapType = "encapTypeNone"

      if "ipv6" in args:
         ipProtocol = "ipv6"
      else:
         assert "ipv4" in args
         ipProtocol = "ipv4"
      gv.config.flowMatchCriteria = Tac.Value( "Clb::FlowMatchCriteria",
                                               encapType, ipProtocol )

   noOrDefaultHandler = clearFlowMatchCriteria

ClbConfigMode.addCommandClass( FlowMatchEncapCommand )

prefixLenMatcher = CliMatcher.IntegerMatcher(
   0, 129, helpdesc='Network address prefix length' )

class DestinationGroupingComand( CliCommand.CliCommandClass ):
   syntax = '''destination grouping ( vtep | ( bgp field-set ) |
               ( prefix length PREFIXLEN ) )'''
   noOrDefaultSyntax = 'destination grouping ...'
   data = {
      'destination': 'Configure destination settings',
      'grouping': 'Configure destination grouping settings',
      'vtep': 'Perform destination grouping using VXLAN tunnel endpoint',
      'bgp': 'Perform destination grouping using BGP',
      'field-set': 'Perform destination grouping using BGP field-set',
      'prefix': 'Perform destination grouping using address prefix',
      'length': 'Specify prefix length for destination grouping',
      'PREFIXLEN': prefixLenMatcher
   }

   @staticmethod
   def handler( mode, args ):
      if 'bgp' in args:
         gv.config.destinationGroupingType = 'destGroupTypeBgpFieldSet'
         gv.config.destinationGroupingPrefixLen = 0
      elif 'vtep' in args:
         gv.config.destinationGroupingType = 'destGroupTypeVtep'
         gv.config.destinationGroupingPrefixLen = 0
      else:
         assert 'prefix' in args
         gv.config.destinationGroupingType = 'destGroupTypePrefix'
         gv.config.destinationGroupingPrefixLen = args[ 'PREFIXLEN' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      gv.config.destinationGroupingType = 'destGroupTypeDefault'
      gv.config.destinationGroupingPrefixLen = 0

ClbConfigMode.addCommandClass( DestinationGroupingComand )

class FlowAgingCommand( CliCommand.CliCommandClass ):
   syntax = "aging timeout AGING seconds"
   noOrDefaultSyntax = "aging timeout ..."
   data = {
      "aging": "Configure flow aging",
      "timeout": "Configure flow aging timeout",
      "AGING": CliMatcher.IntegerMatcher( 30, 0x7FFFFFFF,
                                          helpdesc="Number of seconds" ),
      "seconds": "Specify time units",
   }

   @staticmethod
   def handler( mode, args ):
      gv.config.flowAgingTime = args[ 'AGING' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      gv.config.flowAgingTime = gv.config.flowAgingTimeDefault

ClbLearningConfigMode.addCommandClass( FlowAgingCommand )

class FlowLearningLimitCommand( CliCommand.CliCommandClass ):
   syntax = "limit LIMIT"
   noOrDefaultSyntax = "limit ..."
   data = {
      "limit": "Configure global flow learning limit",
      "LIMIT": CliMatcher.IntegerMatcher( 1, 0xffffffff,
                                          helpdesc="Number of learned flows" ),
   }

   @staticmethod
   def handler( mode, args ):
      gv.config.globalFlowLearnLimit = args[ 'LIMIT' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      gv.config.globalFlowLearnLimit = gv.config.globalFlowLearnLimitDefault

ClbLearningConfigMode.addCommandClass( FlowLearningLimitCommand )

class FlowMonitorCommand( CliCommand.CliCommandClass ):
   syntax = "flow monitor"
   noOrDefaultSyntax = syntax
   data = {
      "flow": flowKw,
      "monitor": "Monitor the flows without affecting forwarding",
   }

   @staticmethod
   def handler( mode, args ):
      gv.config.flowMonitor = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      clearFlowMonitor( mode, args )

ClbConfigMode.addCommandClass( FlowMonitorCommand )

class PortGroupFlowLimitCommand( CliCommand.CliCommandClass ):
   syntax = "flow limit LIMIT"
   noOrDefaultSyntax = "flow limit ..."
   data = {
      "flow": flowKw,
      "limit": "Configure flow limits",
      "LIMIT": CliMatcher.IntegerMatcher( 1, 0xffffffff,
                          helpdesc='Maximum number of flows per port group' ),
   }

   @staticmethod
   def handler( mode, args ):
      intfGroup = mode.intfGroup()
      intfGroup.flowLimit = args[ 'LIMIT' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfGroup = mode.intfGroup()
      intfGroup.flowLimit = intfGroup.flowLimitDefault

HostPortGroupModelet.addCommandClass( PortGroupFlowLimitCommand )

class PortGroupFlowWarningCommand( CliCommand.CliCommandClass ):
   syntax = "flow warning FLOWCOUNT"
   noOrDefaultSyntax = "flow warning ..."
   data = {
      "flow": flowKw,
      "warning": "Configure flow warning threshold",
      "FLOWCOUNT": CliMatcher.IntegerMatcher( 0, 0xffffffff,
                     helpdesc='Warning threshold of flows per port group' )
   }

   @staticmethod
   def handler( mode, args ):
      intfGroup = mode.intfGroup()
      intfGroup.flowWarningThreshold = args[ 'FLOWCOUNT' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfGroup = mode.intfGroup()
      intfGroup.flowWarningThreshold = intfGroup.flowWarningThresholdDefault

HostPortGroupModelet.addCommandClass( PortGroupFlowWarningCommand )

class PortGroupBalanceFactorCommand( CliCommand.CliCommandClass ):
   syntax = "balance factor BALANCEFACTOR"
   noOrDefaultSyntax = "balance factor ..."
   data = {
      "balance": "Configure port group balancing",
      "factor": " Configure port group balancing factor",
      "BALANCEFACTOR": CliMatcher.IntegerMatcher( 0, 0xFFFFFFFF,
                                           helpdesc="Balance factor value" )
   }

   @staticmethod
   def handler( mode, args ):
      intfGroup = mode.intfGroup()
      intfGroup.flowWriteLengthThreshold = args[ 'BALANCEFACTOR' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfGroup = mode.intfGroup()
      intfGroup.flowWriteLengthThreshold = intfGroup.flowWriteLengthThresholdDefault

HostPortGroupModelet.addCommandClass( PortGroupBalanceFactorCommand )

actionKw = CliMatcher.KeywordMatcher( 'action', 'Configure forwording action' )

class PortGroupFlowExhaustionActionCommand( CliCommand.CliCommandClass ):
   syntax = "flow exhaustion action { ( dscp DSCP ) | ( traffic-class TC ) }"
   noOrDefaultSyntax = "flow exhaustion action ..."
   data = {
      "flow": flowKw,
      "exhaustion": exhaustionKw,
      "action": "Configure forwording action",
      "dscp": CliCommand.Node(
         CliMatcher.KeywordMatcher( "dscp", "Set packet DSCP" ),
         maxMatches=1 ),
      "DSCP": CliMatcher.IntegerMatcher( 0, 63, helpdesc="DSCP value" ),
      "traffic-class": CliCommand.Node(
         CliMatcher.KeywordMatcher( "traffic-class", "Set packet traffic-class" ),
         maxMatches=1 ),
      'TC': CliMatcher.DynamicIntegerMatcher(
         trafficClassRangeFn, helpdesc='Traffic class value' )
      }

   @staticmethod
   def handler( mode, args ):
      action = Tac.Value( "Clb::TrafficPolicyAction" )
      dscp = args.get( 'DSCP' )
      if dscp:
         action.dscp = dscp[ 0 ]
      tc = args.get( 'TC' )
      if tc:
         action.trafficClass = tc[ 0 ]

      intfGroup = mode.intfGroup()
      intfGroup.flowExhaustionAction = action

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfGroup = mode.intfGroup()
      intfGroup.flowExhaustionAction = Tac.Value( "Clb::TrafficPolicyAction" )

HostPortGroupModelet.addCommandClass( PortGroupFlowExhaustionActionCommand )

class FlowExhaustionMatchCmd( CliCommand.CliCommandClass ):
   syntax = "flow exhaustion match NAME"
   noOrDefaultSyntax = "flow exhaustion match ..."
   data = {
      "flow": flowKw,
      "exhaustion": exhaustionKw,
      "match": "Specify match policy",
      "NAME": matchPolicyNameMatcher
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      gv.config.flowExhaustionMatchPolicy = args[ 'NAME' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      gv.config.flowExhaustionMatchPolicy = ''

ClbConfigMode.addCommandClass( FlowExhaustionMatchCmd )

class PortGroupIntfCommand( CliCommand.CliCommandClass ):
   syntax = "member | interface INTFS"
   noOrDefaultSyntax = "member | interface ..."
   data = {
      "member": "Configure interfaces",
      "interface": CliCommand.hiddenKeyword( "interface" ),
      "INTFS": IntfRange.IntfRangeMatcher(
         explicitIntfTypes=( EthPhyAutoIntfType, SwitchAutoIntfType, ),
         allowedSubIntfTypes=set() )
   }

   @staticmethod
   def handler( mode, args ):
      # intfNames() returns a generator, so convert to a list
      # as we iterate more than once.
      intfNames = set( args[ 'INTFS' ] )
      intfGroup = mode.intfGroup()
      intfGroup.useInterfaceToken = "interface" in args
      # avoid clearing the entire set
      for i in intfGroup.member:
         if i not in intfNames:
            del intfGroup.member[ i ]
      for i in intfNames:
         intfGroup.member[ i ] = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intfGroup = mode.intfGroup()
      intfGroup.member.clear()

ClbPortGroupConfigMode.addCommandClass( PortGroupIntfCommand )

class EnterMatchPolicyMode( CliCommand.CliCommandClass ):
   syntax = "flow match policy NAME"
   noOrDefaultSyntax = syntax
   data = {
      "flow": flowKw,
      "match": "Configure flow match",
      "policy": "Configure flow match policies",
      "NAME": matchPolicyNameMatcher
   }
   hidden = True

   handler = "ClbTrafficPolicyHandlers.enterMatchPolicyMode"

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      del gv.config.flowMatchPolicy[ args[ 'NAME' ] ]

ClbConfigMode.addCommandClass( EnterMatchPolicyMode )

def Plugin( entityManager ):
   gv.config = ConfigMount.mount( entityManager, "clb/config",
                                  "Clb::Config", "w" )
   gv.status = LazyMount.mount( entityManager, "clb/status",
                                "Clb::Status", "r" )
   gv.qosHwStatus = LazyMount.mount( entityManager, "qos/hardware/status/global",
                                     "Qos::HwStatus", "r" )
   gv.hwCapabilities = LazyMount.mount( entityManager,
                                        "clb/hardware/capabilities",
                                        "Clb::Hardware::Capabilities", "r" )
