#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Model, Int, Bool, List, Str, Enum, Float, Submodel
from ArnetModel import IpGenericAddress, MacAddress
from IntfModels import Interface
from IpLibConsts import DEFAULT_VRF
from TableOutput import createTable, Format
import functools
import re

# pylint: disable=consider-using-f-string
# pylint: disable-msg=R1735

def protoLineStr( proto, srcPort, dstPort ):
   return '  Protocol: %d, UDP Ports: %d -> %d' % ( proto, srcPort, dstPort )

def qpLineStr( qp ):
   return '  RoCE Queue-Pair: %d' % qp

def vlanLineStr( vlan, srcIntf, dstMac ):
   return ( '  Vlan: %d RX Interface: %s Destination MAC: %s' %
            ( vlan, srcIntf, dstMac ) )

def cmpFlow( a, b ):
   if a.vrfName != b.vrfName:
      if a.vrfName == DEFAULT_VRF:
         return -1
      elif b.vrfName == DEFAULT_VRF:
         return 1
      elif a.vrfName < b.vrfName:
         return -1
      elif a.vrfName > b.vrfName:
         return 1

   if a.dstAddr < b.dstAddr:
      return -1
   elif a.dstAddr > b.dstAddr:
      return 1

   if a.srcAddr < b.srcAddr:
      return -1
   elif a.srcAddr > b.srcAddr:
      return 1

   if a.queuePair < b.queuePair:
      return -1
   elif a.queuePair > b.queuePair:
      return 1

   return 0

def sortedFlowList( flowList ):
   return sorted( flowList, key=functools.cmp_to_key( cmpFlow ) )

class Flow( Model ):
   vrfName = Str( help='VRF name' )
   srcAddr = IpGenericAddress( help='source address' )
   dstAddr = IpGenericAddress( help='destination address' )
   srcPort = Int( help='source UDP port' )
   dstPort = Int( help='destination UDP port' )
   protocol = Int( help='IP L4 protocol number' )
   queuePair = Int( help='RoCE queue pair' )
   srcIntf = Interface( help='source interface' )
   vlanId = Int( help='VLAN id' )
   dstMac = MacAddress( help='destination MAC address' )

   destType = Enum( values=( 'unknown', 'routed', 'bridgedVxlan' ),
                    help="flow destination type" )
   dstIntf = Interface( help='destination interface', optional=True )
   vtep = IpGenericAddress( help='VTEP address', optional=True )
   ipNextHop = IpGenericAddress( help='IP Next Hop', optional=True )

   directFlowName = Str( help='name of flow in DirectFlow config',
                         optional=True )
   inHardware = Bool( help='true if flow is programmed into hardware' )
   # Need to figure this one out.  General CAPI guidelines are to use
   # enums, not strings, but fundamentally this is encapsulating
   # implementation-specific details that we do not want to encode in
   # a Clb type.  For systems using DirectFlow this is the
   # OpenFlow::FlowStatus enum, but for TrafficPolicy it will be
   # something else (not sure exactly what that is, but it is unlikely
   # to be exactly the same).
   #
   # Review this before we make the model public
   hardwareStatus = Str( help='Description of hardware programming state',
         optional=True )
   programStatus = Enum( values=(
      'Flow Not Programmed',
      'Flow Limit Exceeded',
      'Flow No Group Configured For Source Port',
      'Flow Route Lookup Failed',
      'Flow Programming Communicator Group Id Success',
      'Flow Programming Success' ),
      help='Description of software programming state', optional=True )
   commGroupId = Str( help='flow Group Id' )
   fieldSetId = Int( help='fieldSetId', optional=True )
   routeGroupVtep = IpGenericAddress( help='remote VTEP address', optional=True )

   packetCount = Int( help='Count of packets hitting this flow', optional=True )
   byteCount = Int( help='Count of packet bytes hitting this flow', optional=True )

   def str( self ):
      return '%s: %s -> %s, %s' % ( self.vrfName, self.srcAddr, self.dstAddr,
                                    self.queuePair )

class ShowFlowModel( Model ):
   __public__ = False
   flowList = List( valueType=Flow, help='List of flows' )
   totalFlows = Int( help='Total numbers of flows' )
   _warnAboutProtocolModel = Bool( help='warn user about multi-agent mode',
                                  optional=True )
   # FIXFIXFIX figure out if we need to put this filter info in the
   # model or if it can just be local in the show command
   # implementation
   _detail = Bool( help="Display detailed information", optional=True )
   _countersOnly = Bool( help="Display statistics only", optional=True )
   _filterSrcAddr = IpGenericAddress( help='filter on source IP address',
                                      optional=True )
   _filterDstAddr = IpGenericAddress( help='filter on destination IP address',
                                      optional=True )
   _filterQueuePair = Int( help='filter on RoCE queue pair', optional=True )
   _filterDstMac = MacAddress( help='filter on destination MAC address',
                               optional=True )
   _filterSrcIntfList = List( valueType=Interface, help='filter on source interface',
                              optional=True )
   _filterDstIntfList = List( valueType=Interface,
                              help='filter on destination interface', optional=True )
   _filterVlanId = Int( help='filter on VLAN id', optional=True )
   _filterVrf = Str( help='filter on VRF name', optional=True )

   def detailIs( self, detail ):
      self._detail = detail

   def countersOnlyIs( self, countersOnly ):
      self._countersOnly = countersOnly

   def filterVrfIs( self, vrf ):
      self._filterVrf = vrf

   def filterSrcAddrIs( self, sa ):
      self._filterSrcAddr = sa

   def filterDstAddrIs( self, da ):
      self._filterDstAddr = da

   def filterQueuePairIs( self, qp ):
      self._filterQueuePair = qp

   def filterDstMacIs( self, mac ):
      self._filterDstMac = mac

   def addFilterSrcIntf( self, intf ):
      self._filterSrcIntfList.append( intf )

   def addFilterDstIntf( self, intf ):
      self._filterDstIntfList.append( intf )

   def filterVlanIdIs( self, vlanId ):
      self._filterVlanId = vlanId

   def warnAboutProtocolModelIs( self, b ):
      self._warnAboutProtocolModel = b

   def filterApplied( self ):
      return ( self._filterSrcAddr is not None or
               self._filterDstAddr is not None or
               self._filterQueuePair is not None or
               self._filterDstMac is not None or
               self._filterSrcIntfList or
               self._filterDstIntfList or
               self._filterVlanId is not None or
               self._filterVrf is not None )

   def maybeAddFlow( self, cliFlow ):
      if not self.filterApplied():
         self.flowList.append( cliFlow )
         return

      matches = True

      if( self._filterSrcAddr is not None and
          cliFlow.srcAddr != self._filterSrcAddr ):
         matches = False
      elif( self._filterDstAddr is not None and
            cliFlow.dstAddr != self._filterDstAddr ):
         matches = False
      elif( self._filterQueuePair is not None and
            cliFlow.queuePair != self._filterQueuePair ):
         matches = False
      elif( self._filterDstMac is not None and
            cliFlow.dstMac != self._filterDstMac ):
         matches = False
      elif( self._filterSrcIntfList and
            cliFlow.srcIntf.stringValue not in self._filterSrcIntfList ):
         matches = False
      elif( self._filterDstIntfList and
            ( not cliFlow.dstIntf or
              cliFlow.dstIntf.stringValue not in self._filterDstIntfList ) ):
         matches = False
      elif( self._filterVlanId is not None and
            cliFlow.vlanId != self._filterVlanId ):
         matches = False
      elif self._filterVrf is not None and cliFlow.vrfName != self._filterVrf:
         matches = False

      if matches:
         self.flowList.append( cliFlow )

   def renderDestStr( self, cliFlow ):
      if cliFlow.destType == "bridgedVxlan":
         vtepStr = str( cliFlow.vtep ) if cliFlow.vtep else 'VTEP UNKNOWN'
         output = 'VXLAN: %s %s' % ( cliFlow.dstIntf.shortName, vtepStr )
      elif cliFlow.destType == 'routed':
         output = 'IP NH: %s %s' % ( cliFlow.dstIntf.shortName, cliFlow.ipNextHop )
      else:
         assert cliFlow.destType == 'unknown'
         output = 'destination unknown'
      return output

   def renderTotal( self, displayedFlows ):
      print( 'Total flows: %s, displayed: %d' % ( self.totalFlows, displayedFlows ) )

   def renderNormal( self, outList ):
      leftFmt = Format( justify='left' )
      leftFmt.noPadLeftIs( True )
      rightFmt = Format( justify='right' )

      table = createTable( ( 'VRF', 'SA', 'DA', 'Queue Pair', 'Rx Intf',
                             'Flow Assignment' ) )
      table.formatColumns( leftFmt, leftFmt, leftFmt, rightFmt, leftFmt, leftFmt )

      for cliFlow in outList:
         destStr = self.renderDestStr( cliFlow )
         if not cliFlow.inHardware:
            destStr = '*' + destStr
         table.newRow( cliFlow.vrfName, cliFlow.srcAddr, cliFlow.dstAddr,
                       cliFlow.queuePair, cliFlow.srcIntf.shortName, destStr )

      print( table.output() )
      self.renderTotal( len( outList ) )

   def renderCounters( self, outList ):
      leftFmt = Format( justify='left' )
      leftFmt.noPadLeftIs( True )
      rightFmt = Format( justify='right' )

      table = createTable( ( 'VRF', 'SA', 'DA', 'Queue Pair', 'Packets', 'Bytes' ) )
      table.formatColumns( leftFmt, leftFmt, leftFmt, rightFmt, rightFmt, rightFmt )

      for cliFlow in outList:
         packetCount = cliFlow.packetCount or 0
         byteCount = cliFlow.byteCount or 0
         table.newRow( cliFlow.vrfName, cliFlow.srcAddr, cliFlow.dstAddr,
                       cliFlow.queuePair, packetCount, byteCount )

      print( table.output() )
      self.renderTotal( len( outList ) )

   def renderDetail( self, outList ):
      for cliFlow in outList:
         print( 'VRF: %s %s -> %s' % ( cliFlow.vrfName, cliFlow.srcAddr,
                                        cliFlow.dstAddr ) )
         print( protoLineStr( cliFlow.protocol, cliFlow.srcPort, cliFlow.dstPort ) )
         print( qpLineStr( cliFlow.queuePair ) )
         print( vlanLineStr( cliFlow.vlanId, cliFlow.srcIntf.shortName,
                             cliFlow.dstMac ) )

         print( '  Destination: %s' % self.renderDestStr( cliFlow ) )

         if cliFlow.destType == "bridgedVxlan":
            vtepStr = str( cliFlow.routeGroupVtep ) \
                  if cliFlow.routeGroupVtep else 'VTEP UNKNOWN'
            print( '  Communicator Group: %s, Route Group VTEP: %s' %
               ( cliFlow.commGroupId, vtepStr ) )
         else:
            print( '  Communicator Group: %s, Route Group FieldSetId: %d' %
               ( cliFlow.commGroupId, cliFlow.fieldSetId ) )
         print( '  In hardware: %s' % cliFlow.inHardware )
         if cliFlow.hardwareStatus != '':
            print( '  Status: %s' % cliFlow.hardwareStatus )
         else:
            print( '  Status: %s' % cliFlow.programStatus )

         packetCount = cliFlow.packetCount if cliFlow.packetCount else 0
         byteCount = cliFlow.byteCount if cliFlow.byteCount else 0
         print( '  Packets: %d, Bytes: %d' % ( packetCount, byteCount ) )

         # lines that are only displayed sometimes (on some platforms,
         # for example) always go at the end
         if cliFlow.directFlowName:
            print( '  Direct Flow Name: %s' % cliFlow.directFlowName )

      self.renderTotal( len( outList ) )

   def render( self ):

      if self._warnAboutProtocolModel:
         print( 'Warning: CLB requires configuring the router in multi-agent mode' )

      outList = sortedFlowList( self.flowList )
      if self._countersOnly:
         self.renderCounters( outList )
      elif not self._detail:
         self.renderNormal( outList )
      else:
         self.renderDetail( outList )

class LearnedFlow( Model ):
   srcAddr = IpGenericAddress( help='source address' )
   dstAddr = IpGenericAddress( help='destination address' )
   srcPort = Int( help='source UDP port' )
   dstPort = Int( help='destination UDP port' )
   protocol = Int( help='IP L4 protocol number' )
   queuePair = Int( help='RoCE queue pair' )
   srcIntf = Interface( help='source interface' )
   vlanId = Int( help='VLAN id' )
   dstMac = MacAddress( help='destination MAC address' )
   firstSeen = Float( help='firstSeen seconds' )
   writeLength = Int( help='RoCE RETH write length' )
   remoteKey = Int( help='RoCE RETH remote key' )
   virtualAddress = Int( help='RoCE RETH virtual address' )
   writeLengthHistory = List( valueType=int,
                              help='List of historical write length values' )
   remoteKeyHistory = List( valueType=int,
                            help='List of historical remote key values' )
   virtualAddressHistory = List( valueType=int,
                                 help='List of historical virtual address values' )

class ShowLearnedFlowModel( Model ):
   __public__ = False

   flowList = List( valueType=LearnedFlow, help='List of flows' )
   totalFlows = Int( help='Total numbers of flows' )
   _detail = Bool( help="Display detailed information", optional=True )
   _filterSrcAddr = IpGenericAddress( help='filter on source IP address',
                                      optional=True )
   _filterDstAddr = IpGenericAddress( help='filter on destination IP address',
                                      optional=True )
   _filterQueuePair = Int( help='filter on RoCE queue pair', optional=True )
   _filterDstMac = MacAddress( help='filter on destination MAC address',
                               optional=True )
   _filterSrcIntfList = List( valueType=Interface, help='filter on source interface',
                              optional=True )
   _filterVlanId = Int( help='filter on VLAN id', optional=True )

   def detailIs( self, detail ):
      self._detail = detail

   def filterSrcAddrIs( self, sa ):
      self._filterSrcAddr = sa

   def filterDstAddrIs( self, da ):
      self._filterDstAddr = da

   def filterQueuePairIs( self, qp ):
      self._filterQueuePair = qp

   def filterDstMacIs( self, mac ):
      self._filterDstMac = mac

   def addFilterSrcIntf( self, intf ):
      self._filterSrcIntfList.append( intf )

   def filterVlanIdIs( self, vlanId ):
      self._filterVlanId = vlanId

   def sortedLearnedFlowList( self ):
      def cmpLearnedFlow( a, b ):
         if a.dstAddr < b.dstAddr:
            return -1
         elif a.dstAddr > b.dstAddr:
            return 1

         if a.srcAddr < b.srcAddr:
            return -1
         elif a.srcAddr > b.srcAddr:
            return 1

         if a.queuePair < b.queuePair:
            return -1
         elif a.queuePair > b.queuePair:
            return 1

         return 0

      sortedOutList = sorted( self.flowList,
                              key=functools.cmp_to_key( cmpLearnedFlow ) )
      return sortedOutList

   def filterApplied( self ):
      return ( self._filterSrcAddr is not None or
               self._filterDstAddr is not None or
               self._filterQueuePair is not None or
               self._filterDstMac is not None or
               self._filterSrcIntfList or
               self._filterVlanId is not None )

   def maybeAddFlow( self, cliFlow ):
      if not self.filterApplied():
         self.flowList.append( cliFlow )
         return

      matches = True

      if( self._filterSrcAddr is not None and
          cliFlow.srcAddr != self._filterSrcAddr ):
         matches = False
      elif( self._filterDstAddr is not None and
            cliFlow.dstAddr != self._filterDstAddr ):
         matches = False
      elif( self._filterQueuePair is not None and
            cliFlow.queuePair != self._filterQueuePair ):
         matches = False
      elif( self._filterDstMac is not None and
            cliFlow.dstMac != self._filterDstMac ):
         matches = False
      elif( self._filterSrcIntfList and
            cliFlow.srcIntf.stringValue not in self._filterSrcIntfList ):
         matches = False
      elif( self._filterVlanId is not None and
            cliFlow.vlanId != self._filterVlanId ):
         matches = False

      if matches:
         self.flowList.append( cliFlow )

   def renderTotal( self, displayedFlows ):
      print( 'Total flows: %s, displayed: %d' % ( self.totalFlows, displayedFlows ) )

   def renderNormal( self, outList ):
      fmt = Format( justify='left' )
      fmt.noPadLeftIs( True )

      table = createTable( ( 'SA', 'DA', 'Queue Pair', 'Rx Intf',
                             'FirstSeen' ) )

      table.formatColumns( fmt, fmt, fmt, fmt, fmt, fmt )

      for cliFlow in outList:
         table.newRow( cliFlow.srcAddr, cliFlow.dstAddr, cliFlow.queuePair,
                       cliFlow.srcIntf.shortName, cliFlow.firstSeen )

      print( table.output() )
      self.renderTotal( len( outList ) )

   def renderDetail( self, outList ):
      for cliFlow in outList:
         print( '%s -> %s' % ( cliFlow.srcAddr, cliFlow.dstAddr ) )
         print( protoLineStr( cliFlow.protocol, cliFlow.srcPort, cliFlow.dstPort ) )
         print( qpLineStr( cliFlow.queuePair ) )
         print( vlanLineStr( cliFlow.vlanId, cliFlow.srcIntf.shortName,
                             cliFlow.dstMac ) )

         print( '  FirstSeen: %s' % ( cliFlow.firstSeen ) )
         print( '  RETH: write length: %s, remote key: 0x%x, virtual address: 0x%x' %
                ( cliFlow.writeLength, cliFlow.remoteKey, cliFlow.virtualAddress ) )
         print( '  Historical write length: %s' %
                ', '.join( map( str, cliFlow.writeLengthHistory ) ) )
         print( '  Historical remote key: %s' %
                ', '.join( map( hex, cliFlow.remoteKeyHistory ) ) )
         print( '  Historical virtual address: %s' %
                ', '.join( map( hex, cliFlow.virtualAddressHistory ) ) )
      self.renderTotal( len( outList ) )

   def render( self ):

      outList = self.sortedLearnedFlowList()

      if not self._detail:
         self.renderNormal( outList )
      else:
         self.renderDetail( outList )

class PortGroupCounters( Model ):
   __public__ = False

   pgName = Str( help='Name of this port group' )
   learnedFlows = Int( help='Number of learned flows' )
   allocatedFlows = Int( help='Number of allocated flows' )
   unallocatedFlows = Int( help='Number of flows learned but not allocated' )
   programmedFlows = Int( help='Number of flows programmed in hardware' )
   unprogrammedFlows = Int( help='Number of flows allocated but not programmed' +
                            ' in hardware' )
   normalFallbackPacketCount = Int( help='Number of RoCE packets that do not match' +
                                    ' any flow while operating within TCAM limits' )
   normalFallbackByteCount = Int( help='Number of RoCE bytes that do not match' +
                                  ' any flow while operating within TCAM limits' )
   overflowFallbackPacketCount = Int( help='Number of RoCE packets that do not ' +
                                      'match any flow while exceeding TCAM limits' )
   overflowFallbackByteCount = Int( help='Number of RoCE bytes that do not ' +
                                    'match any flow while exceeding TCAM limits' )

   def initZero( self ):
      self.learnedFlows = 0
      self.allocatedFlows = 0
      self.unallocatedFlows = 0
      self.programmedFlows = 0
      self.unprogrammedFlows = 0
      self.normalFallbackPacketCount = 0
      self.normalFallbackByteCount = 0
      self.overflowFallbackPacketCount = 0
      self.overflowFallbackByteCount = 0

class ShowStatsModel( Model ):
   __public__ = False

   numRawLearnedFlows = Int( help='Number of raw learned flows' )
   numClbFlows = Int( help='Number of CLB flows' )
   numClbCommGroups = Int( help='Number of CLB communicator groups' )
   clbCommGroupMean = Float( help='Mean number of flows in CLB communicator groups' )
   clbCommGroupStDev = Float( help='Standard deviation in number of flows in ' +
                              'CLB communicator groups' )
   numClbRouteGroups = Int( help='Number of CLB route groups' )
   clbRouteGroupMean = Float( help='Mean number of flows in CLB route groups' )
   clbRouteGroupStDev = Float( help='Standard deviation in number of flows in ' +
                              'CLB route groups' )
   numClbDests = Int( help='Number of unique CLB destinations' )
   clbDestMean = Float( help='Mean number of flows per CLB destination' )
   clbDestStDev = Float( help='Standard deviation in number of flows per CLB ' +
                         'destination' )
   pgCounters = List( valueType=PortGroupCounters, help='Port group counters' )

   def addPgCounters( self, counterModel ):
      self.pgCounters.append( counterModel )

   def render( self ):
      print( 'Raw learned flows: %s' % self.numRawLearnedFlows )
      print( 'CLB flows: %s' % self.numClbFlows )
      print( 'Comm groups: %s, mean size %s, std dev: %s' %
             ( self.numClbCommGroups, self.clbCommGroupMean,
               self.clbCommGroupStDev ) )
      print( 'Route groups: %s, mean size %s, std dev: %s' %
             ( self.numClbRouteGroups, self.clbRouteGroupMean,
               self.clbRouteGroupStDev ) )
      print( 'Destinations: %s, mean size %s, std dev: %s' %
             ( self.numClbDests, self.clbDestMean, self.clbDestStDev ) )

      leftFmt = Format( justify='left' )
      leftFmt.noPadLeftIs( True )
      rightFmt = Format( justify='right' )
      table1 = createTable( ( 'Name', 'Learned', 'Allocated', 'Unallocated',
                              'Programmed', 'Unprogrammed' ) )
      table1.formatColumns( leftFmt, rightFmt, rightFmt, rightFmt, rightFmt,
                            rightFmt )

      table2 = createTable( ( 'Name', 'Packets (normal)', 'Bytes (normal)',
                              'Packets (overflow)', 'Bytes (overflow)' ) )
      table2.formatColumns( leftFmt, rightFmt, rightFmt, rightFmt, rightFmt )

      sortedPgCounters = sorted( self.pgCounters, key=lambda x: x.pgName )
      for pgc in sortedPgCounters:
         table1.newRow( pgc.pgName, pgc.learnedFlows, pgc.allocatedFlows,
                        pgc.unallocatedFlows, pgc.programmedFlows,
                        pgc.unprogrammedFlows )
         table2.newRow( pgc.pgName, pgc.normalFallbackPacketCount,
                        pgc.normalFallbackByteCount,
                        pgc.overflowFallbackPacketCount,
                        pgc.overflowFallbackByteCount )
      print( 'Flow counts by port group:' )
      print( table1.output() )
      print( 'Non-matched RoCE traffic counters by port group:' )
      print( table2.output() )

   def initZero( self ):
      self.numRawLearnedFlows = 0
      self.numClbFlows = 0
      self.numClbCommGroups = 0
      self.numClbRouteGroups = 0
      self.clbCommGroupMean = 0.0
      self.clbCommGroupStDev = 0.0
      self.clbRouteGroupMean = 0.0
      self.clbRouteGroupStDev = 0.0
      self.numClbDests = 0
      self.clbDestMean = 0.0
      self.clbDestStDev = 0.0

class Dest( Model ):
   __public__ = False
   destType = Enum( values=( 'routed', 'bridgedVxlan' ),
                   help="Flow destination type" )
   interface = Interface( help='Destination interface' )
   nextHop = IpGenericAddress( help='Next hop address', optional=True )

   def str( self ):
      # dunno why pylint can't find this
      # pylint: disable-msg=E1101
      intfStr = self.interface.stringValue
      return '%s %s %s' % ( self.destType, intfStr,
                            str( self.nextHop ) if self.nextHop else '-' )

class DestAndFlows( Model ):
   __public__ = False
   destination = Submodel( valueType=Dest, help='Destination' )
   flows = List( valueType=Flow, help='List of flows' )

def cmpDest( a, b ):
   if a.destType != b.destType:
      if a.destType < b.destType:
         return -1
      elif a.destType > b.destType:
         return 1

   if a.interface < b.interface:
      return -1
   elif a.interface > b.interface:
      return 1

   if a.nextHop < b.nextHop:
      return -1
   elif a.nextHop > b.nextHop:
      return 1

   return 0

def sortedDestList( destList ):
   return sorted( destList, key=functools.cmp_to_key( cmpDest ) )

class ShowClbDestModel( Model ):
   __public__ = False

   _detail = Bool( help="Display detailed information", optional=True )
   dests = List( valueType=DestAndFlows,
                 help='List of CLB flows for each destination' )

   def detailIs( self, detail ):
      self._detail = detail

   def render( self ):
      flowsByDest = { d.destination: d for d in self.dests }

      for d in sortedDestList( flowsByDest ):
         destAndFlows = flowsByDest[ d ]

         print( '%s: %d flows' % ( d.str(), len( destAndFlows.flows ) ) )
         if self._detail:
            for flow in sortedFlowList( destAndFlows.flows ):
               print( '  %s' % flow.str() )

class ComGroupAndFlows( Model ):
   __public__ = False
   groupId = Str( help='Group Id' )
   flows = List( valueType=Flow, help='List of flows per communicator group' )

class ShowClbComGroupModel( Model ):
   __public__ = False

   _summary = Bool( help="Display summary information", optional=True )
   comGroups = List( valueType=ComGroupAndFlows,
                 help='List of CLB flows for each communicator group' )

   def summaryIs( self, summary ):
      self._summary = summary

   def render( self ):
      flowsByGroup = { group.groupId: group for group in self.comGroups }

      for groupId in sorted( flowsByGroup ):
         comGroupAndFlows = flowsByGroup[ groupId ]
         print( "Group %s: %d flows" % ( groupId,
                len( comGroupAndFlows.flows ) ) )
         if not self._summary:
            for flow in sortedFlowList( comGroupAndFlows.flows ):
               print( '  %s' % flow.str() )

class RouteGroupId( Model ):
   __public__ = False
   groupId = Str( help='Communicator Group Id' )
   fieldSetId = Int( help='BGP field set Id' )
   vtepAddr = IpGenericAddress( help='VTEP address', optional=True )

   def str( self ):
      return 'Communicator GroupId %s fieldSetId %s remote VTEP %s' % \
            ( self.groupId, self.fieldSetId, self.vtepAddr )

class RouteGroupAndFlows( Model ):
   __public__ = False
   routeGroupId = Submodel( valueType=RouteGroupId, help='Route Group Id' )
   flows = List( valueType=Flow, help='List of flows' )

def cmpRouteGroupId( a, b ):
   if a.groupId != b.groupId:
      # commGroupIds will be in the form of '10A or 10B'
      mA = re.search( r'([0-9]+)([AB]?)', a.groupId )
      mB = re.search( r'([0-9]+)([AB]?)', b.groupId )

      if int( mA.group( 1 ) ) < int( mB.group( 1 ) ):
         return -1
      elif int( mA.group( 1 ) ) > int( mB.group( 1 ) ):
         return 1

      if mA.group( 2 ) < mB.group( 2 ):
         return -1
      elif mA.group( 2 ) > mB.group( 2 ):
         return 1

   if a.groupId < b.groupId:
      return -1
   elif a.groupId > b.groupId:
      return 1

   if a.fieldSetId < b.fieldSetId:
      return -1
   elif a.fieldSetId > b.fieldSetId:
      return 1

   if a.vtepAddr < b.vtepAddr:
      return -1
   elif a.vtepAddr > b.vtepAddr:
      return 1

   return 0

def sortedRouteGroupIdList( routeGroupIdList ):
   return sorted( routeGroupIdList, key=functools.cmp_to_key( cmpRouteGroupId ) )

class ShowClbRouteGroupModel( Model ):
   __public__ = False

   _summary = Bool( help="Display summary information", optional=True )
   routeGroups = List( valueType=RouteGroupAndFlows,
                 help='List of CLB flows for each route group' )

   def summaryIs( self, summary ):
      self._summary = summary

   def render( self ):
      flowsByGroup = { group.routeGroupId: group for group in self.routeGroups }
      for group in sortedRouteGroupIdList( flowsByGroup ):
         routeGroupAndFlows = flowsByGroup[ group ]

         print( '%s: %d flows' % ( group.str(), len( routeGroupAndFlows.flows ) ) )
         if not self._summary:
            for flow in sortedFlowList( routeGroupAndFlows.flows ):
               print( '  %s' % flow.str() )

class WarningModel( Model ):
   __public__ = False
   warningType = Enum( values=( 'missingPort', 'duplicatePort',
                                'uplinkInconsistent', 'missingVxlanInterface' ),
                       help='Type of warning' )
   interface = Interface( help='Interface with warning' )

class ShowClbWarningsModel( Model ):
   __public__ = False
   warningList = List( valueType=WarningModel,
                       help='List of CLB configuration warnings' )

   def render( self ):
      if not self.warningList:
         print( 'There are no CLB configuration warnings.' )
         return

      print( 'CLB configuration warnings:' )
      wList = sorted( self.warningList,
                      key=lambda x: '%s-%s' % ( x.interface, x.warningType ) )
      for w in wList:
         intfStr = w.interface.stringValue
         if w.warningType == 'duplicatePort':
            print( '%s is configured in more than one port group.' % intfStr )
         elif w.warningType == 'missingPort':
            print( '%s has learned flows but is not part of any host port group.' %
                   intfStr )
         elif w.warningType == 'uplinkInconsistent':
            print( '%s is part of a host port group but is also used as an uplink.' %
                   intfStr )
         elif w.warningType == 'missingVxlanInterface':
            print( 'Flow match encap type is VXLAN, ' +
                   'but there is no VXLAN interface configured.' )

class PortGroupStatus( Model ):
   __public__ = False
   pgName = Str( help='Name of this port group' )
   fallbackDscp = Int( help='DSCP value for fallback', optional=True )
   fallbackTrafficClass = Int( help='Traffic Class for fallback', optional=True )

class ShowClbStatusModel( Model ):
   __public__ = False
   clbStatus = Enum( values=( 'disabled', 'enabled', 'monitor' ),
                     help='CLB feature status' )
   pgStatus = List( valueType=PortGroupStatus, help='List of port group status' )

   def render( self ):

      print( 'CLB Status: %s' % self.clbStatus )

      if len( self.pgStatus ) == 0:
         return

      table = createTable( ( 'Port Group Name', 'Fallback DSCP',
                             'Fallback Traffic Class' ) )
      leftFmt = Format( justify='left' )
      rightFmt = Format( justify='right' )
      table.formatColumns( leftFmt, rightFmt, rightFmt )

      sortedPgStatus = sorted( self.pgStatus, key=lambda x: x.pgName )
      for pgs in sortedPgStatus:
         dscp = pgs.fallbackDscp if pgs.fallbackDscp is not None else '-'
         tc = ( pgs.fallbackTrafficClass if pgs.fallbackTrafficClass is not None
                else '-' )
         table.newRow( pgs.pgName, dscp, tc )

      print( table.output() )
