# Copyright (c) 2011, 2013, 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import errno
from CliDynamicSymbol import CliDynamicPlugin
from CliPlugin import CliSchedulerCli
import CliSchedulerLib
from CliSchedulerLib import (
   showTechJobName,
   showTechCliCommand,
   showTechIntervalDefault,
   showTechMaxLogFilesDefault,
   showTechMaxTotalSizeDefault,
)
import LazyMount
import Tac
import Url
import UrlPlugin.FlashUrl

CliSchedulerCliModels = CliDynamicPlugin( "CliSchedulerCliModels" )

config = CliSchedulerCli.config
status = None

setSchedErrorMessage = {
      CliSchedulerLib.DATEFMT_ERROR[ "INVALID" ] : "Invalid at argument",
      CliSchedulerLib.DATEFMT_ERROR[ "PAST" ] : "Cannot schedule a command in past",
}

def setSchedConfig( mode, jobName, atDateTime, interval, timeout, maxLogFiles,
                    maxTotalSize, loggingVerbose, logSavePath,
                    compressAlgo, command ):

   schedConfigType = Tac.Type( "System::CliScheduler::ScheduledCli" )

   dateTimeArr = CliSchedulerLib.createDateTimefromAt( atDateTime, interval )
   if dateTimeArr[ 0 ] < 0:
      mode.addError( setSchedErrorMessage[ dateTimeArr[ 0 ] ] )
      return

   if CliSchedulerLib.inPast( dateTimeArr[ 0 ] ):
      if interval == 0:
         mode.addError( setSchedErrorMessage[
               CliSchedulerLib.DATEFMT_ERROR[ "PAST" ] ] )
         return
      mode.addWarning( "Schedule a command starting in past" )

   startAt = dateTimeArr[ 0 ]
   if len( dateTimeArr ) == 2:
      startAt = dateTimeArr[ 1 ]

   jobName = jobName.lower()
   if jobName == "summary":
      mode.addError( f"Job name cannot match keyword {jobName}" )
      return

   if logSavePath != CliSchedulerLib.logPrefixDefault:
      if isinstance( logSavePath, UrlPlugin.FlashUrl.FlashUrl ) and \
            logSavePath.localFilename() is not None and \
            ( not logSavePath.exists() or logSavePath.isdir() ):

         logSavePath = logSavePath.localFilename()
         if not logSavePath.endswith("/"):
            logSavePath += "/"

      elif isinstance( logSavePath, Url.Url ):
         mode.addError( "Invalid scheduled job output location" )
         return

      else:
         logSavePath = CliSchedulerLib.logPrefixDefault

   timeout = timeout if timeout else CliSchedulerLib.timeoutDefault
   # interval should be more than the timeout, otherwise the job (if it has a
   # long running time) will run continuously
   if 0 < interval <= timeout:
      mode.addError( f"For job {jobName}, interval {int(interval)} should be "
                     f"greater than the timeout {int(timeout)}" )
      return

   if maxTotalSize:
      maxTotalSize = maxTotalSize.lower()
      if maxTotalSize.endswith( 'k' ):
         maxTotalSize = int( maxTotalSize[ : -1 ] ) * CliSchedulerLib.oneKb
      elif maxTotalSize.endswith( 'm' ):
         maxTotalSize = int( maxTotalSize[ : -1 ] ) * CliSchedulerLib.oneMeg
      elif maxTotalSize.endswith( 'g' ):
         maxTotalSize = int( maxTotalSize[ : -1 ] ) * CliSchedulerLib.oneGig
      elif maxTotalSize.endswith( 'b' ):
         maxTotalSize = int( maxTotalSize[ : -1 ] )
      else:
         maxTotalSize = int( maxTotalSize )

   schedConfig = schedConfigType( jobName,
                                  command,
                                  dateTimeArr[ 0 ], startAt,
                                  interval, maxLogFiles, maxTotalSize,
                                  loggingVerbose,
                                  timeout * 60,
                                  logDir=logSavePath,
                                  compressAlgo=compressAlgo )

   config.scheduledCli.addMember( schedConfig )

def delSchedConfig( mode, jobName ):
   jobName = jobName.lower()
   del config.scheduledCli[ jobName ]

def setSchedConfigDefault( mode, jobName ):
   jobName = jobName.lower()
   if jobName != showTechJobName:
      delSchedConfig( mode, jobName )
      return

   logSavePath = CliSchedulerLib.logPrefixDefault
   schedConfigType = Tac.Type( "System::CliScheduler::ScheduledCli" )
   schedConfig = schedConfigType( showTechJobName,
                                  showTechCliCommand,
                                  CliSchedulerLib.scheduleNow,
                                  CliSchedulerLib.scheduleNow,
                                  showTechIntervalDefault,
                                  showTechMaxLogFilesDefault,
                                  showTechMaxTotalSizeDefault,
                                  False,
                                  CliSchedulerLib.timeoutDefault * 60,
                                  logDir=logSavePath,
                                  compressAlgo=CliSchedulerLib.compressAlgoDefault )

   config.scheduledCli.addMember( schedConfig )

def handlerScheduleCommand( mode, args ):
   jobName = args[ 'NAME' ]
   timeVal = args.get( 'TIME' )
   if timeVal is not None:
      dateVal = args.get( 'DATE', [] )
      atDateTime = list( timeVal ) + list( dateVal )
   else:
      atDateTime = CliSchedulerLib.scheduleNow

   if CliSchedulerLib.scheduleOnceStr in args:
      interval = CliSchedulerLib.scheduleOnce
   else:
      interval = args.get( 'INT_MIN', 0 )

   timeout = args.get( 'TIMEOUT_MIN', 0 )
   maxLogFiles = args[ 'MAXLOGS' ]
   maxTotalSize = args.get( 'MAX_TOTAL_SIZE', 0 )
   loggingVerbose = 'verbose' in args
   logSavePath = args.get( 'LOGPATH' )
   compressAlgo = args.get( 'COMPRESSION_ALGO', CliSchedulerLib.compressAlgoDefault )
   command = args[ 'COMMAND' ]
   setSchedConfig( mode, jobName, atDateTime, interval, timeout, maxLogFiles,
                   maxTotalSize, loggingVerbose, logSavePath, compressAlgo, command )

def noHandlerScheduleCommand( mode, args ):
   delSchedConfig( mode, args[ 'NAME' ] )

def defaultHandlerScheduleCommand( mode, args ):
   setSchedConfigDefault( mode, args[ 'NAME' ] )

def handlerScheduleMaxJobsCommand( mode, args ):
   config.jobsInProgress = args[ 'JOBS' ]

def noOrDefaultHandlerScheduleMaxJobsCommand( mode, args ):
   config.jobsInProgress = config.jobsInProgressDefault

def handlerScheduleHostnameCommand( mode, args ):
   config.prependHostname = config.prependHostnameDefault

def noHandlerScheduleHostnameCommand( mode, args ):
   config.prependHostname = False

#pylint: disable=protected-access
def showSchedulePopulateModel( key, jobType, mode ):
   job = CliSchedulerCliModels.ScheduleJobModels()
   cliConfig = config.scheduledCli[ key ]
   lastTime = None
   if key in status.scheduledCliJobStatus:
      job._scheduledCliJobStatus = True
      cliStatus = status.scheduledCliJobStatus[ key ]
      job.jobInProgress = cliStatus.jobInProgress
      # jobType is a bool which differentiates between 'summary'|'NAME' cli commands
      # for this specific case
      if jobType:
         if cliStatus.jobInProgress or cliStatus.lastExecutionTime:
            lastTime = cliStatus.lastExecutionTime
            job.lastRunStatus = cliStatus.lastExecutionStatus
      else:
         if cliStatus.jobInProgress:
            job.jobInProgressStartTime = cliStatus.lastExecutionStartTime
         elif cliStatus.lastExecutionTime:
            lastTime = cliStatus.lastExecutionTime
            job.lastRunStatus = cliStatus.lastExecutionStatus
   else:
      job._scheduledCliJobStatus = False

   if cliConfig.logDir != CliSchedulerLib.logPrefixDefault:
      logLocation = Url.filenameToUrl( cliConfig.logDir )
   else:
      logLocation = cliConfig.logDir

   if not logLocation.endswith( "/" ):
      logLocation += "/"

   logLocation += ( f"{cliConfig.name}/" )
   context = Url.Context( *Url.urlArgsFromMode( mode ) )
   schedUrl = Url.parseUrl( logLocation, context )
   logfiles = []
   try:
      logfiles = [ str( schedUrl.child( f ) ) for f in sorted(
                   filter( CliSchedulerLib.extractTsFromFilename,
                           schedUrl.listdir() ),
                   reverse=True, key=CliSchedulerLib.extractTsFromFilename ) ]
   except OSError as e:
      if e.errno != errno.ENOENT:
         raise

   if not logfiles:
      job._logfiles = None
   else:
      job._logfiles = logfiles

   job.verbose = cliConfig.verbose
   job.nextRun = cliConfig.at
   job.lastRun = lastTime
   job.interval = cliConfig.interval
   job.timeout = cliConfig.timeout
   job.maxLogFiles = cliConfig.maxLogFiles
   job.maxTotalSize = cliConfig.maxTotalSize
   job.logfileDirectory = logLocation
   job.compressAlgo = cliConfig.compressAlgo
   job.cliCommand = cliConfig.cliCommand
   return job

def showSchedule( mode, args ):
   model = CliSchedulerCliModels.ShowScheduleJobModels()
   jobName = args.get( 'NAME', 'summary' ).lower()
   if not config.scheduledCli:
      return model
   else:
      if jobName == "summary":
         model._showScheduleAction = True
         for key in config.scheduledCli:
            model.jobs[ key ] = showSchedulePopulateModel( key, True, mode )
         model.maxJobs = config.jobsInProgress
         model.prependHostname = config.prependHostname
      else:
         model._showScheduleAction = False
         if jobName not in config.scheduledCli:
            mode.addError( 'Please enter job name of a scheduled CLI command' )
            return model
         model.jobs[ jobName ] = showSchedulePopulateModel( jobName, False, mode )
         model.maxJobs = config.jobsInProgress
         model.prependHostname = config.prependHostname
      return model

# --------------------------------------------------------------------------
def Plugin( entityManager ):
   global status
   status = LazyMount.mount( entityManager, "sys/clischeduler/status",
                             "System::CliScheduler::Status", "r" )
