# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import (
      Model,
      Str,
      Int,
      Float,
      List,
      Dict,
      Bool )
import CliSchedulerLib
import errno
from FileCliUtil import sizeFmt
import os
from TableOutput import (
      createTable,
      Format )
import time
import urllib.parse

class ScheduleJobModels( Model ):
   """
   Models for Schedule Summaries.
   """
   nextRun = Float( help="Next execution start time (in secs)." )
   cliCommand = Str( help="Scheduled Cli Command." )
   interval = Int( help="Cli command execution interval (in mins)." )
   logfileDirectory = Str( help="Logfile directory location." )
   maxLogFiles = Int( help="Maximum number of logfiles stored." )
   maxTotalSize = Int( help="Maximum total size of log files in bytes." )
   timeout = Float( help="Cli command execution timeout (in secs)." )
   verbose = Bool( help="Verbose logging state." )
   compressAlgo = Str( help="Compression algorithm." )
   jobInProgress = Bool( help="Job in progress.", optional=True )
   _logfiles = List( valueType=str, help="List of logfile paths.", optional=True )

   jobInProgressStartTime = Float( help="Cli commmand in progress execution start "
                                   "time (in secs)", optional=True )
   lastRunStatus = Int( help="Cli command last execution status.",
                        optional=True )
   lastRun = Float( help="Time the job last executed (in secs).",
                    optional=True )
   _scheduledCliJobStatus = Bool( help="Job name is scheduled.", optional=True )

   def renderSummary( self, name, table ):
      lastTime = time.strftime( "%H:%M", time.localtime(
                 self.lastRun ) ) if self.lastRun else "-"
      if self.lastRun and self.maxLogFiles > 0:
         logfileLoc = self.logfileDirectory
      else:
         logfileLoc = "-"
      interval = CliSchedulerLib.scheduleOnceStr
      if self.interval != CliSchedulerLib.scheduleOnce:
         interval = str( self.interval )
      status = "Waiting"
      if self.jobInProgress or self.lastRun:
         if self.jobInProgress:
            status = "Job under progress"
         elif self.lastRunStatus == 0:
            status = "Success"
         elif self.lastRunStatus == -1:
            status = "Timedout"
         else:
            status = "Fail"
      table.newRow( name, CliSchedulerLib.extractAtFromDateTime(
         self.nextRun ), lastTime, interval,
                    int( self.timeout / 60 ), self.maxLogFiles,
                    CliSchedulerLib.diskUseFmt( self.maxTotalSize ),
                    logfileLoc, status )
      return table

   def renderName( self, name ):
      cliStub = f'CLI command "{self.cliCommand}"'
      if self._scheduledCliJobStatus:
         if self.jobInProgress:
            print( "The CLI command execution is under progress for last "
                   f"{int( time.time() - self.jobInProgressStartTime )} seconds" )
         elif self.lastRun:
            if self.lastRunStatus == 0:
               print( "The last CLI command execution was successful" )
            elif self.lastRunStatus == -1:
               print( "The last CLI command execution timed out" )
            else:
               print( "The last CLI command failed with exit status",
                      int( self.lastRunStatus ) )
            if self.interval != CliSchedulerLib.scheduleOnce:
               atTime = time.localtime( self.lastRun +
                                        self.interval*60 )
               atStr = time.strftime( "%H:%M:%S %m/%d/%Y", atTime )
               print( f'{cliStub} is scheduled next at "{atStr}", '
                      f'interval is {self.interval} minutes' )
            else:
               print( f'{cliStub} was scheduled at '
                      f'"{CliSchedulerLib.extractAtFromDateTime( self.nextRun )}" '
                      'for single execution' )
         else:
            if int( self.nextRun ) == CliSchedulerLib.scheduleNow:
               print( f'{cliStub} is scheduled now, interval is {self.interval} '
                      'minutes' )
            else:
               if self.interval != CliSchedulerLib.scheduleOnce:
                  print( f'{cliStub} is scheduled at '
                         f'"{CliSchedulerLib.extractAtFromDateTime( self.nextRun )}"'
                         f', interval is {self.interval} minutes' )
               else:
                  print( f'{cliStub} is scheduled at '
                         f'"{CliSchedulerLib.extractAtFromDateTime( self.nextRun )}"'
                         ' for single execution' )

      else:
         print( f'CLI command "{name}" is currently not scheduled' )

      print( f"Timeout is {int( self.timeout / 60 )} minutes" )

      print( f"Maximum of {self.maxLogFiles} log files will be stored" )

      if self.maxTotalSize:
         print( "Maximum total size of log files is "
                f"{sizeFmt( self.maxTotalSize )}" )

      print( f"Verbose logging is { 'on' if self.verbose else 'off' }" )

      if not self._logfiles:
         print( f"No log files are stored in {self.logfileDirectory}" )
      else:
         print( f"{len( self._logfiles )} log files currently stored in "
                f"{self.logfileDirectory}\n" )
         table = createTable( ( "Start time", "Size", "Filename" ) )
         leftFormat = Format( justify='left' )
         leftFormat.noPadLeftIs( True )
         centerFormat = Format( justify='center' )
         centerFormat.noPadLeftIs( True )
         table.formatColumns( leftFormat, centerFormat, leftFormat )
         for u in self._logfiles:
            filename = os.path.basename( u )
            tm = time.localtime( CliSchedulerLib.extractTsFromFilename(
                                 filename ) )
            # If a logFile <fname>.log extension is being compressed using
            # gzip/bzip2/xz when show CLI command is executed u.size() may fail with
            # OSError exception due to it being renamed to <fname>.log.gz/bz2/xz
            try:
               url = urllib.parse.urlparse( u )
               filePath = '/mnt/' + url.scheme + url.path
               size = os.path.getsize( filePath )
               table.newRow( time.strftime( "%b %d %Y %H:%M", tm ),
                             sizeFmt( size ), filename )
            except OSError as e:
               if e.errno != errno.ENOENT:
                  raise
         print( table.output().rstrip() )

class ShowScheduleJobModels( Model ):
   jobs = Dict( keyType=str, valueType=ScheduleJobModels,
                help="Maps job names to their respective details." )
   maxJobs = Int( help="Maximum number of jobs in progress at the same time.",
                  optional=True )
   prependHostname = Bool( help="Prepend host name to logfile.", optional=True)
   _showScheduleAction = Bool(
                           help="Defines which show schedule command was called." )

   def render( self ):

      if not self.jobs:
         print( "No CLI commands are currently scheduled" )
      else:
         # renders summary action if _showScheduleAction == True, else NAME action
         if self._showScheduleAction:
            table = createTable( ( "Name", "At Time", "Last Time",
                                   "Interval (mins)",
                                   "Timeout (mins)", "Max Log Files",
                                   "Max Logs Size", "Logfile Location",
                                   "Status" ) )
            nameFormat = Format( justify='left', maxWidth=15, wrap=True )
            nameFormat.noPadLeftIs( True )
            nameFormat.noTrailingSpaceIs( True )
            leftFormat = Format( justify='left', maxWidth=8, wrap=True )
            leftFormat.noPadLeftIs( True )
            leftFormat.noTrailingSpaceIs( True )
            atTimeFormat = Format( justify='center',
                                   maxWidth=7,
                                   wrap=True )
            intervalFormat = Format( justify='center',
                                     maxWidth=8,
                                     wrap=True )
            timeoutFormat = Format( justify='center',
                                    maxWidth=7,
                                    wrap=True )
            lastTimeFormat = Format( justify='center', maxWidth=5, wrap=True )
            centerFormat = Format( justify='center' )
            centerFormat.noPadLeftIs( True )
            centerFormat.noTrailingSpaceIs( True )
            maxLogFileSizeFormat = Format( justify='center', maxWidth=5, wrap=True )
            maxTotalSizeFormat = Format( justify='center', maxWidth=4,
                                         wrap=True )
            loc = Format( justify='left', maxWidth=30, wrap=True )
            loc.noPadLeftIs( True )
            loc.noTrailingSpaceIs( True )
            table.formatColumns( nameFormat, atTimeFormat, lastTimeFormat,
                                 intervalFormat, timeoutFormat,
                                 maxLogFileSizeFormat,
                                 maxTotalSizeFormat, loc, leftFormat )
            # Jobs are sorted by the dictionary key, i.e. each individual job name
            for key in sorted( self.jobs ):
               table = self.jobs[key].renderSummary( key, table )
            print( "Maximum concurrent jobs ", str( self.maxJobs ) )
            print( "Prepend host name to logfile: "
                   f"{'Yes' if self.prependHostname else 'No'}" )
            print( table.output().rstrip() )
         else:
            list( self.jobs.values() )[ 0 ].renderName( list( self.jobs )[ 0 ] )

