# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Dict, Int, Model
import TableOutput

#------------------------------------------------------------------------------------
# show system memory errors
#------------------------------------------------------------------------------------
class DimmErrorsModel( Model ):
   correctableErrorCount = Int( help="Correctable error count for this DIMM" )
   uncorrectableErrorCount = Int( help="Uncorrectable error count for this DIMM" )

class MemoryControllerErrorsModel( Model ):
   dimms = Dict( valueType=DimmErrorsModel,
                 help="Maps a DIMM's name to its error counters" )

class ShowSystemMemoryErrorsModel( Model ):
   memoryControllers = Dict( valueType=MemoryControllerErrorsModel,
                             help="Maps a memory controller's name to its DIMMs" )

   def render( self ):
      table = TableOutput.createTable( ( "Memory Controller", "DIMM",
                                          "Correctable Error Count",
                                          "Uncorrectable Error Count" ) )

      textFormat = TableOutput.Format( justify="left" )
      textFormat.noPadLeftIs( True )
      textFormat.padLimitIs( True )
      numberFormat = TableOutput.Format( justify="right" )
      numberFormat.noPadLeftIs( True )
      numberFormat.padLimitIs( True )
      table.formatColumns( textFormat, textFormat, numberFormat, numberFormat )

      correctableErrorTotal = 0
      uncorrectableErrorTotal = 0

      for mcName, mc in sorted( self.memoryControllers.items() ):
         for dimmName, dimm in sorted( mc.dimms.items() ):
            table.newRow( mcName, dimmName, dimm.correctableErrorCount,
                          dimm.uncorrectableErrorCount )
            correctableErrorTotal += dimm.correctableErrorCount
            uncorrectableErrorTotal += dimm.uncorrectableErrorCount

      print( table.output() )
      print( "Total correctable errors:", correctableErrorTotal )
      print( "Total uncorrectable errors:", uncorrectableErrorTotal )
