# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import (
                       Bool,
                       Dict,
                       Model,
                       Float
                     )
from IntfModels import Interface
from ArnetModel import IpGenericPrefix, Ip6Address
from TableOutput import createTable, Format
import Tac
from natsort import natsorted

class Dhcp6InstalledRoute( Model ):
   clientIpAddress = Ip6Address( help="DHCP client's IP address" )
   interface = Interface( help="DHCP client's interface" )
   expiryTime = Float( help="Expiry time of prefix in UTC" )


class Dhcp6InstalledRoutesPerVrf( Model ):
   routes = Dict( keyType=IpGenericPrefix, valueType=Dhcp6InstalledRoute,
                  help="DHCP Relay installed routes keyed by IPv6 prefix" )

class Dhcp6InstalledRoutes ( Model ):
   _isVrfSpecified = Bool( help="User provided VRF as argument to CLI" )
   vrfs = Dict( keyType=str, valueType=Dhcp6InstalledRoutesPerVrf,
                help="DHCP Relay prefixes keyed by VRF" )

   def render( self ):
      fields = [ 'Prefix', 'Client Address', 'Interface', \
                 'Remaining Lifetime (seconds)']

      if not self._isVrfSpecified:
         fields.append( 'VRF' )
      
      currTime = Tac.utcNow()
      outputTableVrf = createTable( fields, tableWidth=100 ) 
      for vrf in sorted( self.vrfs ):
         routes = self.vrfs[ vrf ].routes
         for prefix in natsorted( routes ):
            route = routes[ prefix ]
            expiryTime = route.expiryTime
            remainingTime = int( expiryTime - currTime )
            clientInterface = route.interface.stringValue
            line = [ prefix, route.clientIpAddress, clientInterface,
                     remainingTime ]
            if not self._isVrfSpecified:
               line.append( vrf )
            outputTableVrf.newRow( *line )
   
      f1 = Format( justify='left' )
      f2 = Format( justify='right', maxWidth=18, wrap=True )
      f1.padLimitIs( True )
      f2.padLimitIs( True )
      if not self._isVrfSpecified:
         outputTableVrf.formatColumns( f1, f1, f1, f2, f1 )
      else:
         outputTableVrf.formatColumns( f1, f1, f1, f2 )
      print( outputTableVrf.output() )

   def setVrfSpecified ( self, vrfSpecified ):
      self._isVrfSpecified = vrfSpecified

