# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements IP DHCP Relay configuration.
# It contains interface-specific commands.
#-------------------------------------------------------------------------------

# Unused arguments 'mode' and 'entityManager'
# pylint: disable-msg=W0613

import AclLib, AclCliLib
import CliCommand
import LazyMount, Tac
import CliParser
import ConfigMount
import Arnet
import socket
import CliPlugin.AclCli as AclCli
from CliPlugin.VrfCli import DEFAULT_VRF
from CliPlugin.DhcpRelayHelperCli import InformationOptionMode
from SysConstants.in_h import IPPROTO_UDP

# true if call given from forwarder CLI, used to bypass competing relay check
fromForwarder = False

ipConfig = None
dhcpRelayConfigDir = None
dhcpRelayCounterConfigDir = None
dhcpFwdConfig = None
dhcpRelayStatusDir = None
aclConfig = None
aclCpConfig = None
aclStatus = None
aclCheckpoint = None
mlagDhcpRelayStatusDir = None
platformHardwareSliceDir = None
dhcpRelayReqFloodDisableHwStatusDir = None

dhcp6RouteTypeEnum = Tac.Type( "Ip::Helper::DhcpRelay::Dhcp6RouteType" )

# don't allow syntax from native relay and forwarder to be active at the same time
# in order to switch syntaxes, config must be reset to defaults using currently
# used syntax
def configConflict( mode ):
   if fromForwarder:
      return False
   if dhcpFwdConfig.serverIp or \
          dhcpFwdConfig.serverIntf or \
          dhcpFwdConfig.intfConfig or \
          dhcpFwdConfig.logVerbose:
      mode.addWarning(
         "To configure dhcp helper addresses first remove incompatible configuration"
         " of the form 'ip[v6] dhcp relay [ client | server ] ...'" )
      return True
   return False

def drConfig():
   return dhcpRelayConfigDir

def drCounterConfig():
   return dhcpRelayCounterConfigDir

def drStatus():
   return dhcpRelayStatusDir

def drMlagStatus():
   return mlagDhcpRelayStatusDir

def _drIntfConfig( mode, intfName=None ):
   intfName = intfName if intfName else mode.intf.name
   intfConfig = drConfig().intfConfig.get( intfName )
   return intfConfig

def _getOrCreateDrIntfConfig( mode ):
   intfName = mode.intf.name
   intfConfig = _drIntfConfig( mode )
   if intfConfig is None:
      intfConfig = drConfig().intfConfig.newMember( intfName )
   return intfConfig

def _deleteDrIntfConfig( mode ):
   intfName = mode.intf.name
   if _drIntfConfig( mode ) is not None:
      del drConfig().intfConfig[ intfName ]

def _deleteDrIntfConfigIfAllAttributeHaveDefaults( mode ):
   intfConfig = _drIntfConfig( mode )
   if intfConfig is None:
      return

   if ( not intfConfig.serverIp and
        not intfConfig.serverIp6 and
        not intfConfig.serverHostname and
        not intfConfig.ccapCores and
        not intfConfig.disabledV4 and 
        not intfConfig.disabledV6 and 
        not intfConfig.remoteId and
        intfConfig.circuitId == intfConfig.name and
        intfConfig.route6Install == dhcp6RouteTypeEnum.none and
        intfConfig.smartRelay == 'srDefault' and
        intfConfig.allSubnetsV6 == 'srDefault' ):
      _deleteDrIntfConfig( mode )

def vrfHostnamePair( vrf, hostname ):
   return Tac.Value( "Ip::Helper::DhcpRelay::VrfHostnamePair", vrf,
                     hostname )

def vrfHostnameSrcIntfSrcIp( vrf, hostname, srcIntf, srcIp ):
   intfId = Tac.Value( 'Arnet::IntfId', '' )
   if srcIntf:
      intfId = Tac.Value( 'Arnet::IntfId', srcIntf.name )
   return Tac.Value( "Ip::Helper::DhcpRelay::VrfHostnameSrcIntfSrcIp", vrf,
                     hostname, intfId, Arnet.IpGenAddr( str( srcIp ) ) )

def vrfIpSrcIntfSrcIp( vrf, ip, srcIntf, srcIp ):
   intfId = Tac.Value( 'Arnet::IntfId', '' )
   if srcIntf:
      intfId = Tac.Value( 'Arnet::IntfId', srcIntf.name )
   return Tac.Value( "Ip::Helper::DhcpRelay::VrfIpSrcIntfSrcIp", vrf,
                     Arnet.IpGenAddr( str( ip ) ), intfId,
                     Arnet.IpGenAddr( str( srcIp ) ) )

#-------------------------------------------------------------------------------
# "[no|default] ip helper-address" interface command.
#-------------------------------------------------------------------------------
def noDhcpRelay( mode, args ):
   # disallow removing helper-addresses in native relay in order to maintain
   # consistency between forwarder and relay configs
   if configConflict( mode ):
      return
   intfConfig = _drIntfConfig ( mode )
   if intfConfig:
      intfConfig.serverIp.clear()
      intfConfig.serverHostname.clear()
      _deleteDrIntfConfigIfAllAttributeHaveDefaults( mode )

#-------------------------------------------------------------------------------
# The "[no|default] ip helper-address <ip address/hostname> vrf <vrfName>
# [ source-interface <interface> | source-address <ip address> ]"
# interface command.
#-------------------------------------------------------------------------------
def setIntfDhcpRelay( mode, addr, no=None, vrfName=None, srcIntfIp=None ):
   # This could be converted to ( mode, args ) form, but wasn't because this
   # function is also called from OldDhcpRelayCli (via the callNativeRelay function)
   if configConflict( mode ):
      return
   ipIntfConfig = ipConfig.ipIntfConfig.get( mode.intf.name )
   if not vrfName:
      vrfName = DEFAULT_VRF if ipIntfConfig is None or ipIntfConfig.vrf == "" \
          else ipIntfConfig.vrf
   ipAddr = '0.0.0.0' # INADDR_ANY
   srcIp = '0.0.0.0' # INADDR_ANY
   srcIntf = ''
   vrfHostname = ''

   if no:
      intfConfig = _drIntfConfig( mode )
      if not intfConfig:
         return
   else:
      intfConfig = _getOrCreateDrIntfConfig( mode )

   if srcIntfIp:
      try:
         srcIp = Arnet.IpAddress( srcIntfIp )
      except TypeError:
         srcIntf = srcIntfIp

   # Remove existing VrfIpSrcIntfSrcIp entry where only src intf or src ip changed
   try:
      ipAddr = Arnet.IpAddress( addr )
      vrfIp = vrfIpSrcIntfSrcIp( vrfName, ipAddr, srcIntf, srcIp )
      for key in intfConfig.serverIp:
         if key.vrfName == vrfIp.vrfName and \
            key.ip.v4Addr == vrfIp.ip.v4Addr:
            del intfConfig.serverIp[ key ]
   except ValueError:
      # pylint: disable-msg=E1103
      vrfHostname = vrfHostnameSrcIntfSrcIp( vrfName, addr, srcIntf, srcIp )
      for key in intfConfig.serverHostname:
         if key.vrfName == vrfHostname.vrfName and \
            key.hostname == vrfHostname.hostname:
            del intfConfig.serverHostname[ key ]

   if no:
      _deleteDrIntfConfigIfAllAttributeHaveDefaults( mode )
   else:
      # pylint: disable-msg=simplifiable-if-statement
      if vrfHostname:
         intfConfig.serverHostname[ vrfHostname ] = True
      else:
         intfConfig.serverIp[ vrfIp ] = True 

def setIntfDhcpRelayHandler( mode, args ):
   intf = args.get( 'INTF' ) or args.get( 'ADDR' )
   no = CliCommand.isNoOrDefaultCmd( args )
   setIntfDhcpRelay( mode, args[ 'HOST_OR_ADDR' ], no,
                     vrfName=args.get( 'VRF' ), srcIntfIp=intf )

#-------------------------------------------------------------------------------
# "[no|default] ipv6 helper-address" interface command.
#-------------------------------------------------------------------------------

def noDhcpRelayv6( mode, args ):
   # disallow removing helper-addresses in native relay in order to maintain
   # consistency between forwarder and relay configs
   if configConflict( mode ):
      return
   intfConfig = _drIntfConfig ( mode )
   if intfConfig:
      intfConfig.serverIp6.clear()
      _deleteDrIntfConfigIfAllAttributeHaveDefaults( mode )

#-------------------------------------------------------------------------------
# The "[no|default] ip dhcp smart-relay' interface command.
#-------------------------------------------------------------------------------

def setIntfSmartRelay( mode, args ):
   default = CliCommand.isDefaultCmd( args )
   no = CliCommand.isNoCmd( args )
   if configConflict( mode ):
      return
   if default:
      intfConfig = _drIntfConfig( mode )
   else:
      intfConfig = _getOrCreateDrIntfConfig( mode )
   if intfConfig is None:
      return
   if default:
      # The global config will be overriden to enable smart relay
      intfConfig.smartRelay = 'srDefault'
   elif no:
      # smart relay is disabled regardless of the global config
      intfConfig.smartRelay = 'srOff'
   else:
      intfConfig.smartRelay = 'srOn'
   _deleteDrIntfConfigIfAllAttributeHaveDefaults( mode )
   
#-------------------------------------------------------------------------------
# The "[no|default] ipv6 dhcp realy all-subnets' interface command.
#-------------------------------------------------------------------------------

def setIntfAllSubnetsRelayV6( mode, args ):
   default = CliCommand.isDefaultCmd( args )
   no = CliCommand.isNoCmd( args )
   if configConflict( mode ):
      return
   if default:
      intfConfig = _drIntfConfig( mode )
   else:
      intfConfig = _getOrCreateDrIntfConfig( mode )
   if intfConfig is None:
      return
   if default:
      # The global config will be overriden to enable smart relay
      intfConfig.allSubnetsV6 = 'srDefault'
   elif no:
      # smart relay is disabled regardless of the global config
      intfConfig.allSubnetsV6 = 'srOff'
   else:
      intfConfig.allSubnetsV6 = 'srOn'
   _deleteDrIntfConfigIfAllAttributeHaveDefaults( mode )

#-------------------------------------------------------------------------------
# The "[no|default] ipv6 helper-address <ipv6 address> vrf <vrf name > 
# [ source-interface <interface> | source-address <ipv6 address> ]
# [ link-address <ipv6 address> ]"
# interface command.
#-------------------------------------------------------------------------------

def setIntfDhcp6Relay( mode, args ):
   addrv6 = args[ 'IP6ADDR' ]
   no = CliCommand.isNoOrDefaultCmd( args ) 
   vrfName = args.get( 'VRF' )
   srcIntfIp6 = args.get( 'ADDR' ) or args.get( 'INTF' )
   linkAddr = args.get( 'LINKADDR' )
   if configConflict( mode ):
      return
   srcIp = '::'
   srcIntf= ''
   linkIp = Arnet.IpAddress( 0, addrFamily=socket.AF_INET6 )
   if srcIntfIp6:
      try:
         srcIp = Arnet.IpAddress( srcIntfIp6, addrFamily=socket.AF_INET6 )
      except TypeError:
         srcIntf = srcIntfIp6
   if linkAddr:
      linkIp = Arnet.IpAddress( linkAddr, addrFamily=socket.AF_INET6 )

   ipIntfConfig = ipConfig.ipIntfConfig.get( mode.intf.name )
   if not vrfName:
      vrfName = DEFAULT_VRF if ipIntfConfig is None or ipIntfConfig.vrf == "" \
          else ipIntfConfig.vrf
   if no:
      intfConfig = _drIntfConfig( mode )
      if intfConfig:
         if srcIp != '::' or srcIntf != '':
            del intfConfig.serverIp6[ vrfIpSrcIntfSrcIp( vrfName, addrv6, srcIntf,
                                                         srcIp ) ]
         else:
            for key in intfConfig.serverIp6:
               if key.ip.v6Addr == addrv6 and \
                  key.vrfName == vrfName:
                  del intfConfig.serverIp6[ key ]
         _deleteDrIntfConfigIfAllAttributeHaveDefaults( mode )
   else:
      intfConfig = _getOrCreateDrIntfConfig( mode )
      # Remove existing entry where only src ip or src intf is changed.
      ip6Addr = Arnet.Ip6Addr( addrv6 )
      if ip6Addr:
         for key in intfConfig.serverIp6:
            if key.ip.v6Addr == ip6Addr and \
               key.vrfName == vrfName:
               del intfConfig.serverIp6[ key ]

      if intfConfig:
         intfConfig.serverIp6[ vrfIpSrcIntfSrcIp( vrfName, addrv6, srcIntf,
                                                  srcIp ) ] = linkIp

def setIntfDhcp6RelayRtInstall( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   pd = args.get( 'prefix-delegation' )
   if configConflict( mode ):
      return
   if no:
      intfConfig = _drIntfConfig( mode )
      if intfConfig:
         intfConfig.route6Install = dhcp6RouteTypeEnum.none
         _deleteDrIntfConfigIfAllAttributeHaveDefaults( mode )
   else:
      intfConfig = _getOrCreateDrIntfConfig( mode )
      if intfConfig:
         intfConfig.route6Install = dhcp6RouteTypeEnum.iapd if pd else \
                                    dhcp6RouteTypeEnum.all

#-------------------------------------------------------------------------------
# The "[no|default] ipv6 dhcp relay install routes [prefix-delegation]" 
# interface command.
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# "[no|default] ipv6 dhcp relay add vendor-option ccap-core <ipv6 address>" 
# interface command
#-------------------------------------------------------------------------------
def setDhcp6CcapCores( mode, args ):
   corev6 = args[ 'CORE' ]
   if configConflict( mode ):
      return
   intfConfig = _getOrCreateDrIntfConfig( mode )
   if intfConfig:
      intfConfig.ccapCores[ corev6 ] = True

def noDhcp6CcapCores( mode, args ):
   if configConflict( mode ):
      return
   if 'CORE' in args:
      corev6 = args[ 'CORE' ]
      intfConfig = _drIntfConfig( mode )
      if intfConfig:
         del intfConfig.ccapCores[ corev6 ]
   else:
      intfConfig = _drIntfConfig( mode )
      if intfConfig:
         intfConfig.ccapCores.clear()
   _deleteDrIntfConfigIfAllAttributeHaveDefaults( mode )

#-------------------------------------------------------------------------------
# The "ip dhcp relay information option circuit-id <circuit-id>" interface command.
#-------------------------------------------------------------------------------

def setIntfCircuitId( mode, args ):
   newId = args[ 'ID' ]
   if configConflict( mode ):
      return
   # check validity of format if set to hex
   if drConfig().circuitIdEncodingFormat == '%x':
      try:
         int( newId, 16 )
      except ValueError:
         mode.addError( 'Invalid circuit ID format' )
         return
      if newId.lower().startswith( '0x' ):
         mode.addError( 'Invalid circuit ID format' )
         return
   if newId == mode.intf.name:
      # no need to create the config just to set a default value
      intfConfig = _drIntfConfig( mode )
   else:
      intfConfig = _getOrCreateDrIntfConfig( mode )
   if intfConfig:
      intfConfig.circuitId = newId
      _deleteDrIntfConfigIfAllAttributeHaveDefaults( mode )

def noIntfCircuitId( mode, args ):
   if configConflict( mode ):
      return
   intfConfig = _drIntfConfig( mode )
   if intfConfig:
      intfConfig.circuitId = intfConfig.name
      _deleteDrIntfConfigIfAllAttributeHaveDefaults( mode )

def setIntfRemoteId( mode, args ):
   newId = args[ 'ID' ]
   if configConflict( mode ):
      return
   # check validity of format if set to hex
   if drConfig().remoteIdEncodingFormatV4 == '%x':
      try:
         int( newId, 16 )
      except ValueError:
         mode.addError( 'Invalid remote ID format' )
         return
      if newId.lower().startswith( '0x' ):
         mode.addError( 'Invalid remote ID format' )
         return
   intfConfig = _getOrCreateDrIntfConfig( mode )
   if intfConfig:
      intfConfig.remoteId = newId
      _deleteDrIntfConfigIfAllAttributeHaveDefaults( mode )

def noIntfRemoteId( mode, args ):
   if configConflict( mode ):
      return
   intfConfig = _drIntfConfig( mode )
   if intfConfig:
      intfConfig.remoteId = '' 
      _deleteDrIntfConfigIfAllAttributeHaveDefaults( mode )

def setDhcpRelayDisabledCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   ipv4 = 'ipv4' in args
   disabledState = not no
   if configConflict( mode ):
      return

   intfConfig = _getOrCreateDrIntfConfig( mode )
   # IPv4
   if ipv4:
      intfConfig.disabledV4 = disabledState
   
   # IPv6
   else:
      intfConfig.disabledV6 = disabledState
   
   _deleteDrIntfConfigIfAllAttributeHaveDefaults( mode )

#-------------------------------------------------------------------------------
# "[no|default] dhcp relay ( ipv4 | ipv6 ) disabled" 
# interface command
#-------------------------------------------------------------------------------

#--------------------------------------------------------------------------------
# clear ( ip | ipv6 ) dhcp relay counters INTF
#--------------------------------------------------------------------------------
def resetDhcpRelayCounters( mode, args ):
   intf = args.get( 'INTF' )
   if intf is None:
      # pylint: disable-msg=E1103
      # Pylint generates the following very strange error here:
      # Instance of '_Proxy' has no 'resetAllCountersTrigger' member
      # (but some types could not be inferred)
      drCounterConfig().resetAllCountersTrigger = \
            ( drCounterConfig().resetAllCountersTrigger + 1 ) % 256
   else:
      intfId = intf.name
      # Only act on the clear counter command when there is an intfConfig
      # or egressIntf associated with it. If the counter intfConfig doesn't exist,
      # then create one. Counter intfConfig is never deleted.
      counterStatus = drStatus().counterStatus
      if not counterStatus:
         return
      counterIntfs = counterStatus.counterIntfStatus
      if intfId in counterIntfs:
         intfCounterConfig = drCounterConfig().intfConfig.get( intfId )
         if intfCounterConfig is None:
            intfCounterConfig = drCounterConfig().intfConfig.newMember( intfId )
         intfCounterConfig.resetCountersTrigger = \
            ( intfCounterConfig.resetCountersTrigger + 1 ) % 256
         return
      # do nothing until the actual intfConfig is created

#--------------------------------------------------------------------------------
# clear ip dhcp relay counters access-list
#--------------------------------------------------------------------------------
def clearAclCounters( mode, args ):
   aclType = args[ 'ADDR_FAMILY' ]
   AclCli.clearServiceAclCounters( mode, aclStatus, aclCheckpoint, aclType )

#############################################################
# clear ipv6 dhcp relay routes [ ( VRF | INTF | PREFIX_V6 ) ]
# in mode: enable
##############################################################
def clearDhcpRelayRoutes( mode, args ):
   vrf = args.get( 'VRF', DEFAULT_VRF )
   intf = args.get( 'INTF' )
   prefix =  args.get( 'PREFIX_V6' )
   clearPrefixBindingSpecified = False
   defaultPrefix = "::/0"

   # This allows clearing ::/0 specifically
   if prefix or str( prefix ) == defaultPrefix:
      prefix = Arnet.Ip6Prefix( prefix )
      clearPrefixBindingSpecified = True
   else:
      prefix = Arnet.Ip6Prefix( "::/0" )

   if intf:
      intfConfig = _drIntfConfig( None, intfName=intf.name )
      if intfConfig:
         counter = ( intfConfig.clearAllPrefixBinding + 1 ) % 256
         intfConfig.clearAllPrefixBinding = counter
   else:
      config = drConfig()
      config.clearPrefixBindingSpecified = clearPrefixBindingSpecified
      config.clearThisPrefixBinding = prefix
      config.clearThisPrefixBindingVrf = vrf
      counter = ( config.clearPrefixBinding + 1 ) % 256
      config.clearPrefixBinding = counter

#--------------------------------------------------------------------------------
# [ no | default ] ( ip | ipv6 ) dhcp relay always-on
#--------------------------------------------------------------------------------
def setAlwaysOn( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   if configConflict( mode ):
      return
   drConfig().alwaysOn = not no

#--------------------------------------------------------------------------------
# [ no | default ] ( ip | ipv6 ) dhcp relay qos dscp DSCP
#--------------------------------------------------------------------------------
def setDscp( mode, args ):
   dscpValue = args[ 'DSCP' ]
   if 'ip' in args:
      drConfig().dscpV4 = dscpValue
   else:
      drConfig().dscpV6 = dscpValue

def noDscp( mode, args ):
   if 'ip' in args:
      drConfig().dscpV4 = drConfig().dscpValueDefault
   else:
      drConfig().dscpV6 = drConfig().dscpValueDefault

#--------------------------------------------------------------------------------
# [ no | default ] ip dhcp relay access-group ACLNAME [ vrf VRFNAME ] [ in ]
#--------------------------------------------------------------------------------
def setAcl( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   aclName = args.get( 'ACL' )
   vrfName = args.get( 'VRF' )
   ipv4 = 'ip' in args
   aclType = 'ip' if ipv4  else 'ipv6'
   proto = 'bootps' if ipv4 else 'dhcpv6-server' 
   if no:
      AclCliLib.noServiceAcl( mode, 'dhcpRelay', aclConfig, aclCpConfig, 
                              aclName, aclType, vrfName )
   else:
      AclCliLib.setServiceAcl( mode, 'dhcpRelay', IPPROTO_UDP,
                               aclConfig, aclCpConfig, aclName, aclType, vrfName,
                               port=[ AclLib.getServByName( IPPROTO_UDP, proto ) ] )

#--------------------------------------------------------------------------------
# [ no | default ] ip dhcp relay all-subnets default
#--------------------------------------------------------------------------------
def setSmartRelay( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   if configConflict( mode ):
      return
   drConfig().smartRelayGlobal = not no

#--------------------------------------------------------------------------------
# [ no | default ] ip dhcp smart-relay global
#--------------------------------------------------------------------------------
#--------------------------------------------------------------------------------
# [ no | default ] ip dhcp relay information option
#--------------------------------------------------------------------------------
def setIntfDhcpRelayInfo( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   if configConflict( mode ):
      return
   if not no and drConfig().informationOptOn:
      mode.addWarning( '"ip dhcp relay information option" is superseded'
                       ' by information-option mode' )
   drConfig().circuitIdOptOn = not no

#--------------------------------------------------------------------------------
# [ no | default ] ipv6 dhcp relay option link-layer address
#--------------------------------------------------------------------------------
def setDhcpRelayv6LinkLayerAddrOpt( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   if configConflict( mode ):
      return
   drConfig().clientLinkLayerAddrOptOn = not no

#--------------------------------------------------------------------------------
# "[no|default] ipv6 dhcp relay option remote-id format ( %m:%i | %m:%p )"
#-------------------------------------------------------------------------------
def setDhcpRelayv6RemoteIdOptFormat( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   if configConflict( mode ):
      return
   if no:
      drConfig().remoteIdEncodingFormat = drConfig().remoteIdEncodingFormatDefault
      return
   if "%m:%i" in args:
      drConfig().remoteIdEncodingFormat = '%m:%i'
   elif "%m:%p" in args:
      drConfig().remoteIdEncodingFormat = '%m:%p'

#--------------------------------------------------------------------------------
# [ no | default } ipv6 dhcp relay all-subnets default
#--------------------------------------------------------------------------------
def setSmartRelayV6( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   if configConflict( mode ):
      return
   drConfig().allSubnetsV6Global = not no

#--------------------------------------------------------------------------------
# dhcp relay
# in mode: config
#--------------------------------------------------------------------------------
def setDhcpRelayGlobalDefaultValues( mode, args ):
   # set everything to the default state
   config = drConfig()
   config.serverIpGlobal.clear()
   config.serverHostnameGlobal.clear()
   config.serverIp6Global.clear()
   config.tunnelReqDisable = False
   config.mlagPeerLinkReqDisable = False
   config.reqFloodDisableVlan.clear()
   config.vssControlDisable = False
   config.pdRoutesPersistenceEnabled = True
   config.circuitIdEncodingFormat = config.circuitIdEncodingFormatDefault
   config.remoteIdEncodingFormatV4 = config.remoteIdEncodingFormatDefaultV4
   config.informationOptOn = False

#--------------------------------------------------------------------------------
# information option
# in mode: dhcp relay
#--------------------------------------------------------------------------------
def setInformationOptionMode( mode, args):
   childMode = mode.childMode( InformationOptionMode )
   mode.session_.gotoChildMode( childMode )
   config = drConfig()
   if config.circuitIdOptOn:
      mode.addWarning( "This supersedes previously configured"
                       " 'ip dhcp relay information option' command" )
      config.circuitIdOptOn = False
   config.informationOptOn = True

def setInformationOptionDefault( mode, args ):
   config = drConfig()
   config.informationOptOn = False
   drConfig().circuitIdEncodingFormat = drConfig().circuitIdEncodingFormatDefault
   drConfig().remoteIdEncodingFormatV4 = \
         drConfig().remoteIdEncodingFormatDefaultV4

#--------------------------------------------------------------------------------
# (config-information-option) [ no | default ]
#      ( circuit-id encoding ( %x | %p )
#--------------------------------------------------------------------------------
def setIpv4DhcpRelayOption82CircuitIdFormat( mode, args ):
   if configConflict( mode ):
      return

   drConfig().circuitIdEncodingFormat = args.get( "%x", "%p" )

def noIpv4DhcpRelayOption82CircuitIdFormat( mode, args ):
   if configConflict( mode ):
      return
   drConfig().circuitIdEncodingFormat = drConfig().circuitIdEncodingFormatDefault

#--------------------------------------------------------------------------------
# (config-information-option) [ no | default ]
#      ( remote-id encoding ( %x | %p )
#--------------------------------------------------------------------------------
def setIpv4DhcpRelayOption82RemoteIdFormat( mode, args ):
   if configConflict( mode ):
      return

   drConfig().remoteIdEncodingFormatV4 = args.get( "%x", "%p" )

def noIpv4DhcpRelayOption82RemoteIdFormat( mode, args ):
   if configConflict( mode ):
      return
   drConfig().remoteIdEncodingFormatV4 = \
         drConfig().remoteIdEncodingFormatDefaultV4

#--------------------------------------------------------------------------------
# (config-information-option) [ no | default ] vendor-option
#--------------------------------------------------------------------------------
def setIpv4DhcpRelayOption82VendorSpecificSubOpt( mode, args ):
   if configConflict( mode ):
      return
   drConfig().vendorSpecificSubOptOn = True

def noIpv4DhcpRelayOption82VendorSpecificSubOpt( mode, args ):
   if configConflict( mode ):
      return
   drConfig().vendorSpecificSubOptOn = False

#--------------------------------------------------------------------------------
# server ( IP_GENADDR | HOSTNAME )
# in mode: dhcp relay
# Equivalent to helper-address in interface mode
#--------------------------------------------------------------------------------
def setDhcpRelayGlobalServerCmd( mode, args ):
   if configConflict( mode ):
      return

   ipAddr = args.get( 'IP_ADDR' )
   ipv4 = isinstance( ipAddr, Tac.Type( 'Arnet::IpAddr' ) )
   ipv6 = isinstance( ipAddr, Tac.Type( 'Arnet::Ip6Addr' ) )
   key = args[ 'key' ]
   config = drConfig()
   no = CliCommand.isNoOrDefaultCmd( args )

   # remove
   if no:
      if ipv4:
         del config.serverIpGlobal[ key ]
      elif ipv6:
         del config.serverIp6Global[ key ]
      else:
         del config.serverHostnameGlobal[ key ]

   # add
   else:
      if ipv4:
         config.serverIpGlobal[ key ] = True
      elif ipv6:
         # We should match setIntfDhcp6Relay implementation
         # in order to have a config like intfConfig. 
         linkIp = Arnet.IpAddress( 0, addrFamily=socket.AF_INET6 )
         config.serverIp6Global[ key ] = linkIp
      else:
         config.serverHostnameGlobal[ key ] = True

#--------------------------------------------------------------------------------
# (config)# dhcp relay
# (config-dhcp-relay)# [ no | default ] tunnel requests disabled
#--------------------------------------------------------------------------------
def setTunnelRequestsDisabled( mode, args ):
   config = drConfig()
   no = CliCommand.isNoOrDefaultCmd( args )
   config.tunnelReqDisable = not no

#--------------------------------------------------------------------------------
# (config)# dhcp relay
# (config-dhcp-relay)# [ no | default ] mlag peer-link requests disabled
#--------------------------------------------------------------------------------
def setMlagPeerLinkRequestsDisabled( mode, args ):
   config = drConfig()
   no = CliCommand.isNoOrDefaultCmd( args )
   config.mlagPeerLinkReqDisable = not no

#--------------------------------------------------------------------------------
# (config)# dhcp relay
# (config-dhcp-relay)# [no|default] client requests flooding suppression
# vlan <vlan on vlanset>
#--------------------------------------------------------------------------------
def setReqFloodDisableVlan( mode, args ):
   config = drConfig()
   for vlan in args[ 'VLANSET' ]:
      config.reqFloodDisableVlan[ vlan ] = True

def resetReqFloodDisableVlan( mode, args ):
   config = drConfig()
   vlans = args.get( 'VLANSET' )
   if vlans:
      for vlan in vlans:
         del config.reqFloodDisableVlan[ vlan ]
   else:
      config.reqFloodDisableVlan.clear()

def dhcpRelayReqFloodDisableHwStatusReady( sliceInfo, hwStatus ):
   if sliceInfo and ( sliceInfo.generationId == hwStatus.genId ) and \
         hwStatus.dhcpReqFloodingDisableSupported:
      return True
   elif not sliceInfo and hwStatus.dhcpReqFloodingDisableSupported and \
         ( hwStatus.genId == 0xffffffff ):
      # For a system that creates only one instance of HwStatus, the
      # genId should be set to the max value.
      return True
   else:
      return False

def dhcpRelayReqFloodDisableSupported():
   if not dhcpRelayReqFloodDisableHwStatusDir:
      return False
   for sliceId, hwStatus in dhcpRelayReqFloodDisableHwStatusDir.items():
      sliceInfo = platformHardwareSliceDir.sliceInfo.get( sliceId )
      if dhcpRelayReqFloodDisableHwStatusReady( sliceInfo, hwStatus ):
         return True
   return False

def dhcpRelayReqFloodDisableGuard( mode, token ):
   if dhcpRelayReqFloodDisableSupported():  
      return None
   return CliParser.guardNotThisPlatform

#--------------------------------------------------------------------------------
# (config)# dhcp relay
# (config-dhcp-relay)# [ no | default ] ipv6 routes persistence disabled
#--------------------------------------------------------------------------------
def setPersistenceConfig( mode, args ):
   drConfig().pdRoutesPersistenceEnabled = False

def unsetPersistenceConfig( mode, args ):
   drConfig().pdRoutesPersistenceEnabled = True

#--------------------------------------------------------------------------------
# (config)# dhcp relay
# (config-dhcp-relay)# [ no | default ] vss control disabled
#--------------------------------------------------------------------------------
def setVssControlDisabled( mode, args ):
   drConfig().vssControlDisable = True

def unsetVssControlDisabled( mode, args ):
   drConfig().vssControlDisable = False

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   # Global variable undefined so far
   global ipConfig
   global dhcpRelayConfigDir
   global dhcpRelayCounterConfigDir
   global dhcpFwdConfig
   global dhcpRelayStatusDir
   global aclConfig
   global aclCpConfig
   global aclStatus
   global aclCheckpoint
   global mlagDhcpRelayStatusDir
   global platformHardwareSliceDir
   global dhcpRelayReqFloodDisableHwStatusDir

   # pkgdeps: rpmwith %{_libdir}/SysdbMountProfiles/DhcpRelay
   ipConfig = LazyMount.mount( entityManager, "ip/config", "Ip::Config", "r" )
   dhcpRelayConfigDir = ConfigMount.mount( entityManager,
                                           "ip/helper/dhcprelay/config",
                                           "Ip::Helper::DhcpRelay::Config", "w" )
   dhcpRelayCounterConfigDir = LazyMount.mount( entityManager,
                                    "ip/helper/dhcprelay/counterConfig",
                                    "Ip::Helper::DhcpRelay::CounterConfig", "w" )
   # check the config for the incompatible dhcp-forwarder
   dhcpFwdConfig = LazyMount.mount( entityManager, "ip/dhcp/relay/config",
                                    "Ip::Dhcp::Relay::Config", "r" )
   dhcpRelayStatusDir = LazyMount.mount( entityManager, "ip/helper/dhcprelay/status",
                                         "Ip::Helper::DhcpRelay::Status", "r" )
   mlagDhcpRelayStatusDir = LazyMount.mount( entityManager,
                                             "ip/helper/dhcprelay/mlagStatus",
                                             "MlagDhcpRelay::Status", "r" )
   aclConfig = ConfigMount.mount( entityManager, "acl/config/cli",
                                  "Acl::Input::Config", "w" )
   aclCpConfig = ConfigMount.mount( entityManager, "acl/cpconfig/cli",
                                  "Acl::Input::CpConfig", "w" )
   aclStatus = LazyMount.mount( entityManager, "acl/status/all",
                                "Acl::Status", "r" )
   aclCheckpoint = LazyMount.mount( entityManager, "acl/checkpoint",
                                   "Acl::CheckpointStatus", "w" )
   platformHardwareSliceDir = LazyMount.mount(
         entityManager, "platform/hardware/slice",
         "Hardware::PlatformSliceDir", "r" )
   dhcpRelayReqFloodDisableHwStatusDir = LazyMount.mount(
         entityManager, "ip/helper/dhcprelay/hardware/status",
         "Tac::Dir", "ri" )
