#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Ark
import Arnet
import ConfigMount
import LazyMount
import Tac
import CliPlugin.AclCli as AclCli
import CliDynamicSymbol
from CliPlugin.DhcpRelayHelperCli import (
   drMlagStatus, drStatus,
)
from DhcpSnoopingCliLib import dhcpSnoopingHwStatusReady

aclCpConfig = None
aclStatus = None
aclCheckpoint = None
dhcpSnoopingConfig = None
dhcpSnoopingCounterConfig = None
dhcpSnoopingHwStatusDir = None
dhcpSnoopingStatus = None
bridgingConfig = None
platformHardwareSliceDir = None
dhcp6SnoopingConfig = None
dhcp6SnoopingHwStatusDir = None
dhcp6SnoopingStatus = None
dhcp6SnoopingCounterConfig = None

def getRelayCommonModels():
   return CliDynamicSymbol.loadDynamicPlugin( "DhcpRelayCommonModel" )

def getRelaySnoopingModels():
   return CliDynamicSymbol.loadDynamicPlugin( "DhcpRelaySnoopingModel" )

#--------------------------------------------------------------------------------
# show ( ip | ipv6 ) dhcp relay access-list [ ACLNAME ]
#--------------------------------------------------------------------------------
def showAcl( mode, args ):
   aclType = args[ 'ADDR_FAMILY' ]
   name = args.get( '<aclNameExpr>' )
   return AclCli.showServiceAcl( mode,
                                 aclCpConfig,
                                 aclStatus,
                                 aclCheckpoint,
                                 aclType,
                                 name,
                                 serviceName='dhcpRelay' )

#--------------------------------------------------------------------------------
# show ip dhcp snooping
#--------------------------------------------------------------------------------
def showDhcpSnooping( mode, args ):
   snoopingModel = getRelaySnoopingModels()
   snoop = snoopingModel.DhcpSnoopingModel()
   snoop.enabled = dhcpSnoopingConfig.enabled
   snoop.operational = dhcpSnoopingStatus.enabled
   snoop.enabledVlans = list( dhcpSnoopingConfig.vlan )
   snoop.operationalVlans = list( dhcpSnoopingStatus.vlanStatus )
   snoop.option82Enabled = dhcpSnoopingConfig.informationOption
   snoop.bridgingEnabled = dhcpSnoopingConfig.bridging

   if dhcpSnoopingConfig.informationOption:
      snoop.circuitIdEnabled = dhcpSnoopingConfig.circuitIdTypeValid
      if dhcpSnoopingConfig.circuitIdTypeValid:
         snoop.circuitIdType = dhcpSnoopingConfig.circuitIdType
      snoop.circuitIdFormat = dhcpSnoopingConfig.circuitIdFormatString

      for intfName in dhcpSnoopingConfig.userDefinedCircuitId:
         circuitIdModel = snoopingModel.CircuitIdModel()
         userConfigIntf = dhcpSnoopingConfig.userDefinedCircuitId[ intfName ]
         circuitIdModel.name = intfName
         circuitIdModel.valid = userConfigIntf.circuitIdTypeValid
         circuitIdModel.value = userConfigIntf.circuitIdVal
         if userConfigIntf.circuitIdTypeValid:
            circuitIdModel.circuitType = userConfigIntf.circuitIdType
         snoop.circuitIds[ intfName ] = circuitIdModel
      snoop.bridgeMac = bridgingConfig.bridgeMacAddr      
   return snoop

#--------------------------------------------------------------------------------
# show ip dhcp snooping counters [ vlan VLANSET ] [ detail ]
#--------------------------------------------------------------------------------
def updateCounters( mode, v6=False ):
   dsCounterConfig = dhcp6SnoopingCounterConfig if v6 else dhcpSnoopingCounterConfig
   dsStatus = dhcp6SnoopingStatus if v6 else dhcpSnoopingStatus
   dsCounterConfig.counterUpdateRequestTime = Tac.now()
   def countersUpdated():
      return ( dsStatus.counterUpdateTime >=
               dsCounterConfig.counterUpdateRequestTime )
   try:
      Tac.waitFor( countersUpdated, description="counter update",
                   maxDelay=0.1, sleep=True, timeout=30.0 )
   except Tac.Timeout:
      mode.addWarning( "Displaying stale counters" )

def showDhcpSnoopingCountersHelper( mode, args, v6=False ):
   snoopingModel = getRelaySnoopingModels()
   counterModel = ( snoopingModel.Dhcp6SnoopingCounterModel() if v6 
                    else snoopingModel.DhcpSnoopingCounterModel() )
   dsConfig = dhcp6SnoopingConfig if v6 else dhcpSnoopingConfig
   dsStatus = dhcp6SnoopingStatus if v6 else dhcpSnoopingStatus
   counterModel.enabled = dsConfig.enabled
   if counterModel.enabled:
      updateCounters( mode, v6=v6 )
      vlans = args.get( 'VLANSET', dsStatus.vlanStatus )
      for vlan in vlans:
         vlanStatus = dsStatus.vlanStatus.get( vlan )
         if vlanStatus is None:
            continue
         vlanCounters = snoopingModel.VlanCounters()
         vlanCounters.vlan = vlan
         vlanCounters.requestsReceived = vlanStatus.requestsReceived
         vlanCounters.requestsForwarded = vlanStatus.requestsForwarded
         vlanCounters.requestsDropped = vlanStatus.requestsDropped
         vlanCounters.repliesReceived = vlanStatus.repliesReceived
         vlanCounters.repliesForwarded = vlanStatus.repliesForwarded
         vlanCounters.repliesDropped = vlanStatus.repliesDropped
         vlanCounters.resetTime = Ark.switchTimeToUtc( vlanStatus.lastResetTime )
         for intf in vlanStatus.counterIntfStatus:
            counterIntfStatus = vlanStatus.counterIntfStatus[ intf ]
            intfCounter = snoopingModel.IntfCounters()
            intfCounter.requestsReceived = counterIntfStatus.requestsReceived
            intfCounter.requestsForwarded = counterIntfStatus.requestsForwarded
            intfCounter.requestsDropped = counterIntfStatus.requestsDropped
            intfCounter.repliesReceived = counterIntfStatus.repliesReceived
            intfCounter.repliesForwarded = counterIntfStatus.repliesForwarded
            intfCounter.repliesDropped = counterIntfStatus.repliesDropped
            vlanCounters.interfaces[ intf ] = intfCounter
         counterModel.vlanCounters[ vlan ] = vlanCounters
      counterModel.detail_ = 'detail' in args
   return counterModel
 
def showDhcpSnoopingCounters( mode, args ):
   return showDhcpSnoopingCountersHelper( mode, args, v6=False )

#--------------------------------------------------------------------------------
# show ip dhcp snooping counters debug
#--------------------------------------------------------------------------------
def showDhcpSnoopingDebugCounters( mode, args ):
   snoopingModel = getRelaySnoopingModels()
   debugCounterModel = snoopingModel.DhcpSnoopingDebugCounterModel()
   debugCounterModel.enabled = dhcpSnoopingConfig.enabled

   if dhcpSnoopingConfig.enabled:
      updateCounters( mode )
      counters = dhcpSnoopingStatus.debugCounters
      if not counters:
         return debugCounterModel

      receivedCounters = snoopingModel.SnoopingRelayCounters()
      receivedCounters.snoopingToRelay = counters.requestsReceived
      receivedCounters.relayToSnooping = counters.responsesReceived
      debugCounterModel.receivedCounters = receivedCounters

      forwardedCounters = snoopingModel.SnoopingRelayCounters()
      forwardedCounters.snoopingToRelay = counters.requestsForwarded
      forwardedCounters.relayToSnooping = counters.responsesForwarded
      debugCounterModel.forwardedCounters = forwardedCounters

      vlanIdErrCounters = snoopingModel.SnoopingRelayCounters()
      vlanIdErrCounters.snoopingToRelay = counters.requestsDropVlanIdErr
      vlanIdErrCounters.relayToSnooping = counters.responsesDropVlanIdErr
      debugCounterModel.vlanIdErrCounters = vlanIdErrCounters

      parseErrCounters = snoopingModel.SnoopingRelayCounters()
      parseErrCounters.snoopingToRelay = counters.requestsDropParseErr
      parseErrCounters.relayToSnooping = counters.responsesDropParseErr
      debugCounterModel.parseErrCounters = parseErrCounters

      dhcpOpErrCounters = snoopingModel.SnoopingRelayCounters()
      dhcpOpErrCounters.snoopingToRelay = counters.requestsDropDhcpOpErr
      dhcpOpErrCounters.relayToSnooping = counters.responsesDropDhcpOpErr
      debugCounterModel.dhcpOpErrCounters = dhcpOpErrCounters

      infoOptErrCounters = snoopingModel.SnoopingRelayCounters()
      infoOptErrCounters.snoopingToRelay = counters.requestsDropInfoOptErr
      infoOptErrCounters.relayToSnooping = counters.responsesDropInfoOptErr
      debugCounterModel.infoOptErrCounters = infoOptErrCounters

      disabledErrCounters = snoopingModel.SnoopingRelayCounters()
      disabledErrCounters.snoopingToRelay = counters.requestsDropDisabled
      disabledErrCounters.relayToSnooping = counters.responsesDropDisabled
      debugCounterModel.disabledErrCounters = disabledErrCounters

      debugCounterModel.resetTime = Ark.switchTimeToUtc( counters.lastResetTime ) 
   
   return debugCounterModel

#--------------------------------------------------------------------------------
# show ip dhcp snooping hardware
#--------------------------------------------------------------------------------
def showDhcpSnoopingHardware( mode, args ):
   hardwareModel = getRelaySnoopingModels().DhcpSnoopingHardwareModel()

   for sliceId, hwStatus in dhcpSnoopingHwStatusDir.items():
      sliceInfo = platformHardwareSliceDir.sliceInfo.get( sliceId )
      if not dhcpSnoopingHwStatusReady( sliceInfo, hwStatus ):
         continue

      if hwStatus.hwDhcpSnoopingEnabled:
         hardwareModel.enabled = True
      vlanList = getRelaySnoopingModels().VlanList()
      vlanList.vlans = list( hwStatus.snoopingVlan )
      hardwareModel.vlansPerSlice[ sliceId ] = vlanList
      for vlan in hwStatus.snoopingVlan:
         if vlan not in hardwareModel.vlans:
            hardwareModel.vlans.append( vlan )
   if hardwareModel.enabled is None:
      hardwareModel.enabled = False
   return hardwareModel

#--------------------------------------------------------------------------------
# show ipv6 dhcp relay installed routes
#--------------------------------------------------------------------------------
def showInstalledRoutes( mode, args ):
   relayModel = getRelayCommonModels()
   installedRoutesModel = relayModel.Dhcp6InstalledRoutes()
   vrfs = installedRoutesModel.vrfs
   prefixesAdded = {}
   
   def populateTable( intfStatus ):
      for assignedAddr in intfStatus.prefixBinding:
         # Do not populate the same prefix twice. This we do for the
         # cases where MLAG Split brain gets restored. In that case the
         # same prefix binding would be present in both dhcpRelayStatus
         # and mlagDhcpRelayStatus. We populate the one present in
         # dhcpRelayStatus only.
         pb = intfStatus.prefixBinding.get( assignedAddr )
         if assignedAddr in prefixesAdded or not pb:
            continue
         addrAttrs = pb.addrAttrs
         
         # If user has specified vrf as input and vrf of this prefix
         # is different from user specified vrf, skip this prefix
         currentVrf = addrAttrs.vrfName
         if vrf != None and currentVrf != vrf:
            continue
         vrfs[ currentVrf ] = vrfs.get(currentVrf,
               relayModel.Dhcp6InstalledRoutesPerVrf() )
         routes = vrfs[ currentVrf ].routes
         prefix = Tac.Value( "Arnet::IpGenPrefix", str( assignedAddr ) )
         routes[ prefix ] = routes.get( prefix, relayModel.Dhcp6InstalledRoute() )
         routes[ prefix ].clientIpAddress = addrAttrs.clientAddr
         routes[ prefix ].interface = intfStatus.intfId
         expiryTime = float(  addrAttrs.allotmentTime + addrAttrs.validLifetime )
         routes[ prefix ].expiryTime = expiryTime

         prefixesAdded[ assignedAddr ] = True

   vrf = args.get( 'VRF' )
   installedRoutesModel.setVrfSpecified( bool( vrf is not None ) )
   
   intfs = drStatus().intfStatus
   mlagIntfs = drMlagStatus().intfStatus

   for intfname in Arnet.sortIntf( intfs ):
      intfStatus = intfs[ intfname ]
      populateTable( intfStatus )
   
   for intfname in Arnet.sortIntf( mlagIntfs ):
      intfStatus = mlagIntfs[ intfname ]
      populateTable( intfStatus )
   return installedRoutesModel

#--------------------------------------------------------------------------------
# show ipv6 dhcp snooping
#--------------------------------------------------------------------------------
def showDhcp6Snooping( mode, args ):
   snoop = getRelaySnoopingModels().Dhcp6SnoopingModel()
   snoop.enabled = dhcp6SnoopingConfig.enabled
   snoop.operational = dhcp6SnoopingStatus.enabled
   snoop.enabledVlans = list( dhcp6SnoopingConfig.vlan )
   snoop.operationalVlans = list( dhcp6SnoopingStatus.vlanStatus )
   snoop.remoteIdOption = dhcp6SnoopingConfig.remoteIdOption
   return snoop

#--------------------------------------------------------------------------------
# show ipv6 dhcp snooping counters [ vlan VLANSET ] [ detail ]
#--------------------------------------------------------------------------------
def showDhcp6SnoopingCounters( mode, args ):
   return showDhcpSnoopingCountersHelper( mode, args, v6=True )

#--------------------------------------------------------------------------------
# show ipv6 dhcp snooping counters debug
#--------------------------------------------------------------------------------
def showDhcp6SnoopingDebugCounters( mode, args ):
   snoopingModel = getRelaySnoopingModels()
   debugCounterModel = snoopingModel.Dhcp6SnoopingDebugCounterModel()
   debugCounterModel.enabled = dhcp6SnoopingConfig.enabled

   if dhcp6SnoopingConfig.enabled:
      updateCounters( mode, v6=True )
      counters = dhcp6SnoopingStatus.debugCounters
      if not counters:
         return debugCounterModel

      receivedCounters = snoopingModel.SnoopingRelayCounters()
      receivedCounters.snoopingToRelay = counters.requestsReceived
      receivedCounters.relayToSnooping = counters.responsesReceived
      debugCounterModel.receivedCounters = receivedCounters

      forwardedCounters = snoopingModel.SnoopingRelayCounters()
      forwardedCounters.snoopingToRelay = counters.requestsForwarded
      forwardedCounters.relayToSnooping = counters.responsesForwarded
      debugCounterModel.forwardedCounters = forwardedCounters

      vlanIdErrCounters = snoopingModel.SnoopingRelayCounters()
      vlanIdErrCounters.snoopingToRelay = counters.requestsDropVlanIdErr
      vlanIdErrCounters.relayToSnooping = counters.responsesDropVlanIdErr
      debugCounterModel.vlanIdErrCounters = vlanIdErrCounters

      parseErrCounters = snoopingModel.SnoopingRelayCounters()
      parseErrCounters.snoopingToRelay = counters.requestsDropParseErr
      parseErrCounters.relayToSnooping = counters.responsesDropParseErr
      debugCounterModel.parseErrCounters = parseErrCounters

      dhcpOpErrCounters = snoopingModel.SnoopingRelayCounters()
      dhcpOpErrCounters.snoopingToRelay = counters.requestsDropDhcpOpErr
      dhcpOpErrCounters.relayToSnooping = counters.responsesDropDhcpOpErr
      debugCounterModel.dhcpOpErrCounters = dhcpOpErrCounters

      remoteIdErrCounters = snoopingModel.SnoopingRelayCounters()
      remoteIdErrCounters.snoopingToRelay = counters.requestsDropInfoOptErr
      remoteIdErrCounters.relayToSnooping = counters.responsesDropInfoOptErr
      debugCounterModel.remoteIdErrCounters = remoteIdErrCounters

      disabledErrCounters = snoopingModel.SnoopingRelayCounters()
      disabledErrCounters.snoopingToRelay = counters.requestsDropDisabled
      disabledErrCounters.relayToSnooping = counters.responsesDropDisabled
      debugCounterModel.disabledErrCounters = disabledErrCounters

      debugCounterModel.resetTime = Ark.switchTimeToUtc( counters.lastResetTime ) 

   return debugCounterModel

#--------------------------------------------------------------------------------
# show ipv6 dhcp snooping hardware
#--------------------------------------------------------------------------------
def showDhcp6SnoopingHardware( mode, args ):
   hardwareModel = getRelaySnoopingModels().Dhcp6SnoopingHardwareModel()

   for sliceId, hwStatus in dhcp6SnoopingHwStatusDir.items():
      sliceInfo = platformHardwareSliceDir.sliceInfo.get( sliceId )
      if not dhcpSnoopingHwStatusReady( sliceInfo, hwStatus ):
         continue

      if hwStatus.hwDhcpSnoopingEnabled:
         hardwareModel.enabled = True
      vlanList = getRelaySnoopingModels().VlanList()
      vlanList.vlans = list( hwStatus.snoopingVlan )
      hardwareModel.vlansPerSlice[ sliceId ] = vlanList
      for vlan in hwStatus.snoopingVlan:
         if vlan not in hardwareModel.vlans:
            hardwareModel.vlans.append( vlan )
   if hardwareModel.enabled is None:
      hardwareModel.enabled = False
   return hardwareModel

def Plugin( entityManager ):
   global bridgingConfig
   global dhcpSnoopingConfig
   global dhcpSnoopingCounterConfig
   global dhcpSnoopingStatus
   global dhcpSnoopingHwStatusDir
   global platformHardwareSliceDir
   global dhcp6SnoopingConfig
   global dhcp6SnoopingCounterConfig
   global dhcp6SnoopingStatus
   global dhcp6SnoopingHwStatusDir
   global aclCpConfig
   global aclStatus
   global aclCheckpoint

   bridgingConfig = LazyMount.mount( entityManager, "bridging/config",
                                     "Bridging::Config", "r" )
   dhcpSnoopingConfig = ConfigMount.mount( entityManager,
                                           "bridging/dhcpsnooping/config",
                                           "Bridging::DhcpSnooping::Config", "w" )
   dhcpSnoopingCounterConfig = LazyMount.mount(
         entityManager, "bridging/dhcpsnooping/counterConfig",
         "Bridging::DhcpSnooping::CounterConfig", "w" )
   dhcpSnoopingStatus = LazyMount.mount( entityManager,
                                         "bridging/dhcpsnooping/status",
                                         "Bridging::DhcpSnooping::Status", "r" )
   dhcpSnoopingHwStatusDir = LazyMount.mount(
         entityManager, "bridging/dhcpsnooping/hardware/status",
         "Tac::Dir", "ri" )
   dhcp6SnoopingCounterConfig = LazyMount.mount(
         entityManager, "bridging/dhcpsnooping/dhcp6CounterConfig",
         "Bridging::DhcpSnooping::CounterConfig", "w" )
   dhcp6SnoopingConfig = ConfigMount.mount( entityManager,
                                            "bridging/dhcpsnooping/dhcp6Config",
                                            "Bridging::DhcpSnooping::Dhcp6Config",
                                            "w" )
   dhcp6SnoopingStatus = LazyMount.mount( entityManager,
                                          "bridging/dhcpsnooping/dhcp6Status",
                                          "Bridging::DhcpSnooping::Status", "r" )
   dhcp6SnoopingHwStatusDir = LazyMount.mount(
         entityManager, "bridging/dhcpsnooping/hardware/dhcp6Status",
         "Tac::Dir", "ri" )
   platformHardwareSliceDir = LazyMount.mount( entityManager,
                                               "platform/hardware/slice",
                                               "Hardware::PlatformSliceDir", "r" )
   aclCpConfig = ConfigMount.mount( entityManager, "acl/cpconfig/cli",
                                    "Acl::Input::CpConfig", "w" )
   aclStatus = LazyMount.mount( entityManager, "acl/status/all",
                                "Acl::Status", "r" )
   aclCheckpoint = LazyMount.mount( entityManager, "acl/checkpoint",
                                    "Acl::CheckpointStatus", "w" )
