# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Ark
from CliModel import Model, Enum, Str, Dict, Int, Float, Bool, List, Submodel
from IntfModels import Interface
from ArnetModel import MacAddress
import Vlan
from DhcpSnoopingCliLib import createDhcpSnoopingCountersTable, \
                               createDhcpSnoopingCountersTableDetail
from TableOutput import createTable, Format

class CircuitIdModel( Model ):
   name = Interface( help="Interface name" )
   valid = Bool( help="Circuit ID is valid" )
   value = Str( help="Circuit ID value" )
   circuitType = Int( help="Circuit ID type", optional=True )

class DhcpSnoopingModelBase( Model ):
   enabledVlans = List( valueType=int, help="VLANs on which snooping is configured" )
   operationalVlans = List( valueType=int, 
                            help="VLANs on which snooping is operational" )

   def render( self ):
      dhcpString = ( "DHCP" if isinstance( self, DhcpSnoopingModel )
                     else "DHCPv6" )
      if self.enabled:
         print( '%s Snooping is enabled' % dhcpString )
      else:
         print( '%s Snooping is disabled' % dhcpString )
         return
      print( '%s Snooping is %soperational' % ( dhcpString, '' if self.operational
                                                            else 'not ' ) )
      print( '%s Snooping is configured on following VLANs:' % dhcpString )
      outputStr = Vlan.vlanSetToCanonicalString( sorted( self.enabledVlans ) )
      print( '  %s' % ( outputStr if outputStr else 'None' ) )
      outputStr = \
         Vlan.vlanSetToCanonicalString( sorted( self.operationalVlans ) )
      if isinstance( self, DhcpSnoopingModel ):
         if self.bridgingEnabled:
            print( '%s Snooping bridging is operational on following VLANs:' % \
                   dhcpString )
         else:
            print( '%s Snooping is operational on following VLANs:' % dhcpString )
         print( '  %s' % ( outputStr if outputStr else 'None' ) )
      else:
         print( '%s Snooping is operational on following VLANs:' % dhcpString )
         print( '  %s' % ( outputStr if outputStr else 'None' ) )

class DhcpSnoopingModel( DhcpSnoopingModelBase ):
   enabled = Bool( help="DHCP Snooping is enabled" )
   operational = Bool( help="DHCP Snooping is operational" )
   option82Enabled = Bool( help="Insertion of Option-82 is enabled" )
   circuitIdEnabled = Bool( help="Circuit ID is enabled", optional=True )
   circuitIdType = Int( help="Circuit-id default sub-option type", optional=True )
   circuitIdFormat = Enum( values=[ '%p:%v', '%h:%p' ],
                           help="Format for circuit-id", optional=True )
   circuitIds = Dict( keyType=Interface, valueType=CircuitIdModel,
                      help="Circuit IDs", optional=True )
   bridgeMac = MacAddress( help="Remote MAC address", optional=True )
   bridgingEnabled = Bool( help="DHCP Snooping with bridging enabled",
                           default=False )

   def render( self ):
      super().render()
      if self.option82Enabled:
         print( 'Insertion of Option-82 is enabled' )
         if self.circuitIdEnabled:
            print( '  Circuit-id default sub-option Type: %d' %
                   self.circuitIdType )
         else:
            print( '  Circuit-id default sub-option Type: none' )
         if self.circuitIdFormat == '%p:%v':
            print( '  Circuit-id default format: Interface name:VLAN ID' )
         elif self.circuitIdFormat == '%h:%p':
            print( '  Circuit-id default format: Hostname:Interface name' )

         for intfName in self.circuitIds:
            userConfigIntf = self.circuitIds[ intfName ]
            if userConfigIntf.valid:
               print( '  Circuit-id %s: %s (Type %s)' % ( intfName,
                      userConfigIntf.value,
                      str( userConfigIntf.circuitType ) ) )
            else:
               print( '  Circuit-id %s: %s' % ( intfName,
                                                userConfigIntf.value ) )
         print( '  Remote-id: %s (Switch MAC)' % self.bridgeMac )
      else:
         print( 'Insertion of Option-82 is disabled' )

class Dhcp6SnoopingModel( DhcpSnoopingModelBase ):
   enabled = Bool( help="DHCPv6 Snooping is enabled" )
   operational = Bool( help="DHCPv6 Snooping is operational" )
   remoteIdOption = Bool( help="Insertion of Option-37 is enabled" )

   def render( self ):
      super().render()
      if self.remoteIdOption:
         print( 'Insertion of Option-37 is enabled' )
      else:
         print( 'Insertion of Option-37 is disabled' )

class VlanList( Model ):
   vlans = List( valueType=int, help="VLANs in slice" )

class DhcpSnoopingHardwareModelBase( Model ):
   vlans = List( valueType=int,
                 help="VLANs on which snooping is enabled" )
   vlansPerSlice = Dict( valueType=VlanList, 
                         keyType=str, help="VLANs enabled per slice" )

   def render( self ):
      dhcpString = ( "DHCP" if isinstance( self, DhcpSnoopingHardwareModel )
                     else "DHCPv6" )
      print( "%s Snooping is %s" % ( 
         dhcpString, "enabled" if self.enabled else "disabled" ) )

      if not self.enabled:
         return

      print( '%s Snooping is enabled on following VLANs:' % dhcpString )
      outputStr = Vlan.vlanSetToCanonicalString( sorted( self.vlans ) )
      print( '    %s' % ( outputStr if outputStr else 'None' ) )

      if self.vlansPerSlice:
         print( '    VLANs enabled per Slice' )
         for sliceId in self.vlansPerSlice:
            print( '        Slice: ', sliceId )
            outputStr = Vlan.vlanSetToCanonicalString( sorted(
               self.vlansPerSlice[ sliceId ].vlans ) )
            print( '       %s' % ( outputStr if outputStr else 'None' ) )

class DhcpSnoopingHardwareModel( DhcpSnoopingHardwareModelBase ):
   enabled = Bool( help="DHCP Snooping is enabled" )

class Dhcp6SnoopingHardwareModel( DhcpSnoopingHardwareModelBase ):
   enabled = Bool( help="DHCPv6 Snooping is enabled" )

class IntfCounters( Model ):
   requestsReceived = Int( help="Requests received" )
   requestsForwarded = Int( help="Requests forwarded" )
   requestsDropped = Int( help="Requests dropped" )
   repliesReceived = Int( help="Replies received" )
   repliesForwarded = Int( help="Replies forwarded" )
   repliesDropped = Int( help="Replies dropped" )

class VlanCounters( Model ):
   vlan = Int( help="VLAN ID" )
   requestsReceived = Int( help="Requests received" )
   requestsForwarded = Int( help="Requests forwarded" )
   requestsDropped = Int( help="Requests dropped" )
   repliesReceived = Int( help="Replies received" )
   repliesForwarded = Int( help="Replies forwarded" )
   repliesDropped = Int( help="Replies dropped" )
   resetTime = Float( help="Last time counters were reset" )
   interfaces = Dict( keyType=Interface, valueType=IntfCounters,
                      help="Vlan specific L2 interface counters" )

class DhcpSnoopingCounterModelBase( Model ):
   vlanCounters = Dict( valueType=VlanCounters,
                        keyType=int, help="Per-VLAN counter information" )
   detail_ = Bool( help="Counter values in detail", default=False ) 

   def render( self ):
      dhcpString = ( "DHCP" if isinstance( self, DhcpSnoopingCounterModel ) 
                     else "DHCPv6" )
      if not self.enabled:
         print( "%s Snooping is disabled" % dhcpString )
         return
      
      if self.detail_:
         outputHeader = [ 'Interface',
                          ( 'Dhcp Request Pkts', 'b', ( 'Rcvd', 'Fwdd', 'Drop' ) ),
                          ( 'Dhcp Reply Pkts', 'b', ( 'Rcvd', 'Fwdd', 'Drop' ) ),
                          'Last Cleared' ]
         for vlan in sorted( self.vlanCounters.keys() ):
            print( f"\nVLAN {vlan}" )
            outputTable = createDhcpSnoopingCountersTableDetail( outputHeader,
                                                         self.vlanCounters[ vlan ] )
            print( outputTable.output() )
      else:
         outputHeader = [ 'Vlan',
                          ( 'Dhcp Request Pkts', 'b', ( 'Rcvd', 'Fwdd', 'Drop' ) ),
                          ( 'Dhcp Reply Pkts', 'b', ( 'Rcvd', 'Fwdd', 'Drop' ) ),
                          'Last Cleared' ]
         outputTable = createDhcpSnoopingCountersTable( outputHeader, self )
         print( f'\n{outputTable.output()}' )

class DhcpSnoopingCounterModel( DhcpSnoopingCounterModelBase ):
   enabled = Bool( help="DHCP Snooping is enabled" )

class Dhcp6SnoopingCounterModel( DhcpSnoopingCounterModelBase ):
   enabled = Bool( help="DHCPv6 Snooping is enabled" )

class SnoopingRelayCounters( Model ):
   snoopingToRelay = Int( help="Snooping to relay counter" )
   relayToSnooping = Int( help="Relay to snooping counter" )

class DhcpSnoopingDebugCounterModelBase( Model ):
   receivedCounters = Submodel( valueType=SnoopingRelayCounters,
                                help="Recieved", optional=True )
   forwardedCounters = Submodel( valueType=SnoopingRelayCounters,
                                 help="Forwarded", optional=True )
   vlanIdErrCounters = Submodel( valueType=SnoopingRelayCounters,
                                 help="Dropped - Invalid VlanId", optional=True )
   parseErrCounters = Submodel( valueType=SnoopingRelayCounters,
                                help="Dropped - Parse error", optional=True )
   dhcpOpErrCounters = Submodel( valueType=SnoopingRelayCounters,
                                 help="Dropped - Invalid Dhcp Optype",
                                 optional=True )
   infoOptErrCounters = Submodel( valueType=SnoopingRelayCounters,
                                  help="Dropped - Invalid Info Option",
                                  optional=True )
   remoteIdErrCounters = Submodel( valueType=SnoopingRelayCounters,
                                   help="Dropped - Invalid Remote-ID Option",
                                   optional=True )
   disabledErrCounters = Submodel( valueType=SnoopingRelayCounters,
                                   help="Dropped - Snooping disabled",
                                   optional=True )
   resetTime = Float( help="Last time counters were reset", optional=True )

   def render( self ):
      dhcpString = ( "DHCP" if isinstance( self, DhcpSnoopingDebugCounterModel )
                            else "DHCPv6" )
      if not self.enabled:
         print( '%s Snooping is disabled' % dhcpString )
         return

      outputHeader = [ 'Counter', 'Requests', 'Responses' ]
      output = []
      output.append( [ 'Received', self.receivedCounters.snoopingToRelay,
                       self.receivedCounters.relayToSnooping ] )
      output.append( [ 'Forwarded', self.forwardedCounters.snoopingToRelay,
                       self.forwardedCounters.relayToSnooping ] )
      output.append( [ 'Dropped - Invalid VlanId',
                       self.vlanIdErrCounters.snoopingToRelay,
                       self.vlanIdErrCounters.relayToSnooping ] )
      output.append( [ 'Dropped - Parse error',
                       self.parseErrCounters.snoopingToRelay,
                       self.parseErrCounters.relayToSnooping ] )
      output.append( [ 'Dropped - Invalid Dhcp Optype',
                       self.dhcpOpErrCounters.snoopingToRelay,
                       self.dhcpOpErrCounters.relayToSnooping ] )
      if self.infoOptErrCounters:
         output.append( [ 'Dropped - Invalid Info Option',
                          self.infoOptErrCounters.snoopingToRelay,
                          self.infoOptErrCounters.relayToSnooping ] )
      elif self.remoteIdErrCounters:
         output.append( [ 'Dropped - Invalid Remote-ID Option',
                          self.remoteIdErrCounters.snoopingToRelay,
                          self.remoteIdErrCounters.relayToSnooping ] )
      output.append( [ 'Dropped - Snooping disabled',
                       self.disabledErrCounters.snoopingToRelay,
                       self.disabledErrCounters.relayToSnooping ] )

      outputTable = createTable( outputHeader )
      for row in output:
         outputTable.newRow( *row )
      f1 = Format( justify='left' )
      f2 = Format( justify='right' )
      f1.padLimitIs( True )
      f2.padLimitIs( True )
      outputTable.formatColumns( f1, f2, f2 )
      print( f'\n{outputTable.output()}' )
      print( 'Last Cleared: ', Ark.utcTimeRelativeToNowStr( self.resetTime ) )

class DhcpSnoopingDebugCounterModel( DhcpSnoopingDebugCounterModelBase ):
   enabled = Bool( help="DHCP Snooping is enabled" )

class Dhcp6SnoopingDebugCounterModel( DhcpSnoopingDebugCounterModelBase ):
   enabled = Bool( help="DHCPv6 Snooping is enabled" )

