# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliDynamicSymbol
import CliGlobal
from CliPlugin.DpllCommon import (
   roleMap,
   stateMap,
   updatePllResetConfig,
)
import LazyMount

gv = CliGlobal.CliGlobal(
   allPllConfig=None,
   allPllAgentstatus=None,
   resetConfig=None,
)

def getDpllModels():
   return CliDynamicSymbol.loadDynamicPlugin( "DpllCliModel" )

def showHardwarePll( mode, args ):
   """
   handler for show hardware pll [ module MODULE ]
   args:
      MODULE is an optional linecard to filter with
   """
   slotName = args.get( "MODULE" )
   clocks = {}
   for pllModel in gv.allPllConfig:
      thisPllTypeConfig = gv.allPllConfig.entity[ pllModel ]
      thisPllTypeStatus = gv.allPllAgentstatus.entity[ pllModel ]
      for pll in thisPllTypeConfig.pllConfig.values():
         thisSlotName = pll.name.split( "/" )[ 0 ]
         if ( slotName is not None and
               slotName.tag + slotName.label != thisSlotName ):
            continue
         pllStatus = thisPllTypeStatus.pll.get( pll.name, None )
         if pllStatus and pllStatus.genId != pll.genId:
            # status is stale but config exists
            # display "pllStateUnknown" as state
            pllStatus = None
         pllm = getDpllModels().ClockModel( model=f"{pll.mfgName} {pll.modelName}",
                              description=pll.description, role=roleMap[ pll.role ],
                              state=stateMap[ pllStatus.state if pllStatus
                                             else "pllStateUnknown" ] )
         clocks[ pll.name ] = pllm
   return getDpllModels().PlatformModel( clocks=clocks )

def resetHardwarePll( mode, args ):
   """
   handler for reset hardware pll CLOCK_NAME
   args:
      CLOCK_NAME is the pll name of the clock to reset
   """
   clockName = args[ "CLOCK_NAME" ]
   updatePllResetConfig( gv.allPllConfig, gv.resetConfig )
   for pllModel in gv.resetConfig:
      thisPllType = gv.resetConfig.entity[ pllModel ]
      if clockName in thisPllType.pll:
         thisPllType.pll[ clockName ].resetRequests += 1
         return
   mode.addError( "PLL not found: %s" % clockName )

def Plugin( entityManager ):
   gv.allPllConfig = LazyMount.mount( entityManager,
      "hardware/pllClockMux/config",
      "Tac::Dir",
      "ri" )
   gv.allPllAgentstatus = LazyMount.mount( entityManager,
      "hardware/pllClockMux/status",
      "Tac::Dir",
      "ri" )
   gv.resetConfig = LazyMount.mount( entityManager,
      "hardware/pllClockMux/resetConfig",
      "Tac::Dir",
      "w" )
   # Note: I"m relying on FruCli initialising itself, so calling prepareSlotRule
   # here is in fact redundant/harmful
   # FruCli.prepareSlotRule( entityManager )
