#!/usr/bin/env arista-python
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Arnet import IpGenAddr
import CliCommand
from CliDynamicSymbol import CliDynamicPlugin
from CliPlugin import DynamicPrefixListCli as Globals
from CliPlugin.IraServiceCli import getEffectiveProtocolModel
from CliPlugin.VrfCli import (
      VrfExecCmdDec,
      generateVrfCliModel,
      getVrfNames,
)
from IpLibTypes import ProtocolAgentModelType as ProtoAgentModel
import itertools
from RouteMapLib import getErrorMessageCannotDeleteBecauseUsedBy
import Tac
import Tracing

DynamicPrefixListModel = CliDynamicPlugin( "DynamicPrefixListModel" )

traceHandle = Tracing.Handle( 'DynamicPrefixList' )
t5 = traceHandle.trace5 # Info

def routeMapsMatchingOnDynPrefixList( dynamicPrefixListName ):
   """ @brief Given a dynamic prefix-list name, it returns a set of all route-maps
              in the running-config that "use" (i.e., in a match clause)
              that dynamic prefix-list.
   """
   RouteMapMatchOption = Tac.Type( 'Routing::RouteMap::MatchOption' )
   routeMapsDep = set()
   for mapName, routeMap in Globals.routeMapConfig.routeMap.items():
      for mapSeqno, mapEntry in routeMap.mapEntry.items():
         matchRule = mapEntry.matchRule.get(
            RouteMapMatchOption.matchIpAddrDynamicPrefixList,
            mapEntry.matchRule.get(
               RouteMapMatchOption.matchIpv6AddrDynamicPrefixList ) )
         if matchRule:
            if matchRule.strValue == dynamicPrefixListName:
               routeMapsDep.add( ( mapName, mapSeqno ) )
   return routeMapsDep

def blockDynamiPrefixListDeleteIfInUse( mode, dynPrefixListName ):
   """ @brief Checks if a 'dynamic prefix-list' can be safely deleted without
              breaking any dependency with a route-map.
              Moreover, it prints an error message on the mode (passed as argument)
              in case of dependency.
       @return 'True' if the dynamic prefix-list should not be deleted.
               'False' in the nominal case (no errors).
   """
   if Globals.routeMapConfig.routeMapPolicyReferenceUnconfiguredError:
      routeMapsDep = routeMapsMatchingOnDynPrefixList( dynPrefixListName )
      if routeMapsDep:
         mode.addError( getErrorMessageCannotDeleteBecauseUsedBy(
            'dynamic prefix-list',
            dynPrefixListName,
            routeMapsDep ) )
         return True
   return False

def handlerDynPfxSessionBgpConfig( mode, args ):
   name = args[ 'NAME' ]
   if Globals.getDynPfxList( name ):
      if mode.session.isInteractive():
         mode.addErrorAndStop(
               # pylint: disable-next=consider-using-f-string
               'BGP session dynamic prefix list %s is '
               'defined as a dynamic prefix list' % name )
      mode.addWarning(
               # pylint: disable-next=consider-using-f-string
               'BGP session dynamic prefix list %s was '
               'previously defined as a dynamic prefix list' % name )

   childMode = mode.childMode( Globals.DynPfxBgpSessionConfigMode,
                               sessionBgpDplList=name )
   mode.session_.gotoChildMode( childMode )

def noOrDefaultHandlerDynPfxSessionBgpConfig( mode, args ):
   name = args.get( 'NAME' )
   t5( 'Cleaning config for dynamic prefix-list session bgp', name )
   if name is not None:
      Globals.deleteSessionBgp( name )

def handlerDynamicPrefixListConfig( mode, args ):
   name = args[ 'NAME' ]
   if Globals.getSessionBgp( name ):
      if mode.session.isInteractive():
         mode.addErrorAndStop(
               'Dynamic prefix list %s is ' # pylint: disable=consider-using-f-string
               'defined as a BGP session dynamic prefix list' % name )
      mode.addWarning(
            # pylint: disable-next=consider-using-f-string
            'Dynamic prefix list %s was previously '
            'defined as a BGP session dynamic prefix list' % name )

   childMode = mode.childMode( Globals.DynamicPrefixListConfigMode, dynPfx=name )
   mode.session_.gotoChildMode( childMode )

def noOrDefaultHandlerDynamicPrefixListConfig( mode, args ):
   name = args[ 'NAME' ]
   t5( 'Cleaning config for dynamic prefix-list name', name )
   if blockDynamiPrefixListDeleteIfInUse( mode, name ):
      return
   Globals.deleteDynPfxList( name )

def handlerDynamicPrefixListConfigModeMatchMap( mode, args ):
   name = args[ 'MAPNAME' ]
   t5( 'Setting MatchMap name', name )
   mode.dynPfxEntry.matchMap = name

def noOrDefaultHandlerDynamicPrefixListConfigModeMatchMap( mode, args ):
   name = args[ 'MAPNAME' ]
   if mode.dynPfxEntry.matchMap == name:
      t5( 'Removing MatchMap name', name )
      mode.dynPfxEntry.matchMap = ''

def handlerDynPfxSessionBgpConfigModeMatchSession( mode, args ):
   address = args.get( 'ADDRESS' )
   mode.sessionBgpDplEntry.sessionBgpAddr[ IpGenAddr( str( address ) ) ] = True
   mode.sessionBgpDplEntry.originate = True

def noOrDefaultHandlerDynPfxSessionBgpConfigModeMatchSession( mode, args ):
   address = args[ 'ADDRESS' ]
   del mode.sessionBgpDplEntry.sessionBgpAddr[ address ]

   # Check if the set is empty: if yes, set originate to False
   if not mode.sessionBgpDplEntry.sessionBgpAddr:
      mode.sessionBgpDplEntry.originate = False

def handlerDynamicPrefixListConfigModePrefixList( mode, args ):
   name = args[ 'LISTNAME' ]
   if args.get( 'ipv4' ):
      t5( 'Setting IPv4 PrefixList name', name )
      mode.dynPfxEntry.ipv4PrefixList = name
   elif args.get( 'ipv6' ):
      t5( 'Setting IPv6 PrefixList name', name )
      mode.dynPfxEntry.ipv6PrefixList = name

def noOrDefaultHandlerDynamicPrefixListConfigModePrefixList( mode, args ):
   name = args[ 'LISTNAME' ]
   if args.get( 'ipv4' ):
      if mode.dynPfxEntry.ipv4PrefixList == name:
         t5( 'Removing IPv4 PrefixList', name )
         mode.dynPfxEntry.ipv4PrefixList = ''
   elif args.get( 'ipv6' ):
      if mode.dynPfxEntry.ipv6PrefixList == name:
         t5( 'Removing IPv6 PrefixList', name )
         mode.dynPfxEntry.ipv6PrefixList = ''

def handlerDynamicPrefixListConfigModeAbort( mode, args ):
   t5( 'Aborting changes' )
   mode.abort_ = True
   mode.session_.gotoParentMode()

def handlerSessionBgpDplConfigModeAbort( mode, args ):
   t5( 'Aborting changes' )
   mode.abort_ = True
   mode.session_.gotoParentMode()

dynamicPrefixListVrfModel = \
                  generateVrfCliModel( DynamicPrefixListModel.DynamicPrefixLists,
                                       "Dynamic prefix list information for VRFs" )

@VrfExecCmdDec( getVrfsFunc=getVrfNames, cliModel=dynamicPrefixListVrfModel )
def showPrefixList( mode, vrfName=None, prefixListName=None ):

   def dynamicPrefixListFromConfig():
      for dynamicPrefixListName in Globals.dynPfxListConfigDir.dynamicPrefixList:
         if not prefixListName or prefixListName == dynamicPrefixListName:
            dynamicPrefixListInst = DynamicPrefixListModel.DynamicPrefixListModel()
            dynamicPrefixListInst.state = 'notApplicableToRoutingInstance'
            psuedoData = { 'name' : dynamicPrefixListName }
            dynamicPrefixListInst.processData( psuedoData )
            yield dynamicPrefixListName, dynamicPrefixListInst

   if getEffectiveProtocolModel( mode ) == ProtoAgentModel.multiAgent:
      showDynPfxListCb = Globals.routeMapShowCmdDict[ 'doArBgpShowDynPfxList' ]
   else:
      showDynPfxListCb = Globals.routeMapShowCmdDict[ 'showDynPfxListCb' ]

   if mode.session_.outputFormat_ == "text":
      DynamicPrefixListModel.printDynamicPrefixListLegendInformation()

   args = {}
   args[ 'vrfName' ] = vrfName
   if prefixListName:
      args[ 'prefix-list' ] = prefixListName

   try:
      result = showDynPfxListCb( mode, DynamicPrefixListModel.DynamicPrefixLists,
                                 args )
      return result
   except Globals.routeMapShowCmdDict.get( 'getEmptyResponseExceptionCb' )():
      inst = DynamicPrefixListModel.DynamicPrefixLists()
      if prefixListName:
         inst.dynamicPrefixLists = itertools.chain( inst.dynamicPrefixLists,
                                                    dynamicPrefixListFromConfig() )
      return inst

def handlerShowDynamicPrefixListCmd( mode, args ):
   prefixListName = args.get( 'LISTNAME' )
   vrfName = args.get( 'VRFNAME' )
   return showPrefixList( mode, vrfName=vrfName, prefixListName=prefixListName )

def handlerMatchIpAddrDynamicPrefixListCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   mode.setMatchValue( no, 'matchIpAddrDynamicPrefixList', args[ 'LISTNAME' ] )

def handlerMatchIpv6AddrDynamicPrefixListCmd( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   mode.setMatchValue( no, 'matchIpv6AddrDynamicPrefixList', args[ 'LISTNAME' ] )

