#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from BgpLib import getVpwsName
from CliDynamicSymbol import CliDynamicPlugin
from CliPlugin.ArBgpCli import ArBgpAsyncCliCommand
from CliPlugin.BgpMacVrfConfigCli import (
   getBundleName,
   getStandAloneName,
)
from CliPlugin.RoutingBgpCli import (
   configForVrf,
   RouterBgpBaseAfEvpnMode,
   BgpCmdBaseClass,
)
from CliPlugin.RoutingBgpShowCli import ArBgpShowOutput
from CliPlugin import VxlanModel
import CliPlugin.EvpnCli as Globals
from CliPlugin.EvpnCli import (
   EthernetSegmentConfigMode,
   EvpnMulticastAfIpMode,
   EvpnMulticastAfIpv6Mode,
   EvpnMulticastConfigMode,
)
from CliToken.RoutingBgpConstants import evpnNexthopSelfSupportNlris
import ConfigMount
import LazyMount
import Tac
from Toggles import EvpnToggleLib
from Vlan import computeVlanRangeSet

EvpnCliModels = CliDynamicPlugin( "EvpnCliModels" )

evpnStatus = None
mcastEvpnConfig = None

checkEsid = Tac.Value( 'Evpn::Esid' )
defaultRT = Tac.Value( 'Arnet::EthAddr' )

# Set attributes in config to default values
def setConfigDefault( config, tacName ):
   attrNames = Tac.Type( tacName ).attributes
   for attrName in attrNames:
      if attrName.endswith( 'Default' ) or attrName.endswith( 'Present' ):
         continue

      # Every attribute _should_ have a default value defined, but for those that
      # do not, the reset must be hard-coded by caller.
      attrDefault = attrName + 'Default'
      if hasattr( config, attrDefault ):
         setattr( config, attrName, getattr( config, attrDefault ) )

# Set attributes for iEsi to default
def setIEsiDefault( iEsi ):
   intfId = Tac.Value( "Arnet::IntfId", "" )
   defaultEthernetSegmentCliConfig = Tac.newInstance(
      "Evpn::EthernetSegmentCliConfig",
      intfId )
   iEsi.clone( defaultEthernetSegmentCliConfig )

# Clean-up hook for EVPN mount paths if EVPN address family or BGP config is removed
def deleteEvpnConfigHook( af='evpn' ):
   if af != 'evpn':
      return
   # Reset attributes in EvpnConfig to default values
   setConfigDefault( Globals.evpnConfig, 'Routing::Bgp::MacPlugin::EvpnConfig' )
   Globals.interconnectEsCliConfigDir.remoteDomainIEsConfig = None
   Globals.interconnectEsCliConfigDir.localDomainIEsConfig = None
   Globals.interconnectEsCliConfigDir.allDomainIEsConfig = None

mlagShutdownHostFlapWarning = ( 'EVPN/VXLAN are currently configured. Shutting down '
                                'MLAG in this state may lead to layer 2 loops which '
                                'eventually cause MAC blacklisting.' )

mlagReconfigureHostFlapWarning = ( 'EVPN/VXLAN are currently configured. '
                                   'Reconfiguring certain MLAG parameters in this '
                                   'state will cause MLAG-renegotiation and may '
                                   'lead to transient layer 2 loops which '
                                   'eventually cause MAC blacklisting' )

def canShutdownMlag( isPrimaryOrSecondaryMlag ):
   msg = None
   if isPrimaryOrSecondaryMlag and evpnStatus.vxlanEnabled:
      msg = mlagShutdownHostFlapWarning
   return msg

def canReconfigureMlag( isPrimaryOrSecondaryMlag ):
   msg = None
   if isPrimaryOrSecondaryMlag and evpnStatus.vxlanEnabled:
      msg = mlagReconfigureHostFlapWarning
   return msg

# Check whether "esi" can be applied. Currently, validate Type 0 and
# non-duplicate ESI.
def checkEsidAssignment( allEsConfig, esi, intf ):
   if esi[ : 2 ] != "00":
      return ( False, 'only Type 0 ESID is supported' )
   if esi in ( checkEsid.zero, checkEsid.max ):
      return ( False, 'reserved ESID entered' )
   for intfName, cfg in allEsConfig.intfConfig.items():
      if cfg.esid.value == esi and intfName != intf:
         # pylint: disable=bad-option-value,consider-using-f-string
         return ( False, 'ESID is already assigned to interface %s' % intfName )
   return ( True, '' )

# Check if there are duplicate "rt"
def checkRtAssignment( allEsConfig, rt, intf ):
   for intfName, cfg in allEsConfig.intfConfig.items():
      if cfg.esImportRouteTarget == rt and intfName != intf:
         # pylint: disable=bad-option-value,consider-using-f-string
         return ( True, 'Import RT is already assigned to interface %s' % intfName )
   return ( False, '' )

def gotoEthernetSegmentConfigMode( mode, args ):
   if isinstance( mode, RouterBgpBaseAfEvpnMode ):
      if 'domain' not in args:
         return
      intf = ''
      if 'all' in args:
         domain = 'domain all'
      elif 'remote' in args:
         domain = 'domain remote'
      elif 'local' in args:
         domain = 'domain local'
      else:
         domain = ''
   else:
      intf = mode.intf.name
      domain = None
   param = ( intf, domain )
   childMode = mode.childMode( EthernetSegmentConfigMode, param=param )
   # Do not enter evpn ethernet-segment submode when configured via interface
   # range config mode.
   if mode.session.mode_ != childMode.parent_:
      return
   mode.session_.gotoChildMode( childMode )

def deleteEthernetSegmentConfigMode( mode, args ):
   if isinstance( mode, RouterBgpBaseAfEvpnMode ):
      if 'all' in args:
         Globals.interconnectEsCliConfigDir.allDomainIEsConfig = None
      elif 'remote' in args:
         Globals.interconnectEsCliConfigDir.remoteDomainIEsConfig = None
      elif 'local' in args:
         Globals.interconnectEsCliConfigDir.localDomainIEsConfig = None
      else:
         # Delete all interconnect ethernet segments
         Globals.interconnectEsCliConfigDir.allDomainIEsConfig = None
         Globals.interconnectEsCliConfigDir.remoteDomainIEsConfig = None
         Globals.interconnectEsCliConfigDir.localDomainIEsConfig = None
   else:
      intf = mode.intf.name
      del Globals.ethernetSegmentConfigDir.intfConfig[ intf ]

def EvpnEthernetSegmentCmd_handler( mode, args ):
   gotoEthernetSegmentConfigMode( mode, args )

def EvpnEthernetSegmentCmd_noOrDefaultHandler( mode, args ):
   deleteEthernetSegmentConfigMode( mode, args )

def EvpnEthernetSegmentDomainCmd_handler( mode, args ):
   gotoEthernetSegmentConfigMode( mode, args )

def EvpnEthernetSegmentDomainCmd_noOrDefaultHandler( mode, args ):
   deleteEthernetSegmentConfigMode( mode, args )

def IdentifierEsiCmd_handler( mode, args ):
   esi = None
   if EvpnToggleLib.toggleEvpnType1EsiEnabled() and 'lacp' in args:
      newEsiType = Tac.Value( "Evpn::EsiType", 'type1' )
   else:
      newEsiType = Tac.Value( "Evpn::EsiType", 'type0' )
      esi = args[ 'ESI' ]
   config = mode.config()
   # If identifier value is automatically generated we do not perform any checks
   # here to verify if ESI is valid but we do so within the EVPN agent.
   if not EvpnToggleLib.toggleEvpnType1EsiEnabled() or 'auto' not in args:
      ( allow, reason ) = \
         checkEsidAssignment( Globals.ethernetSegmentConfigDir, esi, mode.intf_ )
      if not allow:
         mode.session_.addError(
               f'Unable to configure identifier because {reason}' )
         return
   config.esiType = newEsiType
   newEsi = Tac.Value( "Evpn::EsiOrNone" )
   if esi:
      newEsi.value = esi
   config.esid = newEsi

def IdentifierEsiCmd_noOrDefaultHandler( mode, args ):
   config = mode.config()
   config.esid = Tac.Value( 'Evpn::EsiOrNone' )
   config.esiType = Tac.Value( "Evpn::EsiType" )

def checkConflictingEthernetSegmentName( esName, esIntfName, domain ):
   # Allow any name to be set for domain == local or remote.
   # Also, allow any name for ( domain == all and interface name is not 'Vxlan1'.
   if domain == "domain local" or domain == "domain remote" or \
         ( domain == "domain all" and esIntfName != "Vxlan1" ):
      return False

   # Always allow esName == esIntfName
   if esName == esIntfName:
      return False

   # Check name conflict in ethernetSegmentConfigDir.
   for intfId, intfConfig in Globals.ethernetSegmentConfigDir.intfConfig.items():
      if esIntfName != intfId and ( esName == intfConfig.esName or
            esName.startswith( intfId ) ):
         return True

   # Special treatment for "Vxlan1": Unless the current interface is Vxlan1,
   # we disallow 'Vxlan1' being used as the ethernet segment name.
   if esIntfName != 'Vxlan1' and esName == 'Vxlan1':
      return True

   # Check name conflict against all interface names.
   # Name conflicts against interface names are done with startswith().
   # This is because subinterfaces are created dynamically so a subinterface
   # with name X might appear after some other interface already set its
   # ethernet segment name to X.  Since the prefix of a subinterface name
   # corresponds to its parent interface name, we match the prefix of the
   # ethernet segment name with all interface names.
   for intfId in Globals.allIntfStatusDir.intfStatus:
      if esIntfName != intfId and esName.startswith( intfId ):
         return True

   return False

def EthSegmentNameCmd_handler( mode, args ):
   esName = args[ 'ETH_SEGMENT_NAME' ]
   # esName should not be already used as an ethernet segment name
   # or as an interface name.
   if checkConflictingEthernetSegmentName( esName, mode.intf_, mode.domain ):
      mode.session_.addError(
            f"Ethernet segment name {esName} conflicts with other "
            "ethernet segment names or interface names" )
      return

   config = mode.config()
   config.esName = esName

def EthSegmentNameCmd_noOrDefaultHandler( mode, args ):
   config = mode.config()
   config.esName = config.intfId

def RedundancyCmd_handler( mode, args ):
   config = mode.config()
   if 'all-active' in args:
      config.multihomeMode = 'multihomeModeAllActive'
   else:
      config.multihomeMode = 'multihomeModeSingleActive'

def RedundancyCmd_noOrDefaultHandler( mode, args ):
   config = mode.config()
   config.multihomeMode = config.multihomeModeDefault

def ElectionAlgorithmCmd_handler( mode, args ):
   config = mode.config()
   if 'hrw' in args:
      config.dfElectionAlgorithm = 'dfaHrw'
   elif 'preference' in args:
      config.dfElectionAlgorithm = 'dfaPreference'
   else:
      config.dfElectionAlgorithm = 'dfaDefault'
   config.dfElectionPreference = args.get( 'PREFERENCE', 0xffffffff )
   config.dfElectionDontPreempt = 'dont-preempt' in args

def ElectionAlgorithmCmd_noOrDefaultHandler( mode, args ):
   ElectionAlgorithmCmd_handler( mode, args )

def HoldTimeCmd_handler( mode, args ):
   config = mode.config()
   config.holdTime = Tac.Value( 'Evpn::DFElectionHoldTime', args[ 'HOLD_TIME' ] )
   delayTime = 0
   delayTime = args.get( 'DELAY_TIME', 0 ) / 1000.0
   config.dfElectionDelayTime = delayTime

def HoldTimeCmd_noOrDefaultHandler( mode, args ):
   config = mode.config()
   config.holdTime = Globals.electionHoldTimeVal.defaultHoldTime
   config.dfElectionDelayTime = 0

def ReachabilityRequiredCmd_handler( mode, args ):
   config = mode.config()
   config.dfElectionCandidateReachableOnly = True

def ReachabilityRequiredCmd_noOrDefaultHandler( mode, args ):
   config = mode.config()
   config.dfElectionCandidateReachableOnly = False

def EncapFloodFilterTimeCmd_handler( mode, args ):
   config = mode.config()
   time_ms = args[ 'TIME' ] / 1000.0
   config.encapBumFilterHoldTime = time_ms

def EncapFloodFilterTimeCmd_noOrDefaultHandler( mode, args ):
   config = mode.config()
   config.encapBumFilterHoldTime = 0

def SharedEsiLabelIndexCmd_handler( mode, args ):
   indexVal = args[ 'INDEX' ]
   mode.config().sharedEsiLabelIndex = indexVal

def SharedEsiLabelIndexCmd_noOrDefaultHandler( mode, args ):
   mode.config().sharedEsiLabelIndex = 0

def RouteTargetImportCmd_handler( mode, args ):
   rt = args[ 'RT' ]
   config = mode.config()
   ( warn, reason ) = \
      checkRtAssignment( Globals.ethernetSegmentConfigDir, rt, mode.intf_ )
   if warn:
      mode.session_.addWarning( reason )
   config.esImportRouteTarget = rt

def RouteTargetImportCmd_noOrDefaultHandler( mode, args ):
   config = mode.config()
   config.esImportRouteTarget = defaultRT

def gotoEvpnMulticastConfigMode( mode, args ):
   childMode = mode.childMode( EvpnMulticastConfigMode, vrfName=mode.vrfName )
   mode.session_.gotoChildMode( childMode )

def deleteEvpnMulticastConfigMode( mode, args ):
   vrfName = mode.vrfName
   config = configForVrf( vrfName )
   config.afEvpnOismEnabled = False
   config.afEvpnOismExplicitlyDisabled = False
   config.afIpv6EvpnOismEnabled = False
   config.afIpv6EvpnOismExplicitlyDisabled = False
   del Globals.evpnOismConfig.afEvpnOismEnabledVrf[ vrfName ]
   del Globals.evpnOismConfig.afIpv6EvpnOismEnabledVrf[ vrfName ]
   del mcastEvpnConfig.transitVrf[ vrfName ]

def EvpnMulticastModeCmd_handler( mode, args ):
   gotoEvpnMulticastConfigMode( mode, args )

def EvpnMulticastModeCmd_noOrDefaultHandler( mode, args ):
   deleteEvpnMulticastConfigMode( mode, args )

def EvpnMulitcastAfCommand_handleNormal( mode, args ):
   vrfName = mode.vrfName_
   config = configForVrf( vrfName )
   config.evpnOismLegacy = False
   af = args[ 'AF' ]
   if af == 'ipv4':
      newMode = EvpnMulticastAfIpMode
   else:
      if not config.afIpv6EvpnOismExplicitlyDisabled:
         config.afIpv6EvpnOismEnabled = True
         Globals.evpnOismConfig.afIpv6EvpnOismEnabledVrf[ vrfName ] = True
      newMode = EvpnMulticastAfIpv6Mode

   childMode = mode.childMode( newMode, addrFamily=af, vrfName=vrfName )
   mode.session_.gotoChildMode( childMode )

def EvpnMulitcastAfCommand_handleNoOrDefault( mode, args, noOrDefault ):
   vrfName = mode.vrfName_
   config = configForVrf( vrfName )
   config.evpnOismLegacy = False
   af = args[ 'AF' ]
   if af == 'ipv4':
      # default is 'enabled' for v4
      config.afEvpnOismEnabled = True
      Globals.evpnOismConfig.afEvpnOismEnabledVrf[ vrfName ] = True
      config.afEvpnOismExplicitlyDisabled = False
      del mcastEvpnConfig.transitVrf[ vrfName ]
   else:
      config.afIpv6EvpnOismEnabled = False
      del Globals.evpnOismConfig.afIpv6EvpnOismEnabledVrf[ vrfName ]
      config.afIpv6EvpnOismExplicitlyDisabled = False

def EvpnMulitcastAfCommand_handler( mode, args ):
   BgpCmdBaseClass.callHandler(
      EvpnMulitcastAfCommand_handleNormal,
      EvpnMulitcastAfCommand_handleNoOrDefault,
      mode,
      args )

def EvpnMulitcastAfCommand_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass.callNoOrDefaultHandler(
      EvpnMulitcastAfCommand_handleNoOrDefault,
      mode,
      args )

def EvpnGatewayDrElectionAlgorithmCmd_handler( mode, args ):
   vrfName = mode.vrfName_
   config = configForVrf( vrfName )
   config.evpnOismLegacy = False

   if 'modulus' in args:
      config.gatewayDrElectionAlgorithm = 'dfaDefault'
   elif 'hrw' in args:
      config.gatewayDrElectionAlgorithm = 'dfaHrw'
   elif 'preference' in args:
      config.gatewayDrElectionAlgorithm = 'dfaPreference'
      config.gatewayDrElectionPreference = args[ 'PREFERENCE' ]

def EvpnGatewayDrElectionAlgorithmCmd_noOrDefaultHandler( mode, args ):
   vrfName = mode.vrfName_
   config = configForVrf( vrfName )
   config.evpnOismLegacy = False
   config.gatewayDrElectionAlgorithm = 'dfaHrw'

def OismTransitCmd_handler( mode, args ):
   mcastEvpnConfig.transitVrf[ mode.vrfName_ ] = True

def OismTransitCmd_noOrDefaultHandler( mode, args ):
   del mcastEvpnConfig.transitVrf[ mode.vrfName_ ]

def OismDisabledCmd_handler( mode, args ):
   vrfName = mode.vrfName_
   af = mode.addrFamily
   config = configForVrf( vrfName )
   if af == 'ipv4':
      config.afEvpnOismEnabled = False
      del Globals.evpnOismConfig.afEvpnOismEnabledVrf[ vrfName ]
      config.afEvpnOismExplicitlyDisabled = True
   else:
      config.afIpv6EvpnOismEnabled = False
      del Globals.evpnOismConfig.afIpv6EvpnOismEnabledVrf[ vrfName ]
      config.afIpv6EvpnOismExplicitlyDisabled = True

def OismDisabledCmd_noOrDefaultHandler( mode, args ):
   vrfName = mode.vrfName_
   af = mode.addrFamily
   config = configForVrf( vrfName )
   if af == 'ipv4':
      config.afEvpnOismEnabled = True
      Globals.evpnOismConfig.afEvpnOismEnabledVrf[ vrfName ] = True
      config.afEvpnOismExplicitlyDisabled = False
   else:
      config.afIpv6EvpnOismEnabled = True
      Globals.evpnOismConfig.afIpv6EvpnOismEnabledVrf[ vrfName ] = True
      config.afIpv6EvpnOismExplicitlyDisabled = False

def getSbdVrfName( l3VrfName ):
   return Tac.Type( "Routing::Bgp::SbdMacVrfUtil" ).getSbdMacVrfName( l3VrfName )

def EsDisabledCmd_Handler( mode, args ):
   config = mode.config()
   config.disabled = True

def EsDisabledCmd_noOrDefaultHandler( mode, args ):
   config = mode.config()
   config.disabled = False

def EsAliasingCmd_Handler( mode, args ):
   config = mode.config()
   config.aliasingEnabled = Tac.Type( "Ark::TristateBoolean" ).valueSet( True )

def EsAliasingCmd_noOrDefaultHandler( mode, args ):
   config = mode.config()
   config.aliasingEnabled = Tac.Type( "Ark::TristateBoolean" ).valueSet( False )

@ArBgpShowOutput( "doShowBgpEvpnInstance", arBgpModeOnly=True )
def ShowBgpEvpnInstanceCmd_handler( mode, args ):
   macVrfName = None
   ipVrfName = None
   if args.get( "vlan" ):
      macVrfName = getStandAloneName( args.get( "VLANID" ) )
   elif args.get( "vlan-aware-bundle" ):
      macVrfName = getBundleName(
         Globals.MacVrfTypeEnum.macVrfTypeVlan, mode,
         args.get( "BUNDLE" ), validate=False )
   elif args.get( "vpws" ):
      macVrfName = getVpwsName( args.get( "VPWS" ) )
   elif args.get( "sbd" ):
      macVrfName = getSbdVrfName( args.get( "SBD_VRF_NAME" ) )
   elif args.get( "vrf" ):
      ipVrfName = args.get( "VRF" )
   cmd = ArBgpAsyncCliCommand( mode, "show evpn instance" )
   if macVrfName:
      cmd.addParam( "macVrf", macVrfName )
   elif args.get( "vrf" ):
      cmd.addParam( "ipVrf", ipVrfName )
   cmd.run()
   return EvpnCliModels.BgpEvpnInstances

def RouteImportOverlayIndexCmd_handler( mode, args ):
   Globals.evpnConfig.overlayIndexEnabled = True

def RouteImportOverlayIndexCmd_noOrDefaultHandler( mode, args ):
   Globals.evpnConfig.overlayIndexEnabled = False

def UmrInstallCliCommand_handler( mode, args ):
   Globals.evpnConfig.installUmr = True

def UmrInstallCliCommand_noOrDefaultHandler( mode, args ):
   Globals.evpnConfig.installUmr = False

def ReAdvertiseMacIpDisabledCommand_handler( mode, args ):
   Globals.evpnConfig.reAdvertiseMacIpIntoLocalDomain = not args.get( 'DISABLED' )

def ReAdvertiseMacIpDisabledCommand_noOrDefaultHandler( mode, args ):
   Globals.evpnConfig.reAdvertiseMacIpIntoLocalDomain = True

def EvpnRouteExportEsIpMassWithdrawCmd_handler( mode, args ):
   Globals.ethernetSegmentConfigDir.adPerEsExportL3RT = True

def EvpnRouteExportEsIpMassWithdrawCmd_noOrDefaultHandler( mode, args ):
   Globals.ethernetSegmentConfigDir.adPerEsExportL3RT = False

def EvpnRouteImportEsIpMassWithdrawCmd_handler( mode, args ):
   Globals.ethernetSegmentConfigDir.adPerEsImportL3RT = True

def EvpnRouteImportEsIpMassWithdrawCmd_noOrDefaultHandler( mode, args ):
   Globals.ethernetSegmentConfigDir.adPerEsImportL3RT = False

def HostFlapDetectionParamsCmd_handler( mode, args ):
   timeout = args.get( 'SECONDS', Globals.evpnConfig.duplicateTimeoutDefault )
   Globals.evpnConfig.duplicateTimeout = timeout

   threshold = args.get( 'MOVES', Globals.evpnConfig.duplicateThresholdDefault )
   Globals.evpnConfig.duplicateThreshold = threshold

   expiry = args.get( 'TIMEOUT', Globals.evpnConfig.duplicateExpiryDefault )
   Globals.evpnConfig.duplicateExpiry = expiry

def HostFlapDetectionParamsCmd_noHandler( mode, args ):
   Globals.evpnConfig.duplicateTimeout = 0
   Globals.evpnConfig.duplicateThreshold = 0
   Globals.evpnConfig.duplicateExpiry = 0

def HostFlapDetectionParamsCmd_defaultHandler( mode, args ):
   Globals.evpnConfig.duplicateTimeout = \
      Globals.evpnConfig.duplicateTimeoutDefault
   Globals.evpnConfig.duplicateThreshold = \
      Globals.evpnConfig.duplicateThresholdDefault
   Globals.evpnConfig.duplicateExpiry = \
      Globals.evpnConfig.duplicateExpiryDefault

def EvpnVxlanL3UseNhIgpCostCmd_setUseNhIgpCostCommon( mode, vxlanL3UseNhIgpCost ):
   ''' Handler for 'encapsulation vxlan layer-3 set next-hop igp-cost'
   commands '''
   # config currently only supported in default vrf
   config = configForVrf( mode.vrfName )
   config.afEvpnVxlanL3UseNexthopIgpCost = vxlanL3UseNhIgpCost

def EvpnVxlanL3UseNhIgpCostCmd_handleNormal( mode, args ):
   EvpnVxlanL3UseNhIgpCostCmd_setUseNhIgpCostCommon( mode, True )

def EvpnVxlanL3UseNhIgpCostCmd_handleNoOrDefault( mode, args, noOrDefault ):
   EvpnVxlanL3UseNhIgpCostCmd_setUseNhIgpCostCommon( mode, False )

def EvpnVxlanL3UseNhIgpCostCmd_handler( mode, args ):
   BgpCmdBaseClass.callHandler(
      EvpnVxlanL3UseNhIgpCostCmd_handleNormal,
      EvpnVxlanL3UseNhIgpCostCmd_handleNoOrDefault,
      mode,
      args )

def EvpnVxlanL3UseNhIgpCostCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass.callNoOrDefaultHandler(
      EvpnVxlanL3UseNhIgpCostCmd_handleNoOrDefault,
      mode,
      args )

def EvpnL2FecInPlaceUpdateTimeoutCmd_handleNormal( mode, args ):
   Globals.evpnConfig.inPlaceUpdateEnabled = True
   if 'timeout' in args:
      Globals.evpnConfig.remoteRefreshDelay = args[ 'TIMEOUT' ]
   else:
      Globals.evpnConfig.remoteRefreshDelay = \
         Globals.evpnConfig.remoteRefreshDelayDefault

def EvpnL2FecInPlaceUpdateTimeoutCmd_handleNoOrDefault( mode, args, noOrDefault ):
   Globals.evpnConfig.remoteRefreshDelay = \
      Globals.evpnConfig.remoteRefreshDelayDefault
   Globals.evpnConfig.inPlaceUpdateEnabled = \
      Globals.evpnConfig.inPlaceUpdateEnabledDefault

def EvpnL2FecInPlaceUpdateTimeoutCmd_handler( mode, args ):
   BgpCmdBaseClass.callHandler(
      EvpnL2FecInPlaceUpdateTimeoutCmd_handleNormal,
      EvpnL2FecInPlaceUpdateTimeoutCmd_handleNoOrDefault,
      mode,
      args )

def EvpnL2FecInPlaceUpdateTimeoutCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass.callNoOrDefaultHandler(
      EvpnL2FecInPlaceUpdateTimeoutCmd_handleNoOrDefault,
      mode,
      args )

def ShowBgpEvpnHostFlapCmd_handler( mode, args ):
   def convertToUtcTime( timestamp ):
      timeStampInUtcTime = timestamp + Tac.utcNow() - Tac.now()
      return timeStampInUtcTime

   model = EvpnCliModels.BgpEvpnHostFlapEntries()
   model.duplicationBlacklistedMacs = []
   for entry, time in Globals.macDuplicationBlacklist.blacklist.items():
      blacklistedEntry = EvpnCliModels.BgpEvpnHostFlapEntry()
      blacklistedEntry.vlan = entry.vlanId
      blacklistedEntry.macAddr = entry.macaddr
      blacklistedEntry.time = convertToUtcTime( time )
      model.duplicationBlacklistedMacs.append( blacklistedEntry )
   return model

def ClearBgpEvpnHostFlapCmd_handler( mode, args ):
   Globals.evpnConfig.schedClearTime = Tac.now()

def SetEvpnDomainIdCmd_handler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.domainIdEvpn = args[ 'DOMAIN_ID' ]

def SetEvpnDomainIdCmd_noOrDefaultHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   config.domainIdEvpn = config.domainIdEvpnDefault

def SetEvpnInterDomainIdCmd_handler( mode, args ):
   config = configForVrf( mode.vrfName )
   remoteDomain = 'remote' in args
   if remoteDomain:
      config.remoteDomainIdEvpn = args[ 'DOMAIN_ID' ]
   else:
      config.domainIdEvpn = args[ 'DOMAIN_ID' ]

def SetEvpnInterDomainIdCmd_noOrDefaultHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   remoteDomain = 'remote' in args
   if remoteDomain:
      config.remoteDomainIdEvpn = config.remoteDomainIdEvpnDefault
   else:
      config.domainIdEvpn = config.domainIdEvpnDefault

def SetEvpnAttachInterDomainIdCmd_handler( mode, args ):
   config = configForVrf( mode.vrfName )
   remoteDomain = 'remote' in args
   attachDomain = 'attach' in args
   if remoteDomain:
      config.remoteDomainIdEvpn = args[ 'DOMAIN_ID' ]
   elif attachDomain:
      config.attachDomainIdEvpn = args[ 'DOMAIN_ID' ]
   else:
      config.domainIdEvpn = args[ 'DOMAIN_ID' ]

def SetEvpnAttachInterDomainIdCmd_noOrDefaultHandler( mode, args ):
   config = configForVrf( mode.vrfName )
   remoteDomain = 'remote' in args
   attachDomain = 'attach' in args
   if remoteDomain:
      config.remoteDomainIdEvpn = config.remoteDomainIdEvpnDefault
   elif attachDomain:
      config.attachDomainIdEvpn = config.attachDomainIdEvpnDefault
   else:
      config.domainIdEvpn = config.domainIdEvpnDefault

def RouteTypeSmetAdvertiseCmd_handler( mode, args ):
   Globals.evpnConfig.smetAdvertisedByAll = 'all' in args

def RouteTypeSmetAdvertiseCmd_noOrDefaultHandler( mode, args ):
   Globals.evpnConfig.smetAdvertisedByAll = False

def RouteTypeSmetProxyCmd_handler( mode, args ):
   Globals.evpnConfig.smetProxyInterDomain = True

def RouteTypeSmetProxyCmd_noOrDefaultHandler( mode, args ):
   Globals.evpnConfig.smetProxyInterDomain = False

def RouteTypeEsImportAutoRouteTargetCmd_handler( mode, args ):
   Globals.ethernetSegmentConfigDir.esImportAutoRT = True

def RouteTypeEsImportAutoRouteTargetCmd_noOrDefaultHandler( mode, args ):
   Globals.ethernetSegmentConfigDir.esImportAutoRT = False

def SetEvpnNeighborDefaultNextHopSelfCmd_setNeighborDefaultNexthopSelfEvpn(
      mode, routeTypes, interDomain=False ):
   config = configForVrf( mode.vrfName )
   for rtype in routeTypes:
      if rtype in evpnNexthopSelfSupportNlris:
         # pylint: disable=bad-option-value,consider-using-f-string
         attr = 'nexthopSelfEvpnType%d' % rtype
         if interDomain:
            attr += 'InterDomain'
         setattr( config, attr, "isTrue" )

def SetEvpnNeighborDefaultNextHopSelfCmd_noNeighborDefaultNexthopSelfEvpn(
      mode, routeTypes, noOrDefault, interDomain=False ):
   config = configForVrf( mode.vrfName )
   if config:
      for rtype in routeTypes:
         if rtype in evpnNexthopSelfSupportNlris:
            # pylint: disable=bad-option-value,consider-using-f-string
            attr = 'nexthopSelfEvpnType%d' % rtype
            if interDomain:
               attr += 'InterDomain'
            val = getattr( config, attr + 'Default' )
            setattr( config, attr, val )

def SetEvpnNeighborDefaultNextHopSelfCmd_handleNoOrDefault(
      mode, args, noOrDefault ):
   SetEvpnNeighborDefaultNextHopSelfCmd_noNeighborDefaultNexthopSelfEvpn(
      mode,
      args[ 'ROUTE_TYPES' ],
      noOrDefault,
      args.get( 'inter-domain' ) )

def SetEvpnNeighborDefaultNextHopSelfCmd_handleNormal( mode, args ):
   SetEvpnNeighborDefaultNextHopSelfCmd_setNeighborDefaultNexthopSelfEvpn(
      mode,
      args[ 'ROUTE_TYPES' ],
      args.get( 'inter-domain' ) )

def SetEvpnNeighborDefaultNextHopSelfCmd_handler( mode, args ):
   BgpCmdBaseClass.callHandler(
      SetEvpnNeighborDefaultNextHopSelfCmd_handleNormal,
      SetEvpnNeighborDefaultNextHopSelfCmd_handleNoOrDefault,
      mode,
      args )

def SetEvpnNeighborDefaultNextHopSelfCmd_noOrDefaultHandler( mode, args ):
   BgpCmdBaseClass.callNoOrDefaultHandler(
      SetEvpnNeighborDefaultNextHopSelfCmd_handleNoOrDefault,
      mode,
      args )

_DIRECTION_IMPORT = 0x1
_DIRECTION_EXPORT = 0x2
_DIRECTION_BOTH = _DIRECTION_IMPORT | _DIRECTION_EXPORT

def ShowVxlanControlPlaneCmd_vlanHasVniMapping( vlan ):
   return any( vlan in vtiStatus.vlanToVniMap for
         vtiStatus in Globals.vtiStatusDir.vtiStatus.values() )

def ShowVxlanControlPlaneCmd_directionString( direction ):
   if direction == _DIRECTION_IMPORT:
      return 'import'
   elif direction == _DIRECTION_EXPORT:
      return 'export'
   elif direction == _DIRECTION_BOTH:
      return 'both'
   else:
      return 'unknown'

def ShowVxlanControlPlaneCmd_addEvpnVlansToModel(
      model, filterVlans, onlyImports, onlyExports ):
   encapType = Tac.Type( 'Evpn::EncapType' )
   # Check all vlans configured for EVPN to see if they're vlans of interest
   vlanToDirection = {}
   for macVrfConfig in Globals.bgpMacVrfConfigDir.config.values():
      # Only check vlan macvrfs
      if macVrfConfig.isVlanMacVrf():
         hasImports = bool( macVrfConfig.importRtList )
         hasExports = bool( macVrfConfig.exportRtList )

         # We only care about reporting this macvrf if it is importing or
         # exporting routes
         if not hasImports and not hasExports:
            continue

         direction = 0x0
         if hasImports:
            direction |= _DIRECTION_IMPORT
         if hasExports:
            direction |= _DIRECTION_EXPORT

         for brId in macVrfConfig.brIdToEtId:
            vlan = brId.vlanId
            # If we're filtering by specific vlans, skip vlans that we aren't
            # interested in
            if filterVlans and vlan not in filterVlans:
               continue

            # We're only interested in vlans that have an encap type of Vxlan,
            if vlan in evpnStatus.imetLabel:
               label = evpnStatus.imetLabel[ vlan ]
               if label.encapType != encapType.encapVxlan:
                  continue
            else:
               # If we can't check the encap type, just skip it
               continue

            # We're only interested in vlans that have a vni mapping
            if not ShowVxlanControlPlaneCmd_vlanHasVniMapping( vlan ):
               continue

            # If this is a vlan we haven't seen yet, remember the direction
            # If we've already seen this vlan, OR the new direction with the
            # existing direction
            # This handles the case where multiple macvrfs are importing/exporting
            # from the same vlans with different directions
            if vlan not in vlanToDirection:
               vlanToDirection[ vlan ] = direction
            else:
               vlanToDirection[ vlan ] |= direction

   # Update the model with the vlans of interest from EVPN
   for vlan, direction in vlanToDirection.items():
      # Do any direction filtering here, because multiple macvrfs can contribute
      # different directions to the same VLAN, and we want to filter on the
      # "combined" direction, not the direction of any individual macvrf
      if ( onlyImports and ( direction == _DIRECTION_EXPORT ) ) or \
            ( onlyExports and ( direction == _DIRECTION_IMPORT ) ):
         continue

      if vlan not in model.vlans:
         model.vlans[ vlan ] = VxlanModel.VlanControlPlaneConfigModel()

      if 'EVPN' not in model.vlans[ vlan ].controlPlanes:
         model.vlans[ vlan ].controlPlanes[ 'EVPN' ] = \
            VxlanModel.ControlPlaneConfigSourceModel()

      direction = VxlanModel.ControlPlaneConfigDirectionModel(
            direction=ShowVxlanControlPlaneCmd_directionString( direction ) )
      model.vlans[ vlan ].controlPlanes[ 'EVPN' ].sources[ 'configuration' ] = \
            direction

def ShowVxlanControlPlaneCmd_addVcsVlansToModel(
      model, filterVlans, onlyImports, onlyExports ):
   def processStaticVlanDirections( directionAndSourceMap, vlanRange, direction,
         vtiStatus ):
      for vlan in computeVlanRangeSet( vlanRange ):
         # If we're filtering by specific vlans, skip vlans that we aren't
         # interested in
         if filterVlans and vlan not in filterVlans:
            continue

         # We're only interested in vlans that have a vni mapping
         if vlan not in vtiStatus.vlanToVniMap:
            continue

         # If this is a vlan we haven't seen yet, remember the direction and
         # source.  If we've already seen this vlan, OR the new direction with the
         # existing direction
         if vlan not in directionAndSourceMap:
            directionAndSourceMap[ vlan ] = { 'configuration' : direction }
         elif 'configuration' not in directionAndSourceMap[ vlan ]:
            directionAndSourceMap[ vlan ][ 'configuration' ] = direction
         else:
            directionAndSourceMap[ vlan ][ 'configuration' ] |= direction

   # Combine configuration for both statically and dynamically configured vlans
   vlanToDirectionAndSource = {}

   for vtiIntfName, vtiStatus in Globals.vtiStatusDir.vtiStatus.items():
      # Skip adding any VCS entries if controller client mode is disabled on
      # this VTI
      if not vtiStatus.controllerClientMode:
         continue

      # Process vlans that have a static configuration for import or export
      vtiIntfId = Tac.Value( "Arnet::IntfId", vtiIntfName )
      vtiConfig = Globals.vtiConfigDir.vtiConfig.get( vtiIntfId )
      processStaticVlanDirections(
            vlanToDirectionAndSource,
            vtiConfig.cliVccImportVlans,
            _DIRECTION_IMPORT,
            vtiStatus )
      processStaticVlanDirections(
            vlanToDirectionAndSource,
            vtiConfig.cliVccExportVlans,
            _DIRECTION_EXPORT,
            vtiStatus )

      # Process vlans that have dynamic configurations
      for vlan, vsp in vtiStatus.vlanToVniMap.items():
         # If we're filtering by specific vlans, skip vlans that we aren't
         # interested in
         if filterVlans and vlan not in filterVlans:
            continue

         # Skip any vlans whose vni mapping source isn't in the dynamic vlan
         # source whitelist
         try:
            sourceValue = Tac.enumValue( 'Vxlan::VniSource', vsp.source )
            if ( sourceValue & vtiConfig.vccVlansSourceWhitelist ) != sourceValue:
               continue
         except AttributeError:
            # If the source doesn't map to a dynamic enum value, we assume
            # it's a static entry from the CLI (static entries have an empty
            # source)
            continue

         # Currently, all dynamic vlans are treated as having direction "both"
         if vlan not in vlanToDirectionAndSource:
            vlanToDirectionAndSource[ vlan ] = { vsp.source : _DIRECTION_BOTH }
         elif vsp.source not in vlanToDirectionAndSource[ vlan ]:
            vlanToDirectionAndSource[ vlan ][ vsp.source ] = _DIRECTION_BOTH
         else:
            vlanToDirectionAndSource[ vlan ][ vsp.source ] |= _DIRECTION_BOTH

   # Update the model with the vlans of interest from the VCS configuration
   # sources
   for vlan, sourceAndDirection in vlanToDirectionAndSource.items():
      for source, direction in sourceAndDirection.items():
         # Do any direction filtering
         if ( onlyImports and ( direction == _DIRECTION_EXPORT ) ) or \
               ( onlyExports and ( direction == _DIRECTION_IMPORT ) ):
            continue

         if vlan not in model.vlans:
            model.vlans[ vlan ] = VxlanModel.VlanControlPlaneConfigModel()

         if 'VCS' not in model.vlans[ vlan ].controlPlanes:
            model.vlans[ vlan ].controlPlanes[ 'VCS' ] = \
                  VxlanModel.ControlPlaneConfigSourceModel()

         directionModel = VxlanModel.ControlPlaneConfigDirectionModel(
               direction=ShowVxlanControlPlaneCmd_directionString( direction ) )
         model.vlans[ vlan ].controlPlanes[ 'VCS' ].sources[ source ] = \
               directionModel

def ShowVxlanControlPlaneCmd_handler( mode, args ):
   onlyImports = 'import' in args
   onlyExports = 'export' in args
   filterVlans = args.get( 'VLANSET' )

   model = VxlanModel.VxlanControlPlaneConfigModel()

   ShowVxlanControlPlaneCmd_addEvpnVlansToModel(
      model, filterVlans, onlyImports, onlyExports )
   ShowVxlanControlPlaneCmd_addVcsVlansToModel(
      model, filterVlans, onlyImports, onlyExports )

   return model

def Plugin( entityManager ):
   global evpnStatus
   global mcastEvpnConfig

   # Need read access to evpnStatus for the implementation of
   # the vxlan show control-plane command
   evpnStatus = LazyMount.mount(
      entityManager, "evpn/status", "Evpn::EvpnStatus", "r" )

   # Need write access to mcastEvpnConfig to save transit config
   typeName = 'Routing::Multicast::Evpn::Config'
   mcastEvpnConfig = ConfigMount.mount(
      entityManager, Tac.Type( typeName ).mountPath( 'ipv4' ), typeName, "w" )
