#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AgentDirectory
from BgpLib import (
   getVpwsEviName,
   routeTargetToExtCommU64Value,
)
from CliDynamicSymbol import CliDynamicPlugin
from CliPlugin.ArBgpCli import (
   ArBgpAsyncCliCommand,
)
import CliPlugin.EvpnCli as Globals
from CliPlugin.IraServiceCli import getEffectiveProtocolModel
from CliPlugin.RoutingBgpShowCli import ArBgpShowOutput
from IpLibTypes import ProtocolAgentModelType
import LazyMount
import json
from io import StringIO
import sys
import Tac

BgpVpnCliHandler = CliDynamicPlugin( "BgpVpnCliHandler" )
VpnCliHelperCommand = BgpVpnCliHandler.VpnCliHelperCommand
EvpnCliModels = CliDynamicPlugin( "EvpnCliModels" )
ArBgpCliHandler = CliDynamicPlugin( "ArBgpCliHandler" )

configSanityFeatureDir = None

class EvpnExtCommParser:
   # constants and bit shifts for constructing extended communities
   TYPE_AND_SUBTYPE_BGP_ENCAP = 0x030c

   TYPE_EVPN = 0x06
   TYPE_EVPN_SUBTYPE_MAC_MOBILITY = 0x00
   TYPE_EVPN_SUBTYPE_ESI_LABEL = 0x01
   TYPE_EVPN_SUBTYPE_ROUTER_MAC = 0x03
   ENCAP_TUNNEL_TYPE = 0x8

   SHIFT_TYPE = 56
   SHIFT_SUBTYPE = 48
   SHIFT_STICKY_BIT = 40
   SHIFT_SINGLE_ACTIVE = 40

   @staticmethod
   def tunnelEncapExtComm( tunnelTypeStr ):
      # The format of the Tunnel Encapsulation Extended Community can be found in
      # RFC5512 section 4.5.
      # The tunnel types are defined in draft-ietf-bess-evpn-overlay-05 section 13.
      #
      # TODO - Currently we only support "vxlan" tunnel type. Need to add a matcher
      # with other possible tokens when we decide to support more types.
      assert tunnelTypeStr == 'vxlan'
      C = EvpnExtCommParser
      c = ( C.TYPE_AND_SUBTYPE_BGP_ENCAP << C.SHIFT_SUBTYPE ) | C.ENCAP_TUNNEL_TYPE
      return c

   @staticmethod
   def macMobilityExtComm( seqNoStr ):
      # The format of the Mac Mobility extended community can be found in
      # RFC7432 section 7.7
      C = EvpnExtCommParser
      c = ( C.TYPE_EVPN << C.SHIFT_TYPE )
      c |= ( C.TYPE_EVPN_SUBTYPE_MAC_MOBILITY << C.SHIFT_SUBTYPE )
      if seqNoStr == 'sticky':
         c |= ( 0x1 << C.SHIFT_STICKY_BIT )
      else:
         c |= int( seqNoStr )
      return c

   @staticmethod
   def esiLabelExtComm( esiLabel ):
      # The format of the ESI Label extended community can be found in
      # RFC7432 section 7.5
      C = EvpnExtCommParser
      c = ( C.TYPE_EVPN << C.SHIFT_TYPE ) | \
         ( C.TYPE_EVPN_SUBTYPE_ESI_LABEL << C.SHIFT_SUBTYPE )
      if esiLabel[ 'esiLabelRedundancyMode' ] == 'single-active':
         c |= ( 0x1 << C.SHIFT_SINGLE_ACTIVE )
      c |= int( esiLabel[ 'esiLabelValue' ] )
      return c

   @staticmethod
   def routerMacExtComm( macStr ):
      # The format of the Router Mac extended community from:
      # draft-sajassi-l2vpn-evpn-inter-subnet-forwarding-05 sec 6.1, mentioned below.
      """ A new EVPN BGP Extended Community called Router's MAC is introduced
      here. This new extended community is a transitive extended community
      with the Type field of 0x06 (EVPN) and the Sub-Type of 0x03. It may
      be advertised along with BGP Encapsulation Extended Community define
      in section 4.5 of [RFC5512].
      The Router's MAC Extended Community is encoded as an 8-octet value as
      follows:
      0                   1                   2                   3
      0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1 2 3 4 5 6 7 8 9 0 1
      +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
      | Type=0x06     | Sub-Type=0x03 |        Router's MAC           |
      +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
      |                      Router's MAC Cont'd                      |
      +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+ """

      C = EvpnExtCommParser
      mac = Tac.Value( 'Arnet::EthAddr', stringValue=macStr )
      c = ( C.TYPE_EVPN << C.SHIFT_TYPE )
      c |= ( C.TYPE_EVPN_SUBTYPE_ROUTER_MAC << C.SHIFT_SUBTYPE )
      c |= ( mac.word0 << 32 ) | ( mac.word1 << 16 ) | mac.word2
      return c

class EvpnCliHelperCommand( VpnCliHelperCommand ):
   def __init__( self, mode, command, **kwargs ):
      super().__init__(
         mode, command, "l2vpnEvpn", **kwargs )

   @staticmethod
   def flattenExtCommArgs( fromKwargs, toKwargs ):
      """ Input: fromKwargs: cli input params in following format:
      fromKwargs[ "extCommunityValues" ]["rt"]= 1:1
      fromKwargs[ "extCommunityValues" ]["tunnelEncap"]="vxlan" etc.

      convertExtCommunityValues: populates toKwargs[ "extCommValues" ] with a string:
      '|' separated, U64 values of extcommunities passed in cli. such as:
      toKwargs[ "extCommValues" ]='43189991171031040|21550481834311688|5629542483886'
      toKwargs[ "exact" ] = fromKwargs["exact" ], if it is set.
      """
      exact = fromKwargs.get( "exact", None )
      if exact:
         toKwargs[ "extendedCommunitiesExactMatch" ] = exact
      commValues = fromKwargs[ "extCommunityValues" ]
      extCommunities = ""
      C = EvpnExtCommParser
      for ( extCommType, extCommValue ) in commValues:
         if extCommunities != "":
            extCommunities += "|"
         if extCommType == 'rt':
            extCommunities += "|".join( [ str( routeTargetToExtCommU64Value( i ) )
               for i in extCommValue ] )
         elif extCommType == 'tunnelEncap':
            extCommunities += str( C.tunnelEncapExtComm( extCommValue ) )
         elif extCommType == 'macMobility':
            extCommunities += str( C.macMobilityExtComm( extCommValue ) )
         elif extCommType == 'esiLabel':
            extCommunities += str( C.esiLabelExtComm( extCommValue ) )
         elif extCommType == 'routerMac':
            extCommunities += str( C.routerMacExtComm( extCommValue ) )
         else:
            assert False, f"invalid extComType: {extCommType}"
      toKwargs[ "extCommValues" ] = extCommunities

   @staticmethod
   def flattenMulticastArgs( fromKwargs, toKwargs ):
      """ Input: fromKwargs: cli input params in following format:
      fromKwargs[ "multicast" ]["first"]= IPv4 or IPv6 addr
      fromKwargs[ "multicast" ]["second"]= Optional IPv4 or IPv6 addr

      Only applies to Multicast routes ( Type 6,7,8, and 10 )
      if only first is specified, output routes with source or group that match
      - toKwargs[ "multicast" ] = '<IP addr>'
      if second IP addr also exists, output routes where the source,group matches
      first,second or second,first depending on which one is the unicast or
      multicast address.
      - toKwargs[ "multicast" ] = '<first IP>|<second IP>'
      """
      first = fromKwargs.get( "first" )
      second = fromKwargs.get( "second" )
      if second:
         toKwargs[ "multicast" ] = f"{first}|{second}"
      else:
         toKwargs[ "multicast" ] = first

   @staticmethod
   def flattenArgs( fromKwargs, toKwargs ):
      k = 'extCommValuesAndExact'
      extComm = fromKwargs.get( k, None )
      if extComm:
         EvpnCliHelperCommand.flattenExtCommArgs( fromKwargs[ k ], toKwargs )

      multicast = fromKwargs.pop( 'multicast', None )
      if multicast:
         EvpnCliHelperCommand.flattenMulticastArgs( multicast, toKwargs )

      VpnCliHelperCommand.flattenArgs( fromKwargs, toKwargs )

   def run( self, **kwargs ):
      super().run( **kwargs )
      return EvpnCliModels.EvpnRoutes

def doShowBgpEvpn( mode, **kwargs ):
   kwargs.update( kwargs.pop( 'args', {} ) )
   kwargs[ 'vniDottedNotation' ] = Globals.vxlanCtrlConfig.vniInDottedNotation

   # Flatten the arguments
   setKwargs = {}
   EvpnCliHelperCommand.flattenArgs( kwargs, setKwargs )

   if setKwargs.get( "prefixValue", False ):
      # Set a disctinct prefix value to not colide with MplsVpn prefix value
      setKwargs[ "evpnPrefixValue" ] = setKwargs.pop( "prefixValue" )

   if setKwargs.get( "internal", False ):
      prefix = setKwargs.pop( "evpnPrefixValue", None )
      ArBgpAsyncCliCommand( mode, "show bgp <af> internal",
                            prefix=prefix, **setKwargs ).run()
      return EvpnCliModels.EvpnRoutes

   return EvpnCliHelperCommand( mode, 'show bgp evpn', **setKwargs ).run()

def doShowBgpEvpnL2( mode, **kwargs ):
   kwargs.update( kwargs.pop( 'args', {} ) )
   kwargs[ 'vniDottedNotation' ] = Globals.vxlanCtrlConfig.vniInDottedNotation
   # Flatten the arguments
   setKwargs = {}
   EvpnCliHelperCommand.flattenArgs( kwargs, setKwargs )
   if 'arp' in setKwargs:
      setKwargs[ 'nlriType' ] = 'macArp'
   else:
      setKwargs[ 'nlriType' ] = 'macAddress'
      
   return EvpnCliHelperCommand( mode, 'show bgp evpn l2', **setKwargs ).run()

def doShowBgpEvpnRouteTypeCount( mode, args ):
   ArBgpAsyncCliCommand( mode, 'show bgp evpn route-type count' ).run()
   return EvpnCliModels.EvpnPathRouteTypeCount

def createItem( name, status, reasonNotOk=None ):
   item = EvpnCliModels.EvpnSanityItem()
   item.name = name
   item.status = status
   if reasonNotOk:
      item.reasonNotOk = reasonNotOk
   return item

def checkSendCommunity( mode, args ):
   neighborConfig = Globals.bgpConfig.neighborConfig
   afStateEnum = Tac.Type( "Routing::Bgp::PeerAFState" )
   items = []
   sendCommHeading = "Send community"
   sendExtCommError = "Extended community not sent to %s"
   for peerConfigKey, peerConfig in neighborConfig.items():
      if ( peerConfig.isPeerGroupPeer or
           peerConfig.afEvpn != afStateEnum.afActive ):
         continue

      if peerConfigKey.type == 'peerIpv4':
         peerName = peerConfigKey.v4Addr
      elif peerConfigKey.type == 'peerGroup':
         peerName = peerConfigKey.group
      else:
         peerName = peerConfigKey.v6Addr

      sendComm = peerConfig.sendCommunity
      sendExtComm = peerConfig.sendExtendedCommunity or sendComm
      if not sendExtComm:
         detail = sendExtCommError % peerName
         items.append( createItem( sendCommHeading, 'fail', detail ) )

   if not items:
      items.append( createItem( sendCommHeading, 'ok' ) )
   return items

def checkMultiAgent( mode, args ):
   multiAgentHeading = "Multi-agent mode"
   multiAgentError = 'Configure "service routing protocols model multi-agent"'
   if getEffectiveProtocolModel( mode ) != ProtocolAgentModelType.multiAgent:
      return [ createItem( multiAgentHeading, 'fail',
                           multiAgentError ) ]
   else:
      return [ createItem( multiAgentHeading, 'ok' ) ]

@ArBgpShowOutput( 'doCheckEvpnNeighbors', arBgpModeOnly=True )
def doCheckEvpnNeighbors( mode, args ):
   strBuff = StringIO()
   cmd = ArBgpAsyncCliCommand( mode, "show summary", nlriAfiSafi='l2vpnEvpn' )
   cmd.run( stringBuff=strBuff, forceOutputFormat='json' )
   try:
      data = json.loads( strBuff.getvalue() )
   except ValueError:
      mode.addError( 'Error loading information from "show bgp evpn summary"' )
      return None
   evpnNeighborHeading = "Neighbor established"
   if not data or not data.get( 'vrfs' ):
      return [ createItem( evpnNeighborHeading, 'ok' ) ]
   notEstabEvpnPeers = []
   for vrf in data[ 'vrfs' ].values():
      for peerName in vrf[ 'peers' ]:
         if vrf[ 'peers' ][ peerName ][ 'peerState' ] != 'Established':
            notEstabEvpnPeers.append( peerName )

   errorMsg = ( "Following EVPN sessions are not established: " +
                ', '.join( notEstabEvpnPeers ) )
   if notEstabEvpnPeers:
      return [ createItem( evpnNeighborHeading, 'warn', errorMsg ) ]
   else:
      return [ createItem( evpnNeighborHeading, 'ok' ) ]

def checkEvpnNeighbors( mode, args ):
   # Checking of EVPN neighbors can only be done while Bgp agent is active
   # and the system is in 'multi-agent' mode
   bgpAgentPresent = AgentDirectory.agentIsRunning( mode.entityManager.sysname(),
                                                    'Bgp' )
   multiAgentMode = ( Globals.l3ProtocolAgentModelStatus.protocolAgentModel ==
                      ProtocolAgentModelType.multiAgent )

   if multiAgentMode and bgpAgentPresent:
      return doCheckEvpnNeighbors( mode, args )
   return None

# Gets the show output ready name of the mac vrf
def getMacVrfName( macVrfConfig ):
   if macVrfConfig.macVrfType == Globals.MacVrfTypeEnum.macVrfTypeVlan:
      if macVrfConfig.isBundle:
         # vlanbundle.bundleName -> bundleName
         return macVrfConfig.name[ 11 : ]
      else:
         # vlan.# -> VLAN #
         return 'VLAN ' + macVrfConfig.name[ 5 : ]
   elif macVrfConfig.macVrfType == Globals.MacVrfTypeEnum.macVrfTypeVpws:
      # vpws.vpwsName -> vpwsName
      return getVpwsEviName( macVrfConfig.name )
   elif macVrfConfig.macVrfType == Globals.MacVrfTypeEnum.macVrfTypeVni:
      # vnibundle.bundleName -> bundleName
      return macVrfConfig.name[ 10 : ]
   return macVrfConfig.name

def getSortedMacVrfConfigList():
   def macVrfSortFunc( macVrfConfig ):
      name = getMacVrfName( macVrfConfig )
      if ( macVrfConfig.macVrfType == Globals.MacVrfTypeEnum.macVrfTypeVlan
           and not macVrfConfig.isBundle ):
         # Standalone VLANs come first in sorting and are sorted numerically
         return ( False, int( name.split()[ 1 ] ) )
      return ( True, name )
   return sorted( Globals.bgpMacVrfConfigDir.config.values(), key=macVrfSortFunc )

def checkMacVrfRouteTarget( mode, args ):
   routeTargetHeading = "MAC-VRF route-target import and export"
   warnExportImport = "The MAC-VRF %s does not import or export a route-target"
   warnExport = "The MAC-VRF %s does not export a route-target"
   warnImport = "The MAC-VRF %s does not import a route-target"

   items = []
   for macVrfConfig in getSortedMacVrfConfigList():
      hasExport = ( macVrfConfig.exportRtList or
                    macVrfConfig.exportRemoteDomainRtList )
      hasImport = ( macVrfConfig.importRtList or
                    macVrfConfig.importRemoteDomainRtList )

      if not hasExport or not hasImport:
         if not hasExport and not hasImport:
            warnMsg = warnExportImport
         elif not hasExport:
            warnMsg = warnExport
         else:
            warnMsg = warnImport

         reasonNotOk = warnMsg % getMacVrfName( macVrfConfig )
         items.append( createItem( routeTargetHeading, 'warn', reasonNotOk ) )

   if not items:
      items.append( createItem( routeTargetHeading, 'ok' ) )
   return items

def checkMacVrfRouteDistinguisher( mode, args ):
   macVrfRdHeading = "MAC-VRF route-distinguisher"
   warnMsg = "The MAC-VRF %s is not configured with a route-distinguisher"

   items = []
   for macVrfConfig in getSortedMacVrfConfigList():
      if ( macVrfConfig.rd == 'INVALID' and macVrfConfig.remoteRd == 'INVALID'
           and not macVrfConfig.autoRd ):
         reasonNotOk = warnMsg % getMacVrfName( macVrfConfig )
         items.append( createItem( macVrfRdHeading, 'warn', reasonNotOk ) )

   if not items:
      items.append( createItem( macVrfRdHeading, 'ok' ) )
   return items

def checkMacVrfRedistribute( mode, args ):
   macVrfRedistHeading = 'MAC-VRF redistribute'
   warnMsg = ( 'The MAC-VRF %s should enable "redistribute learned" ' +
               'or "redistribute static"' )

   items = []
   for macVrfConfig in getSortedMacVrfConfigList():
      redistLearn = macVrfConfig.isInRedistribute( 'redistributeLearned' )
      redistStatic = macVrfConfig.isInRedistribute( 'redistributeStatic' )
      if not redistLearn and not redistStatic:
         reasonNotOk = warnMsg % getMacVrfName( macVrfConfig )
         items.append( createItem( macVrfRedistHeading, 'warn', reasonNotOk ) )
   if not items:
      items.append( createItem( macVrfRedistHeading, 'ok' ) )
   return items

def checkMacVrfOverlap( mode, args ):
   macVrfOverlapHeading = 'MAC-VRF overlapping VLAN'
   failMsg = 'VLAN %s is a member of the following MAC-VRFs: %s'
   bridgeTypeVlan = Tac.Type( 'Bridging::BrType' ).brTypeVlan

   seen = {}
   for macVrfConfig in Globals.bgpMacVrfConfigDir.config.values():
      for bridgeId in macVrfConfig.brIdToEtId:
         if bridgeId.brType != bridgeTypeVlan:
            continue
         vlanNum = int( bridgeId.vlanId )
         macVrfName = getMacVrfName( macVrfConfig )
         if vlanNum in seen:
            seen[ vlanNum ].append( macVrfName )
         else:
            seen[ vlanNum ] = [ macVrfName ]

   items = []
   for vlanNum in sorted( seen.keys() ):
      macVrfs = seen[ vlanNum ]
      vlanId = str( vlanNum )
      if len( macVrfs ) > 1:
         macVrfs.sort()
         reasonNotOk = failMsg % ( vlanId, ", ".join( macVrfs ) )
         items.append( createItem( macVrfOverlapHeading, 'fail', reasonNotOk ) )

   if not items:
      items.append( createItem( macVrfOverlapHeading, 'ok' ) )
   return items

def getVxlanEnabledMacVrfs( mode ):
   strBuff = StringIO()
   cmd = ArBgpAsyncCliCommand( mode, "show evpn instance" )
   cmd.run( stringBuff=strBuff, forceOutputFormat='json' )
   try:
      data = json.loads( strBuff.getvalue() )
   except ValueError:
      mode.addError( 'Error loading information from "show bgp instance"' )
      return None
   if not data or not data.get( 'bgpEvpnInstances' ):
      return None

   vxlanEnabledVlans = []
   for vlanName, vlanDict in data[ 'bgpEvpnInstances' ].items():
      if vlanDict[ 'encapType' ] == 'vxlan':
         vxlanEnabledVlans.append( vlanName )
   return vxlanEnabledVlans

def checkMacVrfVlanToVni( mode, args ):
   bgpAgentPresent = AgentDirectory.agentIsRunning( mode.entityManager.sysname(),
                                                    'Bgp' )
   if not ( Globals.vtiConfigDir.vtiConfig and bgpAgentPresent ):
      return []
   vlanToVniHeading = 'VLAN to VNI map for MAC-VRF'
   vxlanMacVrfs = getVxlanEnabledMacVrfs( mode )
   if not vxlanMacVrfs:
      return [ createItem( vlanToVniHeading, 'ok' ) ]

   bridgeTypeVlan = Tac.Type( 'Bridging::BrType' ).brTypeVlan
   vlans = set()
   for macVrfConfig in Globals.bgpMacVrfConfigDir.config.values():
      macVrfName = getMacVrfName( macVrfConfig )
      if macVrfConfig.isBundle:
         macVrfName = 'VLAN-aware bundle ' + macVrfName
      if macVrfName not in vxlanMacVrfs:
         continue
      for bridgeId in macVrfConfig.brIdToEtId:
         if bridgeId.brType == bridgeTypeVlan:
            vlans.add( bridgeId.vlanId )

   for vtiConfig in Globals.vtiConfigDir.vtiConfig.values():
      for vlanId in vtiConfig.vlanToVniMap:
         vlans.discard( vlanId )

   items = []
   vlanInts = [ int( vlan ) for vlan in vlans ]
   warnMsg = 'VLAN to VNI mapping missing for MAC-VRF VLAN %s'
   for vlanNum in sorted( vlanInts ):
      reasonNotOk = warnMsg % str( vlanNum )
      items.append( createItem( vlanToVniHeading, 'warn', reasonNotOk ) )
   if not items:
      items.append( createItem( vlanToVniHeading, 'ok' ) )
   return items

def checkIpVrfToVni( mode, args ):
   if not Globals.vtiConfigDir.vtiConfig:
      return []

   vrfToVniHeading = 'VRF to VNI map for IP-VRF'
   warnMsg = 'VRF to VNI mapping missing for IP-VRF %s'

   vrfs = set( Globals.vrfConfigDir.vrfConfig )
   for vtiConfig in Globals.vtiConfigDir.vtiConfig.values():
      for vrfName in vtiConfig.vrfToVniMap:
         vrfs.discard( vrfName )

   items = []
   for vrf in sorted( vrfs ):
      reasonNotOk = warnMsg % vrf
      items.append( createItem( vrfToVniHeading, 'warn', reasonNotOk ) )
   if not items:
      items.append( createItem( vrfToVniHeading, 'ok' ) )
   return items

def checkMacAddrSuppressionlist( mode, args ):
   macSuppressedHeading = 'Suppressed MAC'
   warnMsg = ( 'MAC Addresses have been suppressed, run "show bgp evpn ' +
               'host-flap" for more details' )
   if Globals.macDuplicationBlacklist.blacklist:
      return [ createItem( macSuppressedHeading, 'warn', warnMsg ) ]
   else:
      return [ createItem( macSuppressedHeading, 'ok' ) ]

evpnSanityChecks = {
   'General' : {
      'order' : 0,
      'description' : 'General EVPN verification',
      'checks' : [ checkSendCommunity,
                   checkMultiAgent,
                   checkEvpnNeighbors, ]
   },
   'L2' : {
      'order' : 1,
      'description' : 'Layer 2 EVPN verification',
      'checks' : [ checkMacVrfRouteTarget,
                   checkMacVrfRouteDistinguisher,
                   checkMacVrfRedistribute,
                   checkMacVrfOverlap,
                   checkMacAddrSuppressionlist ]
   },
   'VXLAN' : {
      'order' : 3,
      'description' : 'VXLAN encapsulation EVPN verification',
      'checks' : [ checkMacVrfVlanToVni,
                   checkIpVrfToVni, ]
   }
}

def saveEvpnSanityToSysdb( model ):
   def capiToConfigSanityResult( status ):
      return ( 'Pass' if status.lower() == 'ok'
                else 'Fail' if status.lower() == 'fail' else 'Warn' )

   evpnFeature = configSanityFeatureDir.features.newMember( 'evpn' )
   newSeqNum = 1 if evpnFeature.seqNum == sys.maxsize else evpnFeature.seqNum + 1

   # Delete stale entries
   for categoryName, categorySysdb in evpnFeature.categories.items():
      if categoryName not in model.categories:
         del categorySysdb
         continue
      itemIds = [ item.id() for item in model.categories[ categoryName ].items ]
      for itemId in categorySysdb.items:
         if itemId not in itemIds:
            del categorySysdb.items[ itemId ]

   for categoryName, category in model.categories.items():
      evpnFeature.categories.newMember( categoryName )
      categorySysdb = evpnFeature.categories[ categoryName ]
      categorySysdb.description = category.description
      categorySysdb.result = capiToConfigSanityResult( category.getCategoryStatus() )
      categorySysdb.priority = category.order()

      for item in category.items:
         itemResult = capiToConfigSanityResult( item.status )
         itemDetail = item.reasonNotOk or ''
         itemSysdb = Tac.Value( 'ConfigSanity::Item', item.name,
                                itemDetail, itemResult )
         categorySysdb.items[ item.id() ] = itemSysdb
   evpnFeature.seqNum = newSeqNum

def addIdToItems( itemList ):
   for counter, item in enumerate( itemList ):
      item.idIs( item.name + '_' + str( counter ) )

def showEvpnSanity( mode, args ):
   if 'detail' in args:
      model = EvpnCliModels.EvpnSanityModelDetail()
   elif 'brief' in args:
      model = EvpnCliModels.EvpnSanityModelBrief()
   else:
      model = EvpnCliModels.EvpnSanityModel()

   # EVPN not active on any peer, skip checks
   if not Globals.evpnActivatedForPeer():
      saveEvpnSanityToSysdb( model )
      return model

   for categoryName, categoryInfo in evpnSanityChecks.items():
      category = EvpnCliModels.EvpnSanityCategory()
      category.description = categoryInfo[ 'description' ]
      category.orderIs( categoryInfo[ 'order' ] )
      for itemCheck in categoryInfo[ 'checks' ]:
         items = itemCheck( mode, args )
         if items:
            addIdToItems( items )
            category.items.extend( items )
      if not category.items:
         continue
      model.categories[ categoryName ] = category

   # Write ConfigSanityModel to Sysdb so that CVP can stream state via
   # TerminAttr Prevent write to Sysdb when supervisor is not active
   if mode.entityManager.redundancyStatus().mode == 'active':
      saveEvpnSanityToSysdb( model )
   return model

@ArBgpShowOutput( 'doShowBgpDebugPolicyEvpn', arBgpModeOnly=True )
def policyDebugEvpnHandler( mode, args ):
   # Only EVPN type 5 (route-type ip-prefix) is supported
   nlriAttrs = {}
   # We populate the RD for export routes at a different stage
   if not args.get( 'export' ):
      assert args.get( 'route-type' )
      assert args.get( 'ip-prefix' )
      assert args.get( 'RD' )
      # Get the NLRI type
      # The 'nlriType' value should correspond to the enum values in BgpNlriType.tac
      nlriAttrs = { 'rd' : args.get( 'RD' ) }
   if args.get( 'PREFIX6' ) is not None:
      # The IPv6 prefix is stored as an IpGenAddr so we need to check if the attr
      # "is not None" to correctly process the default prefix ::/0
      nlriType = 'evpnType5Ipv6'
      nlriAttrs[ 'ipPrefix' ] = args[ 'PREFIX6' ].stringValue()
   elif args.get( 'PREFIX' ):
      # The IPv4 prefix is stored as a string so processing the default prefix
      # works as intended
      nlriType = 'evpnType5Ipv4'
      nlriAttrs[ 'ipPrefix' ] = args[ 'PREFIX' ]
   else:
      assert False, 'Either PREFIX or PREFIX6 must be specified'
   return ArBgpCliHandler.policyDebugHandlerCommon( mode, args, nlriType, nlriAttrs )

def Plugin( entityManager ):
   global configSanityFeatureDir

   # Need write access to configSanityFeatureDir to save EVPN sanity to Sysdb
   configSanityFeatureDir = LazyMount.mount( entityManager,
                           'configsanity/feature',
                           'ConfigSanity::FeatureDir', 'w' )
