# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import AgentCommandRequest
from Arnet import IpGenAddr
from ArnetModel import IpGenericAddrAndPort
import Cell
import CliGlobal
from CliModel import cliPrinted
from CliDynamicSymbol import CliDynamicPlugin
from CliPlugin.FlowTrackingShowCli import getIntfsForTracker
import CliPlugin.FlowWatcherCliLib
from CliPlugin.FlowWatcherCliLib import (
   FlowWatcherConstants,
   FlowWatcherInactiveReason,
   FlowWatcherInactiveReasonEnum,
   NucleusInactiveReasonEnum,
   switchTimeToUtc,
)
import ExtensionMgrLib
from FlowTrackerCliUtil import (
   ftrTypeHardware,
   ftrTypeSampled,
   hwFtrConfigPathPrefix,
   hwFtrConfigType,
)
from IpLibConsts import DEFAULT_VRF
import LazyMount
import math
import Shark
import SharkLazyMount
import json
from io import StringIO
import SmashLazyMount
import sys
import Tac
import Tracing
from TypeFuture import TacLazyType

traceHandle = Tracing.Handle( 'FlowWatcherShowCli' )
t0 = traceHandle.trace0
t1 = traceHandle.trace1

AddressFamily = TacLazyType( "Arnet::AddressFamily" )
FlowWatcherModel = CliDynamicPlugin( "FlowWatcherModel" )
NamespaceName = TacLazyType( 'Arnet::NamespaceName' )

# Global variable holder.
gv = CliGlobal.CliGlobal(
   dict(
      extMgrStatus=None,
      fwConfig=None,
      fwStatus=None,
      ipfixCollectorCounters=None,
      ipLockingStatus=None,
      flowTable=None,
      dpiStatus=None,
      hwTrackingConfig={},
   ) )

interval5min = FlowWatcherConstants.interval5min
interval1hour = FlowWatcherConstants.interval1hour
interval24hour = FlowWatcherConstants.interval24hour
intervals = [ interval5min, interval1hour, interval24hour ]

# ----------------------------------------------------------------------
# show monitor security awake [nucleus <nucleus>]
# ----------------------------------------------------------------------

def getFlowTracker( hwTrackingConfig ):
   """Get the name and config of the configured FlowTracker"""
   if hwTrackingConfig:
      for ftrName, ftrConfig in hwTrackingConfig.hwFtConfig.items():
         for expConfig in ftrConfig.expConfig.values():
            for ipAndPort in expConfig.collectorIpAndPort.values():
               if IpGenAddr( ipAndPort.ip ).isLoopback:
                  return ftrName, ftrConfig # found tracker
            for ipAndPort in expConfig.collectorIp6AndPort.values():
               if ipAndPort.ip.isLoopback:
                  return ftrName, ftrConfig # found tracker
   return None, None

def addMsa( model ):
   msa = FlowWatcherModel.MsaStatus()
   if gv.fwStatus.enabled:
      # reasons as determined by the agent (agent is running)
      inactiveReason = FlowWatcherInactiveReason( gv.fwStatus.inactiveReason )
   else:
      # reasons as determined by the Cli (agent is not running)
      inactiveReason = FlowWatcherInactiveReason()
      if FlowWatcherConstants.swixName not in gv.extMgrStatus.installedPrimaryPkg:
         inactiveReason.eosExtensionNotInstalled = True
      elif not gv.fwConfig.enabled: # agent is disabled in config
         inactiveReason.monitorSecurityAwakeDisabled = True
      else: # agent is enabled in config but not running yet
         inactiveReason.monitorSecurityAwakeInitializing = True
   for reasonNum in range( len( FlowWatcherInactiveReasonEnum.attributes ) ):
      if inactiveReason.value & ( 1 << reasonNum ) != 0:
         # see BUG652804; pylint: disable=protected-access
         name = Tac.enumName( FlowWatcherInactiveReasonEnum._type, reasonNum )
         msa.inactiveReasons.append(
            FlowWatcherModel.MsaInactiveReason( inactiveReason=name ) )
   msa.active = not msa.inactiveReasons
   msa.lowMemoryMode = gv.fwStatus.lowMemoryMode
   msa.topicName = gv.fwStatus.topic
   msa.monitorPointId = gv.fwStatus.monitorPointId
   msa.flowTableSize = gv.fwStatus.flowTableSize
   msa.flowTableInactiveTimeout = gv.fwStatus.flowTableInactiveTimeout
   ftrName, tracker = getFlowTracker( gv.hwTrackingConfig.get(
      CliPlugin.FlowWatcherCliLib.gv.hwCapabilities.feature ) )
   if msa.active and ftrName:
      activeIntfs, _ = getIntfsForTracker(
         CliPlugin.FlowWatcherCliLib.gv.hwCapabilities.feature, ftrName, tracker )
      msa.activeIntfs += [ intf for intf, _ in activeIntfs ]
   model.status = msa

def addNucleus( model, nucleusFilter=None ):
   for nucleusName, nucleusStatus in (
         CliPlugin.FlowWatcherCliLib.gv.nucleusStatusColl.status.items() ):
      if nucleusFilter and nucleusName != nucleusFilter:
         continue
      nucleus = FlowWatcherModel.MsaNucleus()
      if nucleusStatus.connectionStatus == "connected":
         nucleus.status = "connected"
      else:
         nucleus.status = "inactive"
         for reasonNum in range( len( NucleusInactiveReasonEnum.attributes ) ):
            if nucleusStatus.inactiveReason & ( 1 << reasonNum ) != 0:
               # see BUG652804; pylint: disable=protected-access
               inactiveReason = Tac.enumName( NucleusInactiveReasonEnum._type,
                                              reasonNum )
               errorReason = ( nucleusStatus.errorReason
                               if inactiveReason in ( "connectionLost",
                                                      "connectionFailed" )
                               else None )
               nucleus.inactiveReasons.append(
                  FlowWatcherModel.MsaNucleusInactiveReason(
                     inactiveReason=inactiveReason, errorReason=errorReason ) )
      nucleus.vrf = nucleusStatus.vrf
      nucleus.localIntf = nucleusStatus.localIntf
      nucleus.localAddr = nucleusStatus.localAddr
      nucleus.destinationAddr = nucleusStatus.addr
      nucleus.destinationPort = nucleusStatus.port
      nucleus.sslProfile = nucleusStatus.sslProfile
      nucleus.lastEstablished = switchTimeToUtc(
         nucleusStatus.lastConnectionSuccessTimestamp )
      model.nucleuses[ nucleusName ] = nucleus

def showMsaHandler( mode, args ):
   model = FlowWatcherModel.Msa()
   if 'nucleus' in args:
      nucleusFilter = args.get( 'NUCLEUS_NAME' )
      addNucleus( model, nucleusFilter=nucleusFilter )
   else:
      addMsa( model )
      addNucleus( model )
   return model

# ----------------------------------------------------------------------
# show monitor security awake counters [flows|ipfix|nucleus [<nucleus>]]
# ----------------------------------------------------------------------

def addAppCounters( flowCountersModel, dpiCounters, ipv6=False ):
   if ipv6:
      appCounters = dpiCounters.ipv6AppCounters
      countersModel = flowCountersModel.ipv6AppCounters
   else:
      appCounters = dpiCounters.ipv4AppCounters
      countersModel = flowCountersModel.ipv4AppCounters
   for app, counters in appCounters.items():
      knownApplication = dpiCounters.knownApplications.get( app )
      appName = knownApplication.name if knownApplication else "Other"
      appCountersModel = countersModel.get( appName )
      activeFlows = counters.flows - counters.expiredFlows
      if gv.ipLockingStatus.active and appName == "DHCP":
         # When IP locking is enabled, we should not show DHCP counters
         # and we will remove DHCP counters from global counters
         flowCountersModel.flows -= counters.flows
         flowCountersModel.activeFlows -= activeFlows
         flowCountersModel.expiredFlows -= counters.expiredFlows
         continue
      appCountersModel = countersModel.get( appName, FlowWatcherModel.AppCounters() )
      appCountersModel.flows += counters.flows
      appCountersModel.activeFlows += activeFlows
      appCountersModel.expiredFlows += counters.expiredFlows
      if ( appCountersModel.flows or appCountersModel.activeFlows or
           appCountersModel.expiredFlows ):
         countersModel[ appName ] = appCountersModel

def calculateNewAverage( oldAverage, deltaValue, deltaTime, p ):
   if oldAverage == 0:
      return deltaValue / deltaTime
   else:
      return ( oldAverage * p ) + ( deltaValue / deltaTime ) * ( 1.0 - p )

def movingAvgCoeff( deltaTime, loadInterval ):
   if loadInterval > 0:
      return math.exp( - deltaTime / loadInterval )
   else:
      return 0

def calculateFlowsRate( now, dpiCounters, rate ):
   if not rate:
      return ( 0.0, 0.0 )
   deltaTime = now - rate.lastUpdateTime
   if deltaTime <= 0:
      return ( rate.flowsRate, rate.packetsRate )
   delta = dpiCounters.flows - rate.lastFlows
   p = movingAvgCoeff( deltaTime, rate.interval )
   flowsAvg = calculateNewAverage( rate.flowsRate,
                                   delta,
                                   deltaTime,
                                   p )
   delta = dpiCounters.packets - rate.lastPackets
   packetsAvg = calculateNewAverage( rate.packetsRate,
                                     delta,
                                     deltaTime,
                                     p )
   return( flowsAvg, packetsAvg )

def updateFlowCountersRate( flowCountersModel, dpiCounters ):
   # Calculate exponential moving average rate based on last saved rate and
   # delta counter/time to get an approximate recent rate, but will not save to
   # dpiCounters since only FlowWatcher has write access to dpiCounters SHARK.
   # handleCountersRate() in FlowWatcher agent will update dpiCounters SHARK
   # periodically.
   now = Tac.now()
   for interval in intervals:
      rate = dpiCounters.rates.get( interval )
      if rate:
         ( flowCountersModel.flowsRates[ interval ],
           flowCountersModel.packetsRates[ interval ] ) = (
              calculateFlowsRate( now, dpiCounters, rate ) )
      else:
         flowCountersModel.flowsRates[ interval ] = 0.0
         flowCountersModel.packetsRates[ interval ] = 0.0

def addFlowCounters( model, withQueueFullCounter=False ):
   flowCountersModel = FlowWatcherModel.FlowCounters()
   flowCountersModel.activeFlows = len( gv.flowTable.flowEntry )
   dpiCounters = gv.dpiStatus.dpiCounters.get( "dpiCounters" )
   if dpiCounters:
      flowCountersModel.flows = dpiCounters.flows
      flowCountersModel.expiredFlows = dpiCounters.expiredFlows
      flowCountersModel.packets = dpiCounters.packets
      flowCountersModel.lastCountersClearedTime = switchTimeToUtc(
         dpiCounters.lastResetTime )
      addAppCounters( flowCountersModel, dpiCounters, ipv6=False )
      addAppCounters( flowCountersModel, dpiCounters, ipv6=True )
      if withQueueFullCounter:
         model.activityRecordQueueFull = dpiCounters.activityQueueFull
         model.flowTableFull = dpiCounters.flowTableLimit
      updateFlowCountersRate( flowCountersModel, dpiCounters )
   else:
      for interval in intervals:
         flowCountersModel.flowsRates[ interval ] = 0.0
         flowCountersModel.packetsRates[ interval ] = 0.0
   model.flowCounters = flowCountersModel

def calculateIpfixCountersRate( now, counter, rate ):
   if not rate:
      return ( 0.0, 0.0 )
   deltaTime = now - rate.lastUpdateTime
   if deltaTime <= 0:
      return ( rate.messagesRate, rate.dataRecordsRate )
   p = movingAvgCoeff( deltaTime, rate.key )
   delta = counter.rxMessages - rate.lastMessages
   messagesAvg = calculateNewAverage( rate.messagesRate,
                                      delta,
                                      deltaTime,
                                      p )
   delta = counter.rxDataRecords - rate.lastDataRecords
   dataRecordsAvg = calculateNewAverage( rate.dataRecordsRate,
                                         delta,
                                         deltaTime,
                                         p )
   return( messagesAvg, dataRecordsAvg )

def updateIpfixCountersRate( ipfixCountersModel, counter ):
   # Calculate exponential moving average rate based on last saved rate and
   # delta counter/time to get an approximate recent rate, but will not save to
   # ipfixCounters since only FlowWatcher has write access to ipfixCounters SMASH.
   # handleCountersRate() in FlowWatcher agent will update ipfixCounters SMASH
   # periodically.
   now = Tac.now()
   for interval in intervals:
      rate = gv.ipfixCollectorCounters.rates.get( interval )
      if rate:
         ( ipfixCountersModel.messagesRates[ interval ],
          ipfixCountersModel.dataRecordsRates[ interval ] ) = (
             calculateIpfixCountersRate( now, counter, rate ) )
      else:
         ipfixCountersModel.messagesRates[ interval ] = 0.0
         ipfixCountersModel.dataRecordsRates[ interval ] = 0.0

def addIpfixCounters( model ):
   # get exporter info from fwStatus.collectorStatusDir
   exporterAddr = "0.0.0.0"
   exporterPort = 0
   observationDomainId = 0
   # at this time, there is only one exporter and one ipfixCollectorCounter
   if gv.fwStatus.collectorStatusDir: # is None when agent is not running
      for exporter in gv.fwStatus.collectorStatusDir.exporterMap.exporterToId:
         exporterAddr = exporter.addr
         exporterPort = exporter.srcPort
         observationDomainId = exporter.observationDomainId
         break
   for exporter, counter in gv.ipfixCollectorCounters.counterEntry.items():
      ipfixCounters = FlowWatcherModel.IpfixCounters()
      ipfixCounters.exporter = IpGenericAddrAndPort(
         ip=exporterAddr, port=exporterPort )
      ipfixCounters.observationDomainId = observationDomainId
      ipfixCounters.rxMessages = counter.rxMessages
      ipfixCounters.rxTemplateRecords = counter.rxTemplateRecords
      ipfixCounters.rxOptionsTemplateRecords = counter.rxOptionsTemplateRecords
      ipfixCounters.rxDataRecords = counter.rxDataRecords
      ipfixCounters.rxOptionsDataRecords = counter.rxOptionsDataRecords
      ipfixCounters.templateIdErrors = counter.templateIdErrors
      ipfixCounters.invalidIpfixMsgs = counter.invalidIpfixMsgs
      ipfixCounters.flowRecordQueueFull = counter.flowRecordQueueFull
      ipfixCounters.lastCountersClearedTime = switchTimeToUtc(
         gv.ipfixCollectorCounters.lastResetTime )
      updateIpfixCountersRate( ipfixCounters, counter )
      model.ipfixCounters.append( ipfixCounters )

def calculateNucleusRate( now, nucleusStatus, rate ):
   if not rate:
      return ( 0.0, 0.0, 0.0 )
   deltaTime = now - rate.lastUpdateTime
   if deltaTime <= 0:
      return ( rate.activityRecordsRate, rate.progressRecordsRate )
   p = movingAvgCoeff( deltaTime, rate.interval )
   delta = nucleusStatus.numActivityRecordsTransmitted - rate.lastActivityRecords
   activityRecordsAvg = calculateNewAverage( rate.activityRecordsRate,
                                             delta,
                                             deltaTime,
                                             p )
   delta = nucleusStatus.numProgressRecordsTransmitted - rate.lastProgressRecords
   progressRecordsAvg = calculateNewAverage( rate.progressRecordsRate,
                                             delta,
                                             deltaTime,
                                             p )
   sm = Tac.newInstance( "Arnet::KernelSockInfoSm" )
   ns = ( NamespaceName.defaultNamespace() if nucleusStatus.vrf == DEFAULT_VRF
          else f"ns-{nucleusStatus.vrf}" )
   bytesSent = sm.getBytesSent( nucleusStatus.addr, nucleusStatus.port, ns )
   delta = bytesSent - rate.lastBytesSent
   # bytes per second
   throughput = calculateNewAverage( rate.lastThroughput,
                                     delta,
                                     deltaTime,
                                     p )
   # convert to Mbps
   throughput = throughput * 8 / 1000000
   return ( activityRecordsAvg, progressRecordsAvg, throughput )

def updateNucleusCountersRate( nucleusCountersModel, nucleusStatus ):
   # Calculate exponential moving average rate based on last saved rate and
   # delta counter/time to get an approximate recent rate, but will not save to
   # nucleusStatus since only FlowWatcher has write access to nucleusStatus SHARK.
   # handleCountersRate() in FlowWatcher agent will update nucleusStatus SHARK
   # periodically.
   now = Tac.now()
   for interval in intervals:
      rate = nucleusStatus.rates.get( interval )
      if rate:
         ( nucleusCountersModel.activityRecordsRates[ interval ],
           nucleusCountersModel.progressRecordsRates[ interval ],
           nucleusCountersModel.throughputs[ interval ] ) = (
              calculateNucleusRate( now, nucleusStatus, rate ) )
      else:
         nucleusCountersModel.activityRecordsRates[ interval ] = 0.0
         nucleusCountersModel.progressRecordsRates[ interval ] = 0.0
         nucleusCountersModel.throughputs[ interval ] = 0.0

def addNucleusCounters( model, nucleusFilter=None ):
   for nucleusName, nucleusStatus in (
         CliPlugin.FlowWatcherCliLib.gv.nucleusStatusColl.status.items() ):
      if nucleusFilter and nucleusName != nucleusFilter:
         continue
      nucleusCounters = FlowWatcherModel.NucleusCounters()
      nucleusCounters.numActivityRecordsTransmitted = (
         nucleusStatus.numActivityRecordsTransmitted )
      nucleusCounters.lastActivityTransmissionTimestamp = switchTimeToUtc(
         nucleusStatus.lastActivityTransmissionTimestamp )
      nucleusCounters.numProgressRecordsTransmitted = (
         nucleusStatus.numProgressRecordsTransmitted )
      nucleusCounters.lastProgressTransmissionTimestamp = switchTimeToUtc(
         nucleusStatus.lastProgressTransmissionTimestamp )
      nucleusCounters.lastConnectionSuccessTimestamp = switchTimeToUtc(
         nucleusStatus.lastConnectionSuccessTimestamp )
      nucleusCounters.connectionSuccessCount = nucleusStatus.connectionSuccessCount
      nucleusCounters.lastConnectionErrorTimestamp = switchTimeToUtc(
         nucleusStatus.lastConnectionErrorTimestamp )
      nucleusCounters.connectionErrorCount = nucleusStatus.connectionErrorCount
      nucleusCounters.numActivityRecordsInQueue = (
         nucleusStatus.numActivityRecordsInQueue )
      nucleusCounters.numProgressRecordsInQueue = (
         nucleusStatus.numProgressRecordsInQueue )
      nucleusCounters.lastCountersClearedTime = switchTimeToUtc(
         nucleusStatus.lastCountersClearTimestamp )
      updateNucleusCountersRate( nucleusCounters, nucleusStatus )
      model.nucleusCounters[ nucleusName ] = nucleusCounters

def showMsaCountersHandler( mode, args ):
   model = FlowWatcherModel.MsaCounters()
   if not gv.fwStatus.enabled:
      return model
   if 'flows' in args:
      addFlowCounters( model )
   elif 'ipfix' in args:
      addIpfixCounters( model )
   elif 'nucleus' in args:
      nucleusFilter = args.get( 'NUCLEUS_NAME' )
      addNucleusCounters( model, nucleusFilter=nucleusFilter )
   else:
      addFlowCounters( model, withQueueFullCounter=True )
      addIpfixCounters( model )
      addNucleusCounters( model )
      model.packetBundleQueueFull = gv.fwStatus.packetBundleQueueFull
   return model

# ----------------------------------------------------------------------
# show monitor security awake flow-table [detail]
# [ group <group-name> ][ protocol <protocol> ]
# [ ip <ip> ] [ peer-ip <ip> ] [ port <port> ] [ peer-port <port> ]
# ----------------------------------------------------------------------

def showFlowTableHandler( mode, args ):
   if 'detail' in args:
      flowTableModel = FlowWatcherModel.FlowTableDetail()
   else:
      flowTableModel = FlowWatcherModel.FlowTable()
   if not gv.fwStatus.enabled:
      return flowTableModel
   # When MSA is enabled, NDRSensor.swix has been installed so we can use
   # ShowCliHelper
   SmashLazyMount.force( gv.flowTable )
   fd = sys.stdout.fileno()
   fmt = mode.session_.outputFormat()

   group = args.get( 'GROUP_NAME', 'ipunknown' )
   group = Tac.enumValue( 'Arnet::AddressFamily', group )

   protocol = args.get( 'PROTOCOL', 'Unknown' )
   protocol = Tac.enumValue( 'Arnet::IpProtocolNumber',
                            'ipProto' + protocol.capitalize() )

   ip = args.get( 'IP', '0.0.0.0' )
   ip = IpGenAddr( str( ip ) )
   peerIp = args.get( 'PEER_IP', '0.0.0.0' )
   peerIp = IpGenAddr( str( peerIp ) )
   port = args.get( 'PORT', 0 )
   peerPort = args.get( 'PEER_PORT', 0 )

   try:
      helper = Tac.newInstance( 'FlowWatcher::ShowCliHelper', gv.flowTable,
                                'detail' in args,
                                group, protocol, ip, peerIp, port, peerPort )
   except: # pylint: disable-msg=bare-except
      return flowTableModel
   with Tac.ActivityLockHolder():
      helper.render( fd, fmt )
   return cliPrinted( FlowWatcherModel.FlowTable )

# ----------------------------------------------------------------
# show monitor security awake dpi memory
# ----------------------------------------------------------------

def showMsaDpiMemoryHandler( mode, args ):
   model = FlowWatcherModel.MsaDpiMemory()
   if not gv.fwStatus.enabled:
      return model
   for memoryType in ( 'allocated', 'inUse' ):
      dpiMemory = gv.dpiStatus.dpiMemory.get( memoryType )
      if dpiMemory:
         memoryDetails = FlowWatcherModel.MsaDpiMemoryDetails()
         memoryDetails.current = dpiMemory.current
         memoryDetails.minimum = dpiMemory.minimum
         memoryDetails.maximum = dpiMemory.maximum
         memoryDetails.minimumTime = switchTimeToUtc( dpiMemory.minimumTime )
         memoryDetails.maximumTime = switchTimeToUtc( dpiMemory.maximumTime )
         setattr( model, memoryType, memoryDetails )
   return model

# ----------------------------------------------------------------
# show monitor security awake dpi memory allocations [ nz ] [ nt ]
# [ sort ( current | minimum | maximum ) ]
# ----------------------------------------------------------------

def showMsaDpiMemoryAllocationsHandler( mode, args ):
   sort = None
   if 'sort' in args:
      if 'current' in args:
         sort = 'current'
      elif 'minimum' in args:
         sort = 'minimum'
      else:
         sort = 'maximum'
   model = FlowWatcherModel.MsaDpiMemoryAllocations( _sort=sort,
                                                     _nt='nt' in args )
   if not gv.fwStatus.enabled:
      return model
   nonZero = 'nz' in args
   try:
      stringBuff = StringIO()
      AgentCommandRequest.runSocketCommand(
         mode.entityManager, 'FlowWatcher', 'dpiMemoryAllocations', '',
         throwException=True, stringBuff=stringBuff )
      outputValue = stringBuff.getvalue()
      stringBuff.close()
      jsonVal = json.loads( outputValue ) if outputValue else {}
      for ctxName, ctxAllocations in jsonVal.items():
         ctxMemoryAllocations = None
         for structName, structAllocations in ctxAllocations.items():
            if ( nonZero and structAllocations[ 'nA' ] == 0 and
                 structAllocations[ 'nB' ] == 0 ):
               continue
            structMemoryAllocations = FlowWatcherModel.MsaDpiStructMemoryAllocations(
               numAlloc=structAllocations[ 'nA' ],
               minAlloc=structAllocations[ 'minA' ],
               maxAlloc=structAllocations[ 'maxA' ],
               numBytes=structAllocations[ 'nB' ],
               minBytes=structAllocations[ 'minB' ],
               maxBytes=structAllocations[ 'maxB' ],
               minTime=switchTimeToUtc( structAllocations[ 'minT' ] ),
               maxTime=switchTimeToUtc( structAllocations[ 'maxT' ] ) )
            if not ctxMemoryAllocations:
               ctxMemoryAllocations = FlowWatcherModel.MsaDpiCtxMemoryAllocations()
            ctxMemoryAllocations.structures[ structName ] = structMemoryAllocations
         if ctxMemoryAllocations:
            model.contexts[ ctxName ] = ctxMemoryAllocations
   except ( AgentCommandRequest.RunSocketCommandException,
            ValueError ) as e:
      mode.addError( str( e ) )
   return model

# ----------------------------------------------------------------
# show monitor security awake dpi memory pools
# ----------------------------------------------------------------

def showMsaDpiMemoryPoolsHandler( mode, args ):
   model = FlowWatcherModel.MsaDpiMemoryPools()
   if not gv.fwStatus.enabled:
      return model
   for poolId, dpiMemoryPool in gv.dpiStatus.dpiMemoryPool.items():
      memoryPoolDetails = FlowWatcherModel.MsaDpiMemoryPoolDetails(
         blockSize=dpiMemoryPool.blockSize,
         allocCount=dpiMemoryPool.allocCount,
         allocSize=dpiMemoryPool.allocSize,
         usedCount=dpiMemoryPool.usedCount,
         usedSize=dpiMemoryPool.usedSize,
         minimumCount=dpiMemoryPool.minimumCount,
         minimumSize=dpiMemoryPool.minimumSize,
         minimumTime=switchTimeToUtc( dpiMemoryPool.minimumTime ),
         maximumCount=dpiMemoryPool.maximumCount,
         maximumSize=dpiMemoryPool.maximumSize,
         maximumTime=switchTimeToUtc( dpiMemoryPool.maximumTime ) )
      model.memoryPools[ poolId ] = memoryPoolDetails
   return model

# ----------------------------------------------------------------

def Plugin( em ):
   gv.fwConfig = LazyMount.mount( em, 'flowwatcher/config',
                                  'FlowWatcher::Config', 'r' )
   gv.fwStatus = LazyMount.mount( em, Cell.path( 'flowwatcher/status' ),
                                  'FlowWatcher::Status', 'r' )

   gv.ipLockingStatus = LazyMount.mount( em, 'iplocking/ipLockingStatus',
                                         'IpLocking::IpLockingStatus', 'r' )

   gv.ipfixCollectorCounters = (
      SmashLazyMount.mount( em,
                            'flowwatcher/ipfixCounters',
                            'IpfixCollector::IpfixCounters',
                            SmashLazyMount.mountInfo( 'reader' ) )
   )

   gv.dpiStatus = (
      SharkLazyMount.mount( em,
                            "flowwatcher/dpiStatus",
                            "Dpi::FlowWatcher::DpiStatus",
                            Shark.mountInfo( "shadow" ),
                            True, # autoUnmount
                            unmountTimeout=5 ) )

   gv.flowTable = SmashLazyMount.mount(
      em,
      "flowwatcher/flowTable",
      "Dpi::FlowWatcher::FlowTable",
      SmashLazyMount.mountInfo( 'reader' )
   )

   gv.extMgrStatus = LazyMount.mount(
      em, ExtensionMgrLib.statusPath(), 'Extension::Status', 'r' )

   for ftrType in ( ftrTypeHardware, ftrTypeSampled ):
      # hardware/flowtracking/config/<ftrType> [HwFlowTracking::Config]
      gv.hwTrackingConfig[ ftrType ] = LazyMount.mount(
         em, hwFtrConfigPathPrefix + ftrType, hwFtrConfigType, 'r' )
