# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Arnet import intfNameKey
from CliModel import Enum, List, Dict, Model
from IntfModels import Interface

class EgressInterface( Model ):
   __revision__ = 2
   logical = Interface( help="Logical egress interface" )
   physical = Interface( help="Physical egress interface" )

   def degrade( self, dictRepr, revision ):
      if revision == 1:
         dictRepr[ 'logicalEgressInterface' ] = dictRepr.pop( 'logical' )
         dictRepr[ 'physicalEgressInterface' ] = dictRepr.pop( 'physical' )
      return dictRepr

class PacketDestinationDetailModel( Model ):
   fields = Dict( keyType=str, valueType=str,
                  help="A mapping of detail field name to its value", optional=True )

class PacketDestination( Model ):
   __revision__ = 2
   status = Enum( values=[ 'success', 'timeout' ], help="Status of the request" )
   egressInterfaces = List( help="List of egress interfaces",
                            valueType=EgressInterface )
   details = Dict( keyType=str, valueType=PacketDestinationDetailModel,
                   help="Platform specific details on forwarding destination" )
   _sectionOrder = List( help="List to sort section headings", valueType=str )

   def render( self ):
      if self.status == 'success':
         def filterFunc( intf ):
            if intf.startswith( ( 'Ethernet', 'Vlan', 'Port-Channel', 'Switch' ) ):
               return intfNameKey( intf )
            else:
               # Not all interfaces fit the format intfNameKey expects. As an example
               # L3FloodFap0.0 on Sand platforms.
               return intf

         # Generate a map of physical egress intf -> set of logical egress intfs
         egressIntfs = {}
         for egressIntf in self.egressInterfaces:
            physicalIntf = egressIntf.physical.stringValue
            logicalIntf = egressIntf.logical.stringValue
            if physicalIntf not in egressIntfs:
               egressIntfs[ physicalIntf ] = set()
            if logicalIntf and ( logicalIntf != physicalIntf ):
               egressIntfs[ physicalIntf ].add( logicalIntf )

         # Generate output of egress intfs and their associated logical egress intfs
         combined = []
         for egressIntf in sorted( egressIntfs, key=filterFunc ):
            logicalIntfs = sorted( egressIntfs[ egressIntf ], key=filterFunc )
            newLine = egressIntf
            if logicalIntfs:
               newLine += ' (' + ', '.join( logicalIntfs ) + ')'
            combined.append( newLine )

         print( 'Egress interface(s): ' + ', '.join( combined ) )

         for heading in self._sectionOrder:
            print( '\n' + heading )
            detail = self.details[ heading ]
            for fieldName in sorted( detail.fields ):
               fieldValue = detail.fields[ fieldName ]
               print( fieldName + ': ' + fieldValue )

      elif self.status == 'timeout':
         print( 'The packet was not forwarded' )
