# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import sys, re
from HttpServiceConstants import ServerConstants
import CliCommand
from CliPlugin import AclCli
from CliPlugin import HttpServiceModel
from CliPlugin.HttpService import HttpServerConfigMode
from CliDynamicSymbol import CliDynamicPlugin
import CliGlobal
import ConfigMount
import LazyMount
import Tac


DynamicMode = CliDynamicPlugin( "HttpServiceVrf" )

# -----------------------------------------------------------------------------------
# Mount path holders ( Define all mount path holders here )
# -----------------------------------------------------------------------------------
gv = CliGlobal.CliGlobal(
   dict(
      capiConfig=None,
      serverStatus=None,
      aclCpConfig=None,
      aclStatus=None,
      aclCheckpoint=None,
   )
)


capiConstants = Tac.Type( "HttpService::Constants" )
logLevels = Tac.Type( 'HttpService::LogLevel' )
allOrigin = ServerConstants.ALLOW_ALL_CROSS_ORIGIN_DOMAINS
allOriginStar = ServerConstants.ALLOW_ALL_CROSS_ORIGIN_DOMAINS_STAR
aclServiceName = ServerConstants.serviceName

# -------------------------------------------------------------------------------
# Command Helper Function
# -------------------------------------------------------------------------------
def _setServerStatus( protocol, config, status, port=None ):
   config.enabled = status
   if port:
      config.port = port
      if protocol in ( 'http', 'https' ):
         portsStr = ','.join( str( i ) for i in [ gv.capiConfig.httpConfig.port,
                                                  gv.capiConfig.httpsConfig.port ] )
         for aclType in ( 'ip', 'ipv6' ):
            for serviceAclVrfConfig in \
                  gv.aclCpConfig.cpConfig[ aclType ].serviceAcl.values():
               serviceAclConfig = serviceAclVrfConfig.service.get( aclServiceName )
               if serviceAclConfig:
                  serviceAclConfig.ports = portsStr


def _checkProtocolsDefault():
   for protocol in ( "https", "http", "http localhost", "unix-socket" ):
      config, port, enabled = _getProtocolDefaults( protocol )
      if ( config.enabled != enabled or
           protocol != 'unix-socket' and config.port != port ):
         return False
   return True


# pylint: disable-next=inconsistent-return-statements
def _getProtocolDefaults( protocol ):
   if protocol == "https":
      return ( gv.capiConfig.httpsConfig, capiConstants.defaultSecurePort,
               capiConstants.defaultSecureEnabled )
   elif protocol == "http":
      return ( gv.capiConfig.httpConfig, capiConstants.defaultInsecurePort,
               capiConstants.defaultInsecureEnabled )
   elif protocol == "http localhost":
      return ( gv.capiConfig.localHttpConfig,
               capiConstants.defaultInsecureLocalPort,
               capiConstants.defaultInsecureLocalEnabled )
   elif protocol == "unix-socket":
      return ( gv.capiConfig.unixConfig, None, capiConstants.defaultUnixEnabled )


def setProtocolStatus( protocol, status, port=None ):
   config, defaultPort, _ = _getProtocolDefaults( protocol )
   _setServerStatus( protocol, config, status,
                     port if port and status else defaultPort )


def handleDefaultProtocol( protocol ):
   config, defaultPort, defaultStatus = _getProtocolDefaults( protocol )
   _setServerStatus( protocol, config, defaultStatus, defaultPort )


def _getProtocolType( args ): # pylint: disable=inconsistent-return-statements
   if "https" in args:
      return "https"
   elif "http" in args:
      if "localhost" in args:
         return "http localhost"
      else:
         return "http"


# Make sure server-configuring commands are in "api http-commands" mode xor
# 'http-server' mode.
def checkServerConfigNotSplit( mode ):
   errorStr = "Cannot configure server in mode '%s' because there is server "\
              "configuration in mode '%s'. Please use the same mode for server "\
              "configuration."
   if gv.capiConfig.useHttpServiceCli == isinstance( mode, HttpServerConfigMode ):
      return True
   elif gv.capiConfig.useHttpServiceCli:
      mode.addError( errorStr % ( 'api http-commands', 'http-server' ) )
      return False
   elif serverConfigurationExists():
      mode.addError( errorStr % ( 'http-server', 'api http-commands' ) )
      return False
   else:
      return True


# Check protocol, cert/key, SSL profile, cipher, QOS dscp, content frame
# ancestors, cors and log-level to see if there's existing server configuration
def serverConfigurationExists():
   httpsConfig = gv.capiConfig.httpsConfig
   if ( not _checkProtocolsDefault() or
        httpsConfig.sslKey or httpsConfig.sslCertificate or
        httpsConfig.sslProfile or
        httpsConfig.ciphers or httpsConfig.keyExchange or httpsConfig.mac or
        gv.capiConfig.qosDscp or
        gv.capiConfig.contentFrameAncestor or
        not gv.capiConfig.defaultServicesEnabled or
        gv.capiConfig.corsAllowedOrigins or
        gv.capiConfig.syslogLevel != gv.capiConfig.syslogLevelDefault ):
      return True
   return False


# Set useHttpServiceCli flag to true or false. This function assumes we have
# already run checkServerConfigNotSplit to decide if we're in a valid mode.
def setUseHttpServiceCli( mode ):
   if isinstance( mode, HttpServerConfigMode ):
      if serverConfigurationExists():
         gv.capiConfig.useHttpServiceCli = True
      else:
         gv.capiConfig.useHttpServiceCli = False


def server_config_split( function ):
   """
   Checks if the server-configuration would be split if the config got applied,
   apply config if it's okay, then set the useHttpServiceCli flag based on new config
   """
   def wrapper( *args, **kwargs ):
      mode = args[ 0 ]
      if not checkServerConfigNotSplit( mode ):
         return
      function( *args, **kwargs )
      setUseHttpServiceCli( mode )
   return wrapper


def serverNoVrf( mode, args ): # pylint: disable=useless-return
   def setServerNoVrf( vrf ):
      if vrf in gv.capiConfig.vrfConfig:
         gv.capiConfig.vrfConfig[ vrf ].serverState = 'globalDefault'
         if len( gv.capiConfig.vrfConfig[ vrf ].vrfService ) == 0:
            del gv.capiConfig.vrfConfig[ vrf ]

   vrfName = args.get( 'VRF' )
   # Clear http-server VRF configuration
   # This part is not used for now since we hide '[no|default] shutdown' command
   # in http-server VRF submode
   if vrfName:
      setServerNoVrf( vrfName )
   else:
      for currVrf in gv.capiConfig.vrfConfig:
         setServerNoVrf( currVrf )
   
   # Clear http-server service Acl configuration
   for aclType in ( 'ip', 'ipv6' ):
      vrfList = ( [ vrfName ] if vrfName 
                  else gv.aclCpConfig.cpConfig[ aclType ].serviceAcl )
      for vrf in vrfList:
         childMode = mode.childMode(DynamicMode.HttpServerVrfConfigMode, 
                                    vrfName=vrf )
         childMode.setServiceAcl( aclType, None, remove=True )
   return

# remove VRF configuration from service
def removeVrfFromService( config, serviceName, vrf=None ):
   def removeVrf( vrfName ):
      if vrfName in config.vrfConfig:
         del config.vrfConfig[ vrfName ].vrfService[ serviceName ]

         if ( len( config.vrfConfig[ vrfName ].vrfService ) == 0 and
               config.vrfConfig[ vrfName ].serverState == 'globalDefault' ):
            del config.vrfConfig[ vrfName ]

   if vrf:
      removeVrf( vrf )
   else:
      for currVrf in config.vrfConfig:
         removeVrf( currVrf )
# -------------------------------------------------------------------------------
# BasicCli Command Handler Function
# -------------------------------------------------------------------------------
def enterHttpServerModeHandler(mode, args):
   childMode = mode.childMode( HttpServerConfigMode )
   mode.session_.gotoChildMode( childMode )


def enterHttpServerModeNoOrDefaultHandler(mode, args):
   serverNoVrf( mode, {} )
   if gv.capiConfig.useHttpServiceCli:
      handleDefaultProtocol( "http" )
      handleDefaultProtocol( "https" )
      handleDefaultProtocol( "http localhost")
      handleDefaultProtocol( "unix-socket" )
      gv.capiConfig.defaultServicesEnabled = True
      gv.capiConfig.corsAllowedOrigins.clear()
      gv.capiConfig.httpsConfig.sslProfile = ''
      gv.capiConfig.qosDscp = 0
      gv.capiConfig.contentFrameAncestor = ""
      gv.capiConfig.syslogLevel = gv.capiConfig.syslogLevelDefault
   if not serverConfigurationExists():
      gv.capiConfig.useHttpServiceCli = False


def clearHttpServerAclCountersHandler(mode, args):
   aclType = 'ip' if 'ip' in args else 'ipv6'
   AclCli.clearServiceAclCounters( mode, gv.aclStatus, gv.aclCheckpoint, aclType )

# -------------------------------------------------------------------------------
# HttpServerConfigMode Command Handler Function
# -------------------------------------------------------------------------------

# remove VRF overridden configuration in http-server
def enterVrfConfigHandler( mode, args ):
   vrfName = args[ 'VRF' ]
   childMode = mode.childMode( DynamicMode.HttpServerVrfConfigMode, 
                               vrfName=vrfName )
   mode.session_.gotoChildMode( childMode )

def enterVrfConfigNoOrDefaultHandler( mode, args ): # pylint: disable=useless-return
   def setServerNoVrf( vrf ):
      if vrf in gv.capiConfig.vrfConfig:
         gv.capiConfig.vrfConfig[ vrf ].serverState = 'globalDefault'
         if len( gv.capiConfig.vrfConfig[ vrf ].vrfService ) == 0:
            del gv.capiConfig.vrfConfig[ vrf ]

   vrfName = args.get( 'VRF' )
   # Clear http-server VRF configuration
   # This part is not used for now since we hide '[no|default] shutdown' command
   # in http-server VRF submode
   if vrfName:
      setServerNoVrf( vrfName )
   else:
      for currVrf in gv.capiConfig.vrfConfig:
         setServerNoVrf( currVrf )
   
   # Clear http-server service Acl configuration
   for aclType in ( 'ip', 'ipv6' ):
      vrfList = ( [ vrfName ] if vrfName 
                  else gv.aclCpConfig.cpConfig[ aclType ].serviceAcl )
      for vrf in vrfList:
         childMode = mode.childMode( DynamicMode.HttpServerVrfConfigMode, 
                                     vrfName=vrf )
         childMode.setServiceAcl( aclType, None, remove=True )
   return

@server_config_split
def httpProtocolCmdHandler(mode, args):
   protocol = _getProtocolType( args )
   port = args[ "<number>" ] if "<number>" in args else None
   setProtocolStatus( protocol, True, port )

@server_config_split
def httpProtocolCmdNoOrDefaultHandler(mode, args):
   protocol = _getProtocolType( args )
   if CliCommand.isDefaultCmd( args ):
      handleDefaultProtocol( protocol )
   else:
      setProtocolStatus( protocol, False, None )


@server_config_split
def unixProtocolCmdHandler(mode, args):
   setProtocolStatus( "unix-socket", True, None )

@server_config_split
def unixProtocolCmdNoHandler(mode, args):
   setProtocolStatus( "unix-socket", False, None )

@server_config_split
def unixProtocolCmdDefaultHandler(mode, args):
   handleDefaultProtocol( "unix-socket" )

@server_config_split
def httpsProfileHandler(mode, args):
   gv.capiConfig.httpsConfig.sslProfile = args[ "<profileName>" ]

@server_config_split
def httpsProfileNoOrDefaultHandler(mode, args):
   gv.capiConfig.httpsConfig.sslProfile = ''

@server_config_split
def qosDscpHandler(mode, args):
   gv.capiConfig.qosDscp = args[ "<dscpValue>" ]

@server_config_split
def qosDscpNoOrDefaultHandler(mode, args):
   gv.capiConfig.qosDscp = 0


@server_config_split
def nginxSyslogHandler(mode, args):
   gv.capiConfig.syslogLevel = args[ "<severity>" ]

@server_config_split
def nginxSyslogNoOrDefaultHandler(mode, args):
   if CliCommand.isDefaultCmd( args ):
      gv.capiConfig.syslogLevel = gv.capiConfig.syslogLevelDefault
   else:
      gv.capiConfig.syslogLevel = logLevels.none

@server_config_split
def defaultServicesHandler(mode, args):
   gv.capiConfig.defaultServicesEnabled = True

@server_config_split
def defaultServicesNoHandler(mode, args):
   gv.capiConfig.defaultServicesEnabled = False

@server_config_split
def frameAncestorsHandler(mode, args):
   gv.capiConfig.contentFrameAncestor = args[ "<uri>" ]

@server_config_split
def frameAncestorsNoOrDefaultHandler(mode, args):
   gv.capiConfig.contentFrameAncestor = ""

@server_config_split
def corsOriginsHandler(mode, args):
   if allOrigin in args or allOriginStar in args:
      origin = allOrigin
   else:
      origin = args[ "<originString>" ]

   gv.capiConfig.corsAllowedOrigins[ origin ] = True

@server_config_split
def corsOriginsNoOrDefaultHandler(mode, args):
   if "<corsName>" not in args:
      gv.capiConfig.corsAllowedOrigins.clear()
      return
   originString = args[ "<corsName>" ]
   # pylint: disable-next=consider-using-in
   if ( ( originString == allOrigin or originString == allOriginStar )
         and allOrigin in gv.capiConfig.corsAllowedOrigins ):
      del gv.capiConfig.corsAllowedOrigins[ allOrigin ]
   elif originString in gv.capiConfig.corsAllowedOrigins:
      del gv.capiConfig.corsAllowedOrigins[ originString.lower() ]
   else:
      # pylint: disable-next=consider-using-f-string
      mode.addError( "'%s' is not in allowed origins" % originString )

# -------------------------------------------------------------------------------
# HttpServerVrfConfigMode Command Handler Function
# -------------------------------------------------------------------------------
def vrfModeShutDownHandler(mode, args):
   mode.shutdown()

def vrfModeShutDownNoHandler(mode, args):
   mode.noShutdown()

def vrfModeShutDownDefaultHandler(mode, args):
   mode.defaultShutdown()

def doSetIpAclCmd(mode, args):
   mode.setIpAcl(args)

def noIpAclCmd(mode, args):
   mode.noIpAcl(args)

def doSetIp6AclCmd(mode, args):
   mode.setIp6Acl(args)

def noIp6AclCmd(mode, args):
   mode.noIp6Acl(args)

# -------------------------------------------------------------------------------
# Show Command Handler Function
# -------------------------------------------------------------------------------
def showManagementHttpServerAclHandler(mode, args):
   # Note: the name argument is actually a tuple( name, summary )
   aclType = 'ip' if 'ip' in args else 'ipv6'
   return AclCli.showServiceAcl( mode,
                                 gv.aclCpConfig,
                                 gv.aclStatus,
                                 gv.aclCheckpoint,
                                 aclType,
                                 args[ '<aclNameExpr>' ],
                                 serviceName=ServerConstants.serviceName )

def showHttpServerStatsHandler( mode, args ):
   cmdPrefix = [ 'curl', '-sS', '--unix-socket' ]
   ret = HttpServiceModel.HttpServerStats()
   for ( vrf, status ) in gv.serverStatus.vrfStatus.items():
      # only show enabled VRFs
      if status.enabled:
         currVrfStats = HttpServiceModel.VrfStats()
         execCmd = cmdPrefix + [ ServerConstants.NGINX_STATUS_UDS % vrf,
                                 ServerConstants.NGINX_STATUS_ENDPOINT ]
         output = ""
         try:
            output = Tac.run( execCmd, stdout=Tac.CAPTURE,
                              stderr=sys.stderr, asRoot=True )
         except Tac.SystemCommandError as e:
            # pylint: disable-next=consider-using-f-string
            mode.addWarning( "Fail to get counter info of vrf %s due to %s"
                             % ( vrf, e.output ) )
            continue
         result = [ int( v ) for v in re.findall( r'\d+', output ) ]
         if not result: # pylint: disable=no-else-continue
            # pylint: disable-next=consider-using-f-string
            mode.addWarning( "Fail to parse counter info of vrf %s output" % vrf )
            continue
         else:
            assert len( result ) == 7
            currVrfStats.maxConnections = 1024
            currVrfStats.activeConnections = result[ 0 ]
            currVrfStats.acceptConnections = result[ 1 ]
            currVrfStats.handledConnections = result[ 2 ]
            currVrfStats.refusedConnections = result[ 1 ] - result[ 2 ]
            currVrfStats.requests = result[ 3 ]
            currVrfStats.readingConnections = result[ 4 ]
            currVrfStats.writingConnections = result[ 5 ]
            currVrfStats.waitingConnections = result[ 6 ]
            ret.vrfs[ vrf ] = currVrfStats

   return ret


# Plug-in definition:
def Plugin( entityManager ):
   gv.capiConfig = ConfigMount.mount( entityManager,
                                      "mgmt/capi/config",
                                      "HttpService::Config",
                                      "w" )
   gv.serverStatus = LazyMount.mount( entityManager,
                                      "mgmt/httpserver/status",
                                      "HttpService::Status",
                                      "r" )
   gv.aclCpConfig = ConfigMount.mount( entityManager,
                                       "acl/cpconfig/cli",
                                       "Acl::Input::CpConfig",
                                       "w" )
   gv.aclStatus = LazyMount.mount( entityManager,
                                   "acl/status/all",
                                   "Acl::Status",
                                   "r" )
   gv.aclCheckpoint = LazyMount.mount( entityManager,
                                       "acl/checkpoint",
                                       "Acl::CheckpointStatus",
                                       "w" )