#!/usr/bin/env python3
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
import Tac
import Ethernet
import datetime
from CliModel import Model, Dict, List, Bool, Str, Int, Enum
from IntfModels import Interface
from CliPlugin.IntfCli import Intf
import Toggles.IpLockingToggleLib as ILTL
from ArnetModel import IpGenericAddress, MacAddress
from TableOutput import TableFormatter, Headings, Format, terminalWidth
from Intf.IntfRange import intfListToCanonical
from Vlan import vlanSetToCanonicalString
from IpLockingLib import (
   getAssignedVlansAsStrings,
   getAssignedVlansAsStringsRaw,
   getAssignedVlansAsStringsUnenforced,
   getLeaseQueryStr,
   LeaseTuple,
   LeaseTupleForCompare,
   intfIsLag
)

# -------------------------------------------------------------------------------
#
# cmd: show address locking
#
# -------------------------------------------------------------------------------
ipv4Mode = "IPv4 address locking mode"
ipv6Mode = "IPv6 address locking mode"
ipv4InactiveHelpStr = "Reason that Address Locking is not enabled for IPv4"
ipv6InactiveHelpStr = "Reason that Address Locking is not enabled for IPv6"

class IpLockingVlanEnabled( Model ):
   ipv4Enabled = Bool( help="Address Locking is enabled on VLAN for IPv4" )
   ipv4InactiveReason = Str( help=ipv4InactiveHelpStr, optional=True )
   ipv6Enabled = Bool( help="Address Locking is enabled on VLAN for IPv6" )
   ipv6InactiveReason = Str( help=ipv6InactiveHelpStr, optional=True )
   ipv4Mode = Str( help=ipv4Mode, optional=True )
   ipv6Mode = Str( help=ipv6Mode, optional=True )

class IpLockingIntfEnabled( Model ):
   ipv4Enabled = Bool( help="Address Locking is enabled on interface for IPv4" )
   ipv4Mode = Str( help=ipv4Mode, optional=True )
   ipv4InactiveReason = Str( help=ipv4InactiveHelpStr, optional=True )
   ipv6Enabled = Bool( help="Address Locking is enabled on interface for IPv6" )
   ipv6Mode = Str( help=ipv6Mode, optional=True )
   ipv6InactiveReason = Str( help=ipv6InactiveHelpStr, optional=True )
   assignedVlans = Dict( keyType=int,
         valueType=IpLockingVlanEnabled,
         help="Mapping of VLANs to which the interface is assigned, to "
              "the mode in which they are enabled for address locking" )

class IpLockingProtocol( Model ):
   protocol = Enum( values=( "leaseQuery", "arIpQuery", "arIpv6Query" ),
                    help="Protocol used to query server" )

def printServers( model ):
   print( "DHCP servers:" )
   headings = ( ( "Address", "lh" ),
                ( "Query Protocol", "l" ), )
   t = TableFormatter()
   th = Headings( headings )
   th.doApplyHeaders( t )
   f1 = Format( justify='left', wrap=False, minWidth=1 )
   f2 = Format( justify='left', wrap=True, minWidth=1 )
   for server in model.dhcpV4Servers:
      if model.dhcpV4Servers[ server ].protocol == 'arIpQuery':
         t.newRow( server, "Arista IP Query" )
      else:
         t.newRow( server, "Lease Query" )
   for server in model.dhcpV6Servers:
      t.newRow( server, "Arista IPv6 Query" )
   t.formatColumns( f1, f2 )
   print( t.output() )

class IpLocking( Model ):
   active = Bool( help="IP Locking active state" )
   inactiveReason = Str( help="Reason that IP Locking is inactive", optional=True )
   retryInterval = Int( help="Configured Lease Query Retry Interval" )
   timeout = Int( help="Configured Lease Query Timeout" )
   if ILTL.toggleIpLockingArIpEnabled():
      dhcpV4Servers = Dict( keyType=IpGenericAddress,
         valueType=IpLockingProtocol,
         help=( "Mapping of servers queried for DHCP v4 leases to the "
                "protocol used to query them" ) )
      dhcpV6Servers = Dict( keyType=IpGenericAddress,
         valueType=IpLockingProtocol,
         help=( "Mapping of servers queried for DHCP v6 leases to the "
                "protocol used to query them" ) )
   enabledIntfs = Dict( keyType=Interface,
         valueType=IpLockingIntfEnabled,
         help=( "A mapping of interfaces to their IPv4 and IPv6 address locking "
                "enabled status" ) )
   configuredIpv4Intfs = List(
         valueType=Interface,
         help="List of interfaces configured with address locking IPv4" )
   configuredIpv6Intfs = List(
         valueType=Interface,
         help="List of interfaces configured with address locking IPv6" )
   enabledVlans = Dict( keyType=int,
         valueType=IpLockingVlanEnabled,
         help=( "A mapping of VLANs to their IPv4 and IPv6 address locking "
                "enabled status" ) )
   configuredIpv4Vlans = List(
         valueType=int,
         help="List of VLANs configured with address locking IPv4" )
   configuredIpv6Vlans = List(
         valueType=int,
         help="List of VLANs configured with address locking IPv6" )
   ipv4Enforced = Bool(
         help="Locked-address enforcement is enabled for IPv4" )
   ipv6Enforced = Bool(
         help="Locked-address enforcement is enabled for IPv6" )

   def render( self ):
      active = self.active
      activeStr = 'active' if active else 'inactive'
      fullActiveStr = f'IP Locking is {activeStr}'
      if not active:
         fullActiveStr += f' ({self.inactiveReason})'
      print( fullActiveStr )

      if self.timeout:
         queryTimeoutInfoStr = getLeaseQueryStr( self.retryInterval, self.timeout )
         print( queryTimeoutInfoStr )

      if ILTL.toggleIpLockingArIpEnabled():
         print( '' )
         printServers( self )

      # TODO: Coallesce interface into ranges.
      t = TableFormatter()
      f1 = Format( justify='left', wrap=False, maxWidth=29 )
      f1.noPadLeftIs( True )
      f2 = Format( justify='left', wrap=True, minWidth=1 )
      f2.noPadLeftIs( True )

      if self.configuredIpv4Intfs:
         spaceLeft = terminalWidth() - len( "Configured IPv4 Interfaces: " )
         ipv4IntfStr = ", ".join( intfListToCanonical( self.configuredIpv4Intfs,
                                                       strLimit=spaceLeft ) )
         t.newRow( 'Configured IPv4 Interfaces:', ipv4IntfStr )
      else:
         t.newRow( 'Configured IPv4 Interfaces:' )

      if self.configuredIpv6Intfs:
         spaceLeft = terminalWidth() - len( "Configured IPv6 Interfaces: " )
         ipv6IntfStr = ", ".join( intfListToCanonical( self.configuredIpv6Intfs,
                                                       strLimit=spaceLeft ) )
         t.newRow( 'Configured IPv6 Interfaces:', ipv6IntfStr )
      else:
         t.newRow( 'Configured IPv6 Interfaces:' )

      if self.configuredIpv4Vlans:
         spaceLeft = terminalWidth() - len( "Configured IPv4 VLANs: " )
         ipv4VlanStr = vlanSetToCanonicalString( self.configuredIpv4Vlans )
         t.newRow( 'Configured IPv4 VLANs:', ipv4VlanStr )
      else:
         t.newRow( 'Configured IPv4 VLANs:' )

      if self.configuredIpv6Vlans:
         spaceLeft = terminalWidth() - len( "Configured IPv6 VLANs: " )
         ipv6VlanStr = vlanSetToCanonicalString( self.configuredIpv6Vlans )
         t.newRow( 'Configured IPv6 VLANs:', ipv6VlanStr )
      else:
         t.newRow( 'Configured IPv6 VLANs:' )

      t.formatColumns( f1, f2 )
      print( t.output() )
      f1 = Format( justify='left' )
      f1.noPadLeftIs( True )
      f2 = Format( justify='left' )
      f2.noPadLeftIs( True )
      dispEnforcementLegend = False
      if active:
         if self.enabledIntfs:
            print( "Interface Status" )
            t = TableFormatter( tableWidth=115 )
            headings = ( ( "Interface", "lh" ),
                         ( "IPv4", "l" ),
                         ( "IPv6", "l" ), )
            th = Headings( headings )
            th.doApplyHeaders( t )
            for intf in Arnet.sortIntf( self.enabledIntfs ):
               ipLockingIntfEnabled = self.enabledIntfs[ intf ]
               assignedVlans = None
               assignedVlans = ipLockingIntfEnabled.assignedVlans
               t.startRow()
               t.newCell( intf )
               if ipLockingIntfEnabled.ipv4Mode == "enforcementEnabled":
                  ipv4Str = 'yes'
                  ipv4Str += getAssignedVlansAsStrings( assignedVlans, False,
                                                        'ipv4Mode' )
               elif ipLockingIntfEnabled.ipv4Mode == "enforcementDisabled":
                  ipv4Str = 'yes*'
               elif ipLockingIntfEnabled.ipv4Mode == "fullyDisabled":
                  ipv4Str = f'no ({ipLockingIntfEnabled.ipv4InactiveReason})'
               elif ( ipLockingIntfEnabled.ipv4Mode ==
                      "activeVlanEnforcementDisabled" ):
                  ipv4Str = getAssignedVlansAsStringsUnenforced( assignedVlans )
                  if not ipv4Str:
                     ipv4Str = 'no (not configured)'
               elif not ipLockingIntfEnabled.ipv4Enabled:
                  if intfIsLag( intf ):
                     ipv4Str = 'no (not supported)'
                     assignedVlanStrs = getAssignedVlansAsStringsRaw( assignedVlans,
                                                                      False,
                                                                      'ipv4Mode' )
                     ipv4Str = assignedVlanStrs if assignedVlanStrs else ipv4Str
                  else:
                     ipv4Str = f'no ({ipLockingIntfEnabled.ipv4InactiveReason})'
                     assignedVlanStrs = getAssignedVlansAsStrings( assignedVlans,
                                                                   True,
                                                                   'ipv4Mode' )
                     ipv4Str = assignedVlanStrs if assignedVlanStrs else ipv4Str
               if ( not ILTL.toggleIpLockingArIpEnabled() and
                    not self.ipv6Enforced ):
                  ipv6Str = f'no ({ipLockingIntfEnabled.ipv6InactiveReason})'
               elif ipLockingIntfEnabled.ipv6Mode == "enforcementEnabled":
                  ipv6Str = 'yes'
                  ipv6Str += getAssignedVlansAsStrings( assignedVlans, False,
                                                        'ipv6Mode' )
               elif ipLockingIntfEnabled.ipv6Mode == "enforcementDisabled":
                  ipv6Str = 'yes*'
               elif ipLockingIntfEnabled.ipv6Mode == "fullyDisabled":
                  ipv6Str = f'no ({ipLockingIntfEnabled.ipv6InactiveReason})'
               elif not ipLockingIntfEnabled.ipv6Enabled:
                  ipv6Str = f'no ({ipLockingIntfEnabled.ipv6InactiveReason})'
                  assignedVlanStrs = getAssignedVlansAsStrings( assignedVlans,
                                                                True, 'ipv6Mode' )
                  ipv6Str = assignedVlanStrs if assignedVlanStrs else ipv6Str
               t.newCell( ipv4Str )
               t.newCell( ipv6Str )
               if '*' in ipv4Str or '*' in ipv6Str:
                  dispEnforcementLegend = True
            t.formatColumns( f1, f2, f2 )
            print( t.output() )

         if self.enabledVlans:
            print( "VLAN Status" )
            t = TableFormatter( tableWidth=100 )
            headings = ( ( "VLAN", "lh" ),
                         ( "IPv4", "l" ),
                         ( "IPv6", "l" ), )
            th = Headings( headings )
            th.doApplyHeaders( t )
            for vlan in sorted( self.enabledVlans ):
               ipLockingVlanEnabled = self.enabledVlans[ vlan ]
               t.startRow()
               t.newCell( vlan )
               if ipLockingVlanEnabled.ipv4Mode == "enforcementEnabled":
                  ipv4Str = 'yes'
               elif ipLockingVlanEnabled.ipv4Mode == "enforcementDisabled":
                  ipv4Str = 'yes*'
               elif not ipLockingVlanEnabled.ipv4Enabled:
                  ipv4Str = f'no ({ipLockingVlanEnabled.ipv4InactiveReason})'
               if ( not ILTL.toggleIpLockingArIpEnabled() and
                    not self.ipv6Enforced ):
                  ipv6Str = f'no ({ipLockingVlanEnabled.ipv6InactiveReason})'
               else:
                  if ipLockingVlanEnabled.ipv6Mode == "enforcementEnabled":
                     ipv6Str = 'yes'
                  elif ( ipLockingVlanEnabled.ipv6Mode ==
                         "enforcementDisabled" ):
                     ipv6Str = 'yes*'
                  elif not ipLockingVlanEnabled.ipv6Enabled:
                     ipv6Str = f'no ({ipLockingVlanEnabled.ipv6InactiveReason})'
               t.newCell( ipv4Str )
               t.newCell( ipv6Str )
               if '*' in ipv4Str or '*' in ipv6Str:
                  dispEnforcementLegend = True
            t.formatColumns( f1, f2, f2 )
            print( t.output() )

      if dispEnforcementLegend:
         print( '* Locked address enforcement is disabled' )

# -------------------------------------------------------------------------------
#
# cmd: show address locking

class IpLockingServers( Model ):
   dhcpV4Servers = Dict( keyType=IpGenericAddress,
         valueType=IpLockingProtocol,
         help=( "Mapping of servers queried for DHCP v4 leases to the "
                "protocol used to query them" ) )
   dhcpV6Servers = Dict( keyType=IpGenericAddress,
         valueType=IpLockingProtocol,
         help=( "Mapping of servers queried for DHCP v6 leases to the "
                "protocol used to query them" ) )

   def render( self ):
      printServers( self )

class Lease( Model ):
   clientMac = MacAddress( help="Client's MAC address" )
   source = Enum( values=( "config", "server" ),
               help="lease source type" )
   installed = Bool( help="This lease is installed in hardware" )
   expirationTime = Int( help="Lease expiration time in UTC" )
   overwritten = Bool( help="Lease overwritten as IP is denied on interface",
                       optional=True )

class IntfLeases( Model ):
   leases = Dict( keyType=IpGenericAddress,
                  valueType=Lease,
                  help="A mapping of a lease's IP address to its information" )

class LeasesTable( Model ):
   intfLeases = Dict( keyType=Interface,
      valueType=IntfLeases,
      help="A mapping of an interface to its leases" )
   denyLeases = Dict( keyType=Interface,
      valueType=IntfLeases,
      help="A mapping of an interface to its denied leases" )

   def render( self ):
      t = TableFormatter()
      headings = ( ( "IP Address", "lh" ),
                   ( "MAC Address", "l" ),
                   ( 'Source', "l" ),
                   ( "Interface", "l" ),
                   ( "Installed", "l" ),
                   ( "Expiration Time", "l" ) )
      headings += ( ( "Action", "l" ), )
      th = Headings( headings )
      th.doApplyHeaders( t )
      leases = []
      # Generate a flattened list of leases.
      savedUtcTime = Tac.utcNow()
      permitRuleOverride = False
      for intfId, denyLease in self.denyLeases.items():
         for ip, lease in denyLease.leases.items():
            leases.append(
               LeaseTuple(
                  ip=ip, mac=lease.clientMac, source=lease.source,
                  intf=intfId, installed=lease.installed,
                  expirationTime=lease.expirationTime, action='deny' ) )
      for intfId, intfLease in self.intfLeases.items():
         for ip, lease in intfLease.leases.items():
            # ExpirationTime in lease is the system time that lease will expire. We
            # want to display the number of seconds that this lease will expire.
            # Negative expirationTime due to lease expire and is in the process
            # of requery should be display as 0.
            expirationTime = max( 0, int( lease.expirationTime - savedUtcTime ) )
            action = 'permit'
            installed = lease.installed
            if lease.overwritten:
               action = 'permit*'
               installed = False
               permitRuleOverride = True
            leases.append(
               LeaseTuple(
                  ip=ip, mac=lease.clientMac, source=lease.source,
                  intf=intfId, installed=installed,
                  expirationTime=expirationTime, action=action ) )
      leases = sorted( leases, key=LeaseTupleForCompare )

      macZero = Tac.Type( "Arnet::EthAddr" ).ethAddrZero
      for lease in leases:
         t.startRow()
         t.newCell( lease.ip )
         macStr = Ethernet.convertMacAddrCanonicalToDisplay( lease.mac.stringValue )
         if lease.mac.stringValue == macZero:
            macStr = '-'
         t.newCell( macStr )
         t.newCell( lease.source )
         intfStr = lease.intf
         if not intfStr:
            intfStr = '-'
         t.newCell( Intf.getShortname( intfStr ) )
         installedString = "installed" if lease.installed else "not installed"
         t.newCell( installedString )
         # Do not display the lease expiration time for the static lease since static
         # lease will not expire.
         if lease.source == 'server':
            timeString = 'in '
            timeString += str( datetime.timedelta( seconds=lease.expirationTime ) )
         else:
            timeString = '-'
         t.newCell( timeString )
         t.newCell( lease.action )
      print( t.output() )
      print( f"Total IP Address for this criterion: {len( leases )}" )
      if permitRuleOverride:
         print( "* Lease permit rule overridden because lease IP is denied" )
