# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import LazyMount
import SharedMem
import Smash
from CliDynamicSymbol import loadDynamicPlugin
from CliPlugin.IpLockingCliConstants import (
   inactiveReasonHwNotSupported,
   inactiveReasonDisabled,
   inactiveReasonLocalIntfNotConfigured,
   inactiveReasonV4ServerNotConfigured,
   inactiveReasonArpInspectionConfigured,
   inactiveReasonIpsgConfigured,
   inactiveReasonRelayConfigured,
   inactiveReasonDhcpSnoopingConfigured,
   inactiveReasonDhcpServerConfigured,
   inactiveReasonNoInterfaceConfigured,
   inactiveReasonNoInterfaceOrVlanConfigured,
   inactiveReasonV6EnforcementEnabled,
   inactiveReasonV4EnforcementDisabled,
   )
import Toggles.IpLockingToggleLib as ILTL
import Tac
from EosDhcpServerLib import dhcpServerActive

# globals

cliConfig = None
cliStatus = None
hwConfig = None
enabledInterfaceStatus = None
enabledVlanStatus = None
leaseStatus = None
installedConfig = None
arpInspectionStatus = None
ipsgConfig = None
dhcpRelayStatus = None
dhcpSnoopingStatus = None
dhcpSnooping6Status = None
dhcpServerVrfStatus = None
intfVlanListMap = None
fullyDisabled = 'disabled'
notConfigured = 'not configured'
notLayer2 = 'not an active layer 2 interface'
notOperational = 'not operational'
notSupported = 'not supported'
noDhcpServer = 'no dhcp server'
noLocalIntf = 'no local interface'

# -------------------------------------------------------------------------------
# IpLocking show commands helper functions
# -------------------------------------------------------------------------------
def getIpLockingModels():
   return loadDynamicPlugin( "IpLockingModel" )

def getIpLockingCounterModels():
   return loadDynamicPlugin( "IpLockingCounterModel" )

def getIntfV4Mode( intf ):
   if cliStatus.getV4IntfReasonString( intf ) == notLayer2:
      return "invalid"
   # If the below condition is true, then it implies that the interface is
   # configured by its assignment to some VLAN.
   if intf not in cliConfig.interface:
      return "invalid"
   intfConfig = cliConfig.interface[ intf ]
   if intfConfig.ipv4:
      if ( intfConfig.ipv4LockedAddressEnforcementDisabled or
           cliConfig.ipv4LockedAddressEnforcementDisabled ):
         if hwConfig.v4EnforcementDisabledSupported:
            return "enforcementDisabled"
         else:
            return "invalid"
      elif ( cliConfig.localInterface != cliConfig.localInterfaceDefault and
             bool( cliConfig.dhcpV4Server ) ):
         return "enforcementEnabled"
   elif intfConfig.ipv4Disabled:
      return "fullyDisabled"
   elif ( hwConfig.v4EnforcementDisabledSupported and
          intfConfig.ipv4LockedAddressEnforcementDisabled ):
      return "activeVlanEnforcementDisabled"
   return "invalid"

def getIntfV6Mode( intf ):
   if cliStatus.getV6IntfReasonString( intf ) == notLayer2:
      return "invalid"
   # If the below condition is true, then it implies that the interface is
   # configured by its assignment to some VLAN.
   if intf not in cliConfig.interface:
      return "invalid"
   intfConfig = cliConfig.interface[ intf ]
   if intfConfig.ipv6:
      if cliConfig.ipv6LockedAddressEnforcementDisabled:
         return "enforcementDisabled"
      elif ( cliConfig.localInterface != cliConfig.localInterfaceDefault and
             bool( cliConfig.dhcpV6Server ) ):
         return "enforcementEnabled"
   elif intfConfig.ipv6Disabled:
      return "fullyDisabled"
   return "invalid"

def getVlanV4Mode( vlan ):
   if cliStatus.getV4VlanReasonString( vlan ) == notOperational:
      return "invalid"
   vlanConfig = cliConfig.vlan[ vlan ]
   if vlanConfig.ipv4:
      if ( vlanConfig.ipv4LockedAddressEnforcementDisabled or
           cliConfig.ipv4LockedAddressEnforcementDisabled ):
         if hwConfig.v4EnforcementDisabledSupported:
            return "enforcementDisabled"
         else:
            return "invalid"
      elif ( cliConfig.localInterface != cliConfig.localInterfaceDefault and
             bool( cliConfig.dhcpV4Server ) ):
         return "enforcementEnabled"
   return "invalid"

def getVlanV6Mode( vlan ):
   if cliStatus.getV6VlanReasonString( vlan ) == notOperational:
      return "invalid"
   vlanConfig = cliConfig.vlan[ vlan ]
   if vlanConfig.ipv6:
      if cliConfig.ipv6LockedAddressEnforcementDisabled:
         return "enforcementDisabled"
      elif ( cliConfig.localInterface != cliConfig.localInterfaceDefault and
             bool( cliConfig.dhcpV6Server ) ):
         return "enforcementEnabled"
   return "invalid"

def getIntfV4InactiveReason( intf, ipv4Enabled ):
   ipv4InactiveReason = None
   if not ipv4Enabled:
      if intf not in cliConfig.interface:
         return notConfigured
      intfConfig = cliConfig.interface[ intf ]
      if not intfConfig.ipv4:
         if intfConfig.ipv4Disabled:
            ipv4InactiveReason = fullyDisabled
         else:
            ipv4InactiveReason = notConfigured
      elif cliStatus.getV4IntfReasonString( intf ) == notLayer2:
         ipv4InactiveReason = notLayer2
      elif ( not intfConfig.ipv4LockedAddressEnforcementDisabled and
             not cliConfig.ipv4LockedAddressEnforcementDisabled ):
         if cliConfig.localInterface == cliConfig.localInterfaceDefault:
            ipv4InactiveReason = noLocalIntf
         elif not cliConfig.dhcpV4Server:
            ipv4InactiveReason = noDhcpServer
      elif not hwConfig.v4EnforcementDisabledSupported:
         ipv4InactiveReason = notSupported
   return ipv4InactiveReason

def getIntfV6InactiveReason( intf, ipv6Enabled ):
   ipv6InactiveReason = None
   if not ipv6Enabled:
      if intf not in cliConfig.interface:
         return notConfigured
      intfConfig = cliConfig.interface[ intf ]
      if not intfConfig.ipv6:
         if intfConfig.ipv6Disabled:
            ipv6InactiveReason = fullyDisabled
         else:
            ipv6InactiveReason = notConfigured
      elif cliStatus.getV6IntfReasonString( intf ) == notLayer2:
         ipv6InactiveReason = notLayer2
      elif not cliConfig.ipv6LockedAddressEnforcementDisabled:
         if cliConfig.localInterface == cliConfig.localInterfaceDefault:
            ipv6InactiveReason = noLocalIntf
         elif not cliConfig.dhcpV6Server:
            ipv6InactiveReason = noDhcpServer
   return ipv6InactiveReason

def getVlanV4InactiveReason( vlan, ipv4Enabled ):
   ipv4InactiveReason = None
   if not ipv4Enabled:
      vlanConfig = cliConfig.vlan[ vlan ]
      if not vlanConfig.ipv4:
         ipv4InactiveReason = notConfigured
      elif cliStatus.getV4VlanReasonString( vlan ) == notOperational:
         ipv4InactiveReason = notOperational
      elif ( not vlanConfig.ipv4LockedAddressEnforcementDisabled and
             not cliConfig.ipv4LockedAddressEnforcementDisabled ):
         if cliConfig.localInterface == cliConfig.localInterfaceDefault:
            ipv4InactiveReason = noLocalIntf
         elif not cliConfig.dhcpV4Server:
            ipv4InactiveReason = noDhcpServer
      elif not hwConfig.v4EnforcementDisabledSupported:
         ipv4InactiveReason = notSupported
   return ipv4InactiveReason

def getVlanV6InactiveReason( vlan, ipv6Enabled ):
   ipv6InactiveReason = None
   if not ipv6Enabled:
      vlanConfig = cliConfig.vlan[ vlan ]
      if not vlanConfig.ipv6:
         ipv6InactiveReason = notConfigured
      elif cliStatus.getV6VlanReasonString( vlan ) == notOperational:
         ipv6InactiveReason = notOperational
      elif not cliConfig.ipv6LockedAddressEnforcementDisabled:
         if cliConfig.localInterface == cliConfig.localInterfaceDefault:
            ipv6InactiveReason = noLocalIntf
         elif not cliConfig.dhcpV6Server:
            ipv6InactiveReason = noDhcpServer
   return ipv6InactiveReason

def doShowIpLocking( mode, args ):
   config = cliConfig
   active = cliConfig.active
   inactiveReason = ''
   ipLockingModels = getIpLockingModels()

   retryInterval = cliConfig.queryTimeoutInfo.retryInterval
   timeout = cliConfig.queryTimeoutInfo.timeout

   configuredIpv4Intfs, configuredIpv6Intfs = [], []
   configuredIpv4Vlans, configuredIpv6Vlans = [], []

   for intf in cliConfig.interface:
      if cliConfig.interface[ intf ].ipv4:
         configuredIpv4Intfs.append( intf )
      if cliConfig.interface[ intf ].ipv6:
         configuredIpv6Intfs.append( intf )

   for vlan in cliConfig.vlan:
      if cliConfig.vlan[ vlan ].ipv4:
         configuredIpv4Vlans.append( vlan )
      if cliConfig.vlan[ vlan ].ipv6:
         configuredIpv6Vlans.append( vlan )

   if not hwConfig.featureSupported:
      active = False
      inactiveReason = inactiveReasonHwNotSupported
   elif ( cliConfig.ipv4LockedAddressEnforcementDisabled and
          not hwConfig.v4EnforcementDisabledSupported and
          not enabledInterfaceStatus.ipv6 and
          not enabledVlanStatus.ipv6 ):
      active = False
      inactiveReason = inactiveReasonV4EnforcementDisabled
   elif arpInspectionStatus.enabled:
      active = False
      inactiveReason = inactiveReasonArpInspectionConfigured
   elif ipsgConfig.ipsgEnabledIntf:
      active = False
      inactiveReason = inactiveReasonIpsgConfigured
   elif ( dhcpRelayStatus.runControl and
          ( cliConfig.ipv4EnforcedEnabledActive or
            cliConfig.ipv6EnforcedEnabledActive ) ):
      active = False
      inactiveReason = inactiveReasonRelayConfigured
   elif dhcpSnoopingStatus.enabled or dhcpSnooping6Status.enabled:
      active = False
      inactiveReason = inactiveReasonDhcpSnoopingConfigured
   elif dhcpServerActive( dhcpServerVrfStatus ):
      active = False
      inactiveReason = inactiveReasonDhcpServerConfigured
   elif not config.interface and not config.vlan:
      active = False
      if not config.vlan:
         inactiveReason = inactiveReasonNoInterfaceOrVlanConfigured
      else:
         inactiveReason = inactiveReasonNoInterfaceConfigured
   elif not active:
      if config.disabled != config.disabledDefault:
         inactiveReason = inactiveReasonDisabled
      elif config.localInterface == config.localInterfaceDefault:
         inactiveReason = inactiveReasonLocalIntfNotConfigured
      elif not config.dhcpV4Server:
         inactiveReason = inactiveReasonV4ServerNotConfigured
      elif not config.ipv6LockedAddressEnforcementDisabled and configuredIpv6Intfs:
         # We need this since we still do not fully support
         # IpLocking v6 in the enforcement enabled mode. We do not need the v4
         # counterpart for this inactiveReason since there is no case where we
         # want to display the same (every other inactiveReason will have higher
         # preference).
         inactiveReason = inactiveReasonV6EnforcementEnabled
   if ILTL.toggleIpLockingArIpEnabled():
      dhcpV4Servers = {}
      dhcpV6Servers = {}
      IpLockingProtocol = ipLockingModels.IpLockingProtocol
      for server, protocol in cliConfig.dhcpV4Server.items():
         dhcpV4Servers[ server ] = IpLockingProtocol( protocol=protocol )
      for server in cliConfig.dhcpV6Server:
         dhcpV6Servers[ server.stringValue ] = (
               IpLockingProtocol( protocol="arIpv6Query" ) )

   IpLocking = ipLockingModels.IpLocking
   if not active:
      if ILTL.toggleIpLockingArIpEnabled():
         return IpLocking(
            active=False, inactiveReason=inactiveReason,
            retryInterval=retryInterval, timeout=timeout,
            dhcpV4Servers=dhcpV4Servers, dhcpV6Servers=dhcpV6Servers,
            configuredIpv4Intfs=configuredIpv4Intfs,
            configuredIpv6Intfs=configuredIpv6Intfs,
            configuredIpv4Vlans=configuredIpv4Vlans,
            configuredIpv6Vlans=configuredIpv6Vlans,
            ipv4Enforced=cliConfig.ipv4LockedAddressEnforcementDisabled,
            ipv6Enforced=cliConfig.ipv6LockedAddressEnforcementDisabled )
      else:
         return IpLocking(
            active=False, inactiveReason=inactiveReason,
            retryInterval=retryInterval, timeout=timeout,
            configuredIpv4Intfs=configuredIpv4Intfs,
            configuredIpv6Intfs=configuredIpv6Intfs,
            configuredIpv4Vlans=configuredIpv4Vlans,
            configuredIpv6Vlans=configuredIpv6Vlans,
            ipv4Enforced=cliConfig.ipv4LockedAddressEnforcementDisabled,
            ipv6Enforced=cliConfig.ipv6LockedAddressEnforcementDisabled )
   ipLockingIntfEnabledDict = {}
   ipLockingVlanEnabledDict = {}
   for vlan in cliConfig.vlan:
      ipv4VlanMode = getVlanV4Mode( vlan )
      ipv4Enabled = ipv4VlanMode != 'invalid'
      ipv4InactiveReason = getVlanV4InactiveReason( vlan, ipv4Enabled )
      if ( not ILTL.toggleIpLockingArIpEnabled() and
           not cliConfig.ipv6LockedAddressEnforcementDisabled ):
         ipv6Enabled, ipv6VlanMode = False, "invalid"
         ipv6InactiveReason = notSupported
      else:
         ipv6VlanMode = getVlanV6Mode( vlan )
         ipv6Enabled = ipv6VlanMode != 'invalid'
         ipv6InactiveReason = getVlanV6InactiveReason( vlan, ipv6Enabled )
      ipLockingVlanEnabledDict[ vlan ] = ipLockingModels.IpLockingVlanEnabled(
            ipv4Enabled=ipv4Enabled,
            ipv4Mode=ipv4VlanMode,
            ipv4InactiveReason=ipv4InactiveReason,
            ipv6Enabled=ipv6Enabled,
            ipv6Mode=ipv6VlanMode,
            ipv6InactiveReason=ipv6InactiveReason )

   for intf in set().union( cliConfig.interface, intfVlanListMap.intfVlanList ):
      # If the interface is not configured for IP Locking/only has locked
      # address enforcement disabled on it, it needs the IP Locking configuration
      # from the VLANs it is assigned to so as to get enabled for IP Locking
      assignedVlans = {}
      intfAssignedToConfiguredVlan = False
      if intf in intfVlanListMap.intfVlanList:
         for vlan in intfVlanListMap.intfVlanList[ intf ].vlan:
            if vlan in cliConfig.vlan:
               intfAssignedToConfiguredVlan = True
               assignedVlans[ vlan ] = ipLockingVlanEnabledDict[ vlan ]
      # Do not consider interfaces that are not cliConfig.interface and assigned
      # to VLANs that are not configured by any VLAN
      if not intfAssignedToConfiguredVlan and intf not in cliConfig.interface:
         continue
      ipv4IntfMode = getIntfV4Mode( intf )
      ipv4Enabled = ipv4IntfMode in ( 'enforcementEnabled',
                                      'enforcementDisabled' )
      ipv4InactiveReason = getIntfV4InactiveReason( intf, ipv4Enabled )
      if ( not ILTL.toggleIpLockingArIpEnabled() and
           not cliConfig.ipv6LockedAddressEnforcementDisabled ):
         ipv6Enabled, ipv6IntfMode = False, "invalid"
         ipv6InactiveReason = notSupported
      else:
         ipv6IntfMode = getIntfV6Mode( intf )
         ipv6Enabled = ipv6IntfMode in ( 'enforcementEnabled',
                                         'enforcementDisabled' )
         ipv6InactiveReason = getIntfV6InactiveReason( intf, ipv6Enabled )
      IpLockingIntfEnabled = ipLockingModels.IpLockingIntfEnabled
      ipLockingIntfEnabledDict[ intf ] = IpLockingIntfEnabled(
            ipv4Enabled=ipv4Enabled,
            ipv4Mode=ipv4IntfMode,
            ipv4InactiveReason=ipv4InactiveReason,
            ipv6Enabled=ipv6Enabled,
            ipv6Mode=ipv6IntfMode,
            ipv6InactiveReason=ipv6InactiveReason,
            assignedVlans=assignedVlans )

   if ILTL.toggleIpLockingArIpEnabled():
      return IpLocking(
         active=True,
         retryInterval=retryInterval, timeout=timeout,
         dhcpV4Servers=dhcpV4Servers, dhcpV6Servers=dhcpV6Servers,
         configuredIpv4Intfs=configuredIpv4Intfs,
         configuredIpv6Intfs=configuredIpv6Intfs,
         configuredIpv4Vlans=configuredIpv4Vlans,
         configuredIpv6Vlans=configuredIpv6Vlans,
         enabledIntfs=ipLockingIntfEnabledDict,
         enabledVlans=ipLockingVlanEnabledDict,
         ipv4Enforced=cliConfig.ipv4LockedAddressEnforcementDisabled,
         ipv6Enforced=cliConfig.ipv6LockedAddressEnforcementDisabled )
   else:
      return IpLocking(
         active=True,
         retryInterval=retryInterval, timeout=timeout,
         configuredIpv4Intfs=configuredIpv4Intfs,
         configuredIpv6Intfs=configuredIpv6Intfs,
         configuredIpv4Vlans=configuredIpv4Vlans,
         configuredIpv6Vlans=configuredIpv6Vlans,
         enabledIntfs=ipLockingIntfEnabledDict,
         enabledVlans=ipLockingVlanEnabledDict,
         ipv4Enforced=cliConfig.ipv4LockedAddressEnforcementDisabled,
         ipv6Enforced=cliConfig.ipv6LockedAddressEnforcementDisabled )

def doShowIpLockingServers( mode, args ):
   dhcpV4Servers = {}
   dhcpV6Servers = {}
   ipLockingModels = getIpLockingModels()
   IpLockingProtocol = ipLockingModels.IpLockingProtocol
   for server, protocol in cliConfig.dhcpV4Server.items():
      dhcpV4Servers[ server ] = IpLockingProtocol( protocol=protocol )
   for server in cliConfig.dhcpV6Server:
      dhcpV6Servers[ server.stringValue ] = IpLockingProtocol(
            protocol="arIpv6Query" )
   return ipLockingModels.IpLockingServers( dhcpV4Servers=dhcpV4Servers,
                                            dhcpV6Servers=dhcpV6Servers )

def leaseInstalled( ip, intfId, action ):
   '''
   Returns whether or not a lease is installed in hardware. Currently this is always
   false for IPv6 leases.
   '''
   # look for installed status in installedConfig         :
   ipMacLeaseKey = Tac.newInstance(
      "IpLocking::IpIntfKey", ip, intfId )
   if ipMacLeaseKey in installedConfig.installedIpMacLease:
      installedIpMacLease = installedConfig.installedIpMacLease[ ipMacLeaseKey ]
      return ( installedIpMacLease.installedIp and
               installedIpMacLease.installedArp and
               installedIpMacLease.action == action )
   else:
      # Lease is not installed if it hasn't made it into smash yet as the agent
      # should be in the process of installing it. This includes the case when the
      # key exists but the action is different
      return False

def addStaticLeases( permittedIpsPerIntf, deniedIpsPerIntf, filterIntf=None,
                     onlyDisplayInstalled=False, v4=False, v6=False ):
   # Build the static lease -> intfId map from installedConfig.
   ipToIntf = {}
   for installedLease in installedConfig.installedIpMacLease:
      installedEntry = installedConfig.installedIpMacLease[ installedLease ]
      if ( installedLease.ip in cliConfig.staticLease and
           installedEntry.action == 'permit' ):
         ipToIntf[ installedLease.ip ] = installedLease.intfId

   for staticLease in cliConfig.staticLease.values():
      ip = staticLease.ip
      if ip.af == 'ipv4' and not v4:
         continue
      if ip.af == 'ipv6' and not v6:
         continue
      mac = staticLease.mac
      intfId = ipToIntf.get( ip, '' )

      if filterIntf and intfId != filterIntf.name:
         continue
      if intfId not in permittedIpsPerIntf:
         permittedIpsPerIntf[ intfId ] = {}
      leaseIsInstalled = leaseInstalled( ip, intfId, 'permit' )
      if onlyDisplayInstalled and not leaseIsInstalled:
         continue
      # Set static lease's expiration time to 0 since they won't expire.
      if ( intfId in deniedIpsPerIntf and ip in deniedIpsPerIntf[ intfId ] and
           leaseInstalled( ip, intfId, 'drop' ) ):
         leaseModel = getIpLockingModels().Lease(
                        clientMac=mac, source='config',
                        installed=leaseIsInstalled, expirationTime=0,
                        overwritten=True )
      else:
         leaseModel = getIpLockingModels().Lease(
                        clientMac=mac, source='config',
                        installed=leaseIsInstalled, expirationTime=0 )
      permittedIpsPerIntf[ intfId ][ ip ] = leaseModel

def addDeniedEntry( deniedIpsPerIntf, filterIntf=None, onlyDisplayInstalled=False ):
   for intfId, intfConfig in cliConfig.interface.items():
      for ip in intfConfig.ipv4DeniedAddr.keys():
         if filterIntf and intfId != filterIntf.name:
            continue
         if intfId not in deniedIpsPerIntf:
            deniedIpsPerIntf[ intfId ] = {}
         leaseIsInstalled = leaseInstalled( ip, intfId, 'drop' )
         if onlyDisplayInstalled and not leaseIsInstalled:
            continue
         mac = Tac.Value( 'Arnet::EthAddr' ).ethAddrZero
         # Set static deny's expiration time to 0 since they won't expire.
         leaseModel = getIpLockingModels().Lease(
                        clientMac=mac, source='config',
                        installed=leaseIsInstalled, expirationTime=0 )
         deniedIpsPerIntf[ intfId ][ ip ] = leaseModel

def addDynamicLeases( permittedIpsPerIntf, deniedIpsPerIntf,
                      v4=True, filterIntf=None, onlyDisplayInstalled=False ):
   for interfaceLeaseSet in leaseStatus.interfaceLeaseSet.values():
      intfId = interfaceLeaseSet.intfId
      if filterIntf and intfId != filterIntf.name:
         continue
      if intfId not in permittedIpsPerIntf:
         permittedIpsPerIntf[ intfId ] = {}
      for ipMacLease in interfaceLeaseSet.ipMacLease.values():
         ip = ipMacLease.ip
         mac = ipMacLease.mac
         if ( v4 and ip.af == 'ipv6' ) or ( not v4 and ip.af != 'ipv6' ):
            continue
         expirationTime = int( ipMacLease.expirationTime )
         leaseIsInstalled = leaseInstalled( ip, intfId, 'permit' )
         if onlyDisplayInstalled and not leaseIsInstalled:
            continue
         if ( intfId in deniedIpsPerIntf and ip in deniedIpsPerIntf[ intfId ] and
              leaseInstalled( ip, intfId, 'drop' ) ):
            leaseModel = getIpLockingModels().Lease(
                           clientMac=mac, source='server',
                           installed=leaseIsInstalled,
                           expirationTime=expirationTime, overwritten=True )
         else:
            leaseModel = getIpLockingModels().Lease(
                           clientMac=mac, source='server',
                           installed=leaseIsInstalled,
                           expirationTime=expirationTime )
         permittedIpsPerIntf[ intfId ][ ip ] = leaseModel

def doShowAddressLockingTable( mode, args ):
   filterIntf = args.get( 'INTF' )
   onlyDisplayInstalled = 'installed' in args
   displayStaticEntry = 'dynamic' not in args
   displayDynamicLease = 'static' not in args
   displayPermitEntry = 'deny' not in args
   displayDenyEntry = 'permit' not in args
   v4 = 'ipv6' not in args
   v6 = 'ipv4' not in args
   deniedIpsPerIntf = {}
   permittedIpsPerIntf = {}

   # We only support denying address in IPv4 currently
   if v4:
      addDeniedEntry( deniedIpsPerIntf, filterIntf=filterIntf,
                      onlyDisplayInstalled=onlyDisplayInstalled )

   if displayStaticEntry and displayPermitEntry:
      addStaticLeases( permittedIpsPerIntf, deniedIpsPerIntf, filterIntf=filterIntf,
                       onlyDisplayInstalled=onlyDisplayInstalled, v4=v4, v6=v6 )

   if displayDynamicLease and displayPermitEntry:
      addDynamicLeases( permittedIpsPerIntf, deniedIpsPerIntf,
                        filterIntf=filterIntf, v4=v4,
                        onlyDisplayInstalled=onlyDisplayInstalled )

   ipLockingModels = getIpLockingModels()
   IntfLeases = ipLockingModels.IntfLeases
   intfLeasesDict = {}
   for ( intfId, permitLeases ) in permittedIpsPerIntf.items():
      intfLeasesDict[ intfId ] = IntfLeases( leases=permitLeases )
   if ( displayStaticEntry and displayDenyEntry ):
      denyLeasesDict = {}
      for ( intfId, denyLeases ) in deniedIpsPerIntf.items():
         denyLeasesDict[ intfId ] = IntfLeases( leases=denyLeases )
      return ipLockingModels.LeasesTable( intfLeases=intfLeasesDict,
                                          denyLeases=denyLeasesDict )
   return ipLockingModels.LeasesTable( intfLeases=intfLeasesDict )

def getServerCountersModel( serverCounters ):
   return getIpLockingCounterModels().ServerCounters(
      leaseQuery=serverCounters.leaseQuery,
      leaseActiveReceived=serverCounters.leaseActiveReceived,
      leaseActiveDropped=serverCounters.leaseActiveDropped,
      leaseUnknownReceived=serverCounters.leaseUnknownReceived,
      leaseUnknownDropped=serverCounters.leaseUnknownDropped,
      leaseUnassignedReceived=serverCounters.leaseUnassignedReceived,
      leaseUnassignedDropped=serverCounters.leaseUnassignedDropped,
      unknown=serverCounters.unknown,
   )

def getIntfCountersModel( intfCounters ):
   return getIpLockingCounterModels().IntfCounters(
      leaseQuery=intfCounters.leaseQuery,
      leaseActive=intfCounters.leaseActive,
      leaseUnknown=intfCounters.leaseUnknown,
      leaseUnassigned=intfCounters.leaseUnassigned,
   )

def doShowIpLockingCounters( mode, args ):
   intfCounters = {}
   sysdbIntfCounters = cliStatus.intfCounters
   for intf in sysdbIntfCounters:
      intfCounters[ intf ] = getIntfCountersModel( sysdbIntfCounters[ intf ] )
   serverCounters = {}
   sysdbServerCounters = cliStatus.serverCounters
   for server in sysdbServerCounters:
      serverCounters[ server ] = getServerCountersModel(
            sysdbServerCounters[ server ] )
   return getIpLockingCounterModels().IpLockingCountersModel(
      intfCounters=intfCounters, serverCounters=serverCounters )

# -------------------------------------------------------------------------------
# Have the Cli agent mount all needed state from sysdb
# -------------------------------------------------------------------------------
def Plugin( entityManager ):
   global cliConfig
   global cliStatus
   global hwConfig
   global enabledInterfaceStatus
   global enabledVlanStatus
   global leaseStatus
   global installedConfig
   global arpInspectionStatus
   global ipsgConfig
   global dhcpRelayStatus
   global dhcpSnoopingStatus
   global dhcpSnooping6Status
   global dhcpServerVrfStatus
   global intfVlanListMap

   cliConfig = LazyMount.mount( entityManager, "iplocking/cliConfig",
                                  "IpLocking::CliConfig", "r" )
   cliStatus = LazyMount.mount( entityManager, "iplocking/cliStatus",
                                "IpLocking::CliStatus", "r" )
   hwConfig = LazyMount.mount( entityManager, "iplocking/hardware/config",
                               "IpLocking::Hardware::Config", "r" )
   enabledInterfaceStatus = LazyMount.mount( entityManager,
         "iplocking/interfaceStatus", "IpLocking::EnabledInterfaceStatus", "r" )
   enabledVlanStatus = LazyMount.mount( entityManager,
         "iplocking/vlanStatus", "IpLocking::EnabledVlanStatus", "r" )
   leaseStatus = LazyMount.mount( entityManager, "iplocking/leaseStatus",
                                  "IpLocking::LeaseStatus", "r" )
   arpInspectionStatus = LazyMount.mount( entityManager,
                                          "security/arpInspection/status",
                                          "ArpInsp::Status", "r" )
   ipsgConfig = LazyMount.mount( entityManager, "security/ipsg/config",
                                 "Ipsg::Config", "r" )
   dhcpRelayStatus = LazyMount.mount( entityManager, "ip/helper/dhcprelay/status",
                                      "Ip::Helper::DhcpRelay::Status", "r" )
   dhcpSnoopingStatus = LazyMount.mount( entityManager,
                                         "bridging/dhcpsnooping/status",
                                         "Bridging::DhcpSnooping::Status", "r" )
   dhcpSnooping6Status = LazyMount.mount( entityManager,
                                          "bridging/dhcpsnooping/dhcp6Status",
                                          "Bridging::DhcpSnooping::Status", "r" )
   dhcpServerVrfStatus = LazyMount.mount( entityManager, "dhcpServer/vrf/status",
                                       "DhcpServer::VrfStatusDir", "r" )
   intfVlanListMap = LazyMount.mount( entityManager,
                                      "iplocking/intfVlanListMap",
                                      "IpLocking::IntfVlanListMap", "r" )

   shmemEm = SharedMem.entityManager( sysdbEm=entityManager )
   installedConfig = shmemEm.doMount( "iplocking/hardware/installedConfig",
                                      "IpLocking::Hardware::InstalledConfig",
                                      Smash.mountInfo( 'keyshadow' ) )
