#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from CliModel import List
from CliModel import Bool
from CliModel import Int
from CliModel import Model
from CliModel import Str
import TableOutput

#-------------------------------------------------------------------
# model for "show ip hardware fib load-balance distribution"
#-------------------------------------------------------------------

class GlobalDlb( Model ):
   __revision__ = 2
   globalDlbEcmpEnabled = \
           Bool( help="Whether or not dynamic load balancing is enabled globally" )
   dlbPortAssignmentMode = \
           Int( help="DLB operational mode" )
   dlbInactivityDuration = \
           Int( help="DLB inactivity duration in microseconds" )
   dlbSamplingPeriod = \
           Int( help="DLB sampling period in microseconds" )
   dlbPortLoadingWeight = \
           Int( help="DLB average traffic weight" )
   dlbFlowSetSize = Int( help="DLB flow set size" )
   dlbEcmpNhg = List( valueType=str,
                      help="Dynamic load balancing enabled static nexthop groups" )
   randomSelectionControlSeed = Int( help="DLB Hash Seed" )
   lbnBasedDlbHashing = Bool(
         help="Obsolete since release 4.33.1F" )
   lbnBasedStaticEcmpHashing = Bool(
         help="Obsolete since release 4.33.1F" )
   dlbAccessGroupIpv4 = \
           Str( help="Access group configured for IPv4 selective DLB" )
   dlbAccessGroupIpv6 = \
           Str( help="Access group configured for IPv6 selective DLB" )

   def render ( self ):
      print( "" )
      if self.globalDlbEcmpEnabled:
         print( "Global load balance distribution: Dynamic" )
         if self.dlbPortAssignmentMode == 0:
            portAssignmentMode = "Optimal Timer"
         elif self.dlbPortAssignmentMode == 2:
            portAssignmentMode = "Optimal Always"
         print( "Dynamic load balancing member selection: %s" %
                portAssignmentMode )
         print( "Dynamic load balancing inactivity threshold: %d microseconds" %
                self.dlbInactivityDuration )
         print( "Dynamic load balancing sampling period: %d microseconds" %
                self.dlbSamplingPeriod )
         # average-traffic-weight is port-loading-weight in broadcom term
         print( "Dynamic load balancing average traffic weight: %d" %
               self.dlbPortLoadingWeight )
         print( "Dynamic load balancing hash seed: %d" %
                self.randomSelectionControlSeed )
         print( "Dynamic load balancing flow set size: %d" % self.dlbFlowSetSize )
         if self.dlbAccessGroupIpv4:
            print( "Dynamic load balancing IPv4 access group: %s" %
                   self.dlbAccessGroupIpv4 )
         if self.dlbAccessGroupIpv6:
            print( "Dynamic load balancing IPv6 access group: %s" %
                   self.dlbAccessGroupIpv6 )
      else:
         print( "Global load balance distribution: Hash" )
      if len( self.dlbEcmpNhg ):
         print( "Nexthop groups with dynamic load balancing enabled: " )
         for nhgName in self.dlbEcmpNhg:
            print( nhgName )

class FlowsetTableEntry( Model ):
   dlbId = Int( help="DLB id associated with a DLB Flow" )
   portMemberAssignment = Str( help="Port member through which the DLB flow \
is egressing" )

class MacroFlows( Model ):
   flowList = List( valueType = FlowsetTableEntry,
         help="List containing active Flows"  )
   exception = Bool( help="Flag that is set to true when an exception is caused",
         default=False )
   exceptionMsg = Str( help="Display exception message",
         default="No exception raised" )
   enabled = Bool( help="DLB enabled", default=True )
   dlbId = Int( help="Value of dlbId from query, -1 if no dlbId provided")

   def render( self ):
      if not self.enabled:
         print( "DLB is not enabled" )
         return
      if self.exception:
         print( self.exceptionMsg )
         return

      numFlows = len ( self.flowList )
      if self.dlbId == -1:
         if self.flowList:
            print( f"Total number of active flows: { numFlows } " )
         else:
            print( "No active flows present" )
      else:
         if self.flowList:
            print( f"Total number of active flows for group { self.dlbId }: \
{ numFlows }" )
         else:
            print( f"No active flows present for group { self.dlbId }" )

      portMinLen = 23
      portMaxLen = 23
      dlbIdMaxLen = 5
      dlbIdMinLen = 5
      numOfColumns = 2

      tableMaxWidth = ( portMaxLen + dlbIdMaxLen + numOfColumns -1 )
      table = TableOutput.createTable( ( "Port", "GroupId" ), tableWidth =
            tableMaxWidth )
      #Egress Port Field
      f1 = TableOutput.Format( justify = "left", minWidth = portMinLen,
            maxWidth = portMaxLen, isHeading = True )
      f1.noPadLeftIs( True )
      f1.padLimitIs( True )
      #DLB Id Field
      f2 = TableOutput.Format( justify = "left", minWidth = dlbIdMinLen,
            maxWidth = dlbIdMaxLen )
      f2.noPadLeftIs( True )
      f2.padLimitIs( True )

      table.formatColumns( f1, f2 )
      table.formatRows( TableOutput.Format( isHeading=True, border=False ) )
      for entry in self.flowList:
         table.newRow( entry.portMemberAssignment, entry.dlbId )
      if len( self.flowList ) != 0:
         print( table.output() )
