#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
import Arnet.MplsLib
from CliPlugin import VrfCli
from CliPlugin import IraCommonCli
from CliPlugin import IraRouteCommon
from CliPlugin.IraIpCliCommon import (
      setRoutesInExactMatch,
      unsetRoutesInExactMatch,
      IPV6,
      enableOptimizePrefixes,
      vrfFilterWarningCheck,
)
from CliPlugin.IraIpModel import VrfIpRibRoutes
from CliPlugin.IraIp6RouteCliLib import (
      ip6,
      routing6,
      noIpv6RouteTableForVrfMsg,
      noIpv6Route,
      staticIpv6RoutingTable,
      manageIpv6Route
)
from CliPlugin.IraIp6Model import (
      Ip6RouteSummaryForVrf,
      Ip6UrpfInterface,
      Ip6UrpfInterfaces,
)
from CliPlugin.VrfCli import (
      getAllVrfNames,
      vrfExists,
)
from CliPlugin.IraIp6Cli import (
   allVrfStatusLocal,
   printer,
   routedIntfsWithIp6Configured,
   routingHardwareStatus,
   routing6HardwareStatus,
   showRoute,
   vrfSummaryCommon
)
import ConfigMount
import Tac
from TypeFuture import TacLazyType
from IpLibConsts import DEFAULT_VRF, ALL_VRF_NAME
from Toggles.IraToggleLib import (
   toggleUrpfIgnoreDefaultRouteEnabled
)

ip4Config = None
ip6Config = None
runnabilityConfig = None
routing6HardwareConfig = None

#printer = CliPrint().lib
UrpfMode = TacLazyType( "Urpf::UrpfMode" )

def noIpv6Routing( mode, vrfName=DEFAULT_VRF, delStaticRoutes=None ):
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   if not vrfExists( vrfName ):
      # vrf is deleted/not configured, nothing more to do.
      mode.addWarning( f"No such VRF {vrfName}." )
      return
   routing6.config( vrfName ).routing = False
   # only warn if not startup and there are static routes
   if ( not delStaticRoutes and
        not mode.session_.startupConfig() and
        staticIpv6RoutingTable( vrfName ).route ):
      mode.addWarning( "Preserving static routes.  Use "
                       "'no ipv6 unicast-routing delete-static-routes' "
                       "to clear them." )
   if delStaticRoutes == 'delete-static-routes':
      # Delete all of the static routes if requested.
      rt = staticIpv6RoutingTable( vrfName )
      rt.route.clear()
   if vrfName in runnabilityConfig.vrf:
      del runnabilityConfig.vrf[ vrfName ]

def ipv6Routing( mode, vrfName=DEFAULT_VRF ):
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   if not vrfExists( vrfName ):
      # pylint: disable-next=consider-using-f-string
      mode.addError( "No such VRF %s." % vrfName )
      return
   if not routing6HardwareStatus.routingSupported and \
         not mode.session_.startupConfig():
      mode.addWarning( "IPv6 Hardware forwarding is not supported on this "
                       "platform. This means that all IPv6 traffic will be routed "
                       "in software." )
   routing6.config( vrfName ).routing = True
   runnabilityConfig.vrf[ vrfName ] = True

def handlerIpv6UnicastRoutingCmd( mode, args ):
   ipv6Routing( mode, vrfName=args.get( 'VRF', DEFAULT_VRF ) )

def noOrDefaultHandlerIpv6UnicastRoutingCmd( mode, args ):
   delStaticRoutes = None
   if 'keep-static-routes' in args:
      delStaticRoutes = 'keep-static-routes'
   elif 'delete-static-routes' in args:
      delStaticRoutes = 'delete-static-routes'
   noIpv6Routing( mode, vrfName=args.get( 'VRF', DEFAULT_VRF ),
                  delStaticRoutes=delStaticRoutes )

def handlerIpv6IcmpUnreachableRateCmd( mode, args ):
   # Setting rate to 0 makes ICMP _not_ unreachable.
   routing6HardwareConfig.icmpUnreachable = 'RATE' not in args

def setIpv6Redirect( mode, no ):
   vrfNames = getAllVrfNames( mode )
   for vrf in vrfNames:
      r = routing6.config( vrf )
      r.sendRedirects = not no

def noIpv6Redirect( mode ):
   setIpv6Redirect( mode, no=True )

def ipv6Redirect( mode ):
   setIpv6Redirect( mode, no=False )

def handlerIpv6IcmpRedirectCmd( mode, args ):
   ipv6Redirect( mode )

def noHandlerIpv6IcmpRedirectCmd( mode, args ):
   noIpv6Redirect( mode )

def defaultHandlerIpv6IcmpRedirectCmd( mode, args ):
   ipv6Redirect( mode )

def handlerEnhancedDadCmd( mode, args ):
   ip6Config.enhancedDad = True

def noHandlerEnhancedDadCmd( mode, args ):
   ip6Config.enhancedDad = False

def manageIp6Route( mode, prefix, nexthop, routeOptions=None, vrfName=DEFAULT_VRF,
                    leakToVrf=None, egressVrf=None, viaResRibProfile=None ):
   assert vrfName != ''
   if not vrfName:
      vrfName = DEFAULT_VRF

   if not vrfExists( vrfName ):
      # pylint: disable-next=consider-using-f-string
      mode.addError( "%s Create first." % noIpv6RouteTableForVrfMsg % vrfName )
      return

   manageIpv6Route( mode, prefix, nexthop, routeOptions, vrfName, leakToVrf,
                    egressVrf, viaResRibProfile=viaResRibProfile )

def noIp6Route( mode, prefix, nexthop, preference, vrfName, egressVrf,
      configTag=False ):
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )

   if not vrfExists( vrfName ):
      # VRF is deleted, nothing more to do.
      mode.addWarning( f"No such VRF {vrfName}." )
      return

   noIpv6Route( mode, prefix, nexthop, preference, vrfName, egressVrf, configTag )

def handlerIpv6RouteMplsLabelCmd( mode, args ):
   manageIp6Route( mode, args[ 'PREFIX' ], args[ 'NEXTHOP_INTF' ],
                     routeOptions=args.get( 'OPTION' ),
                     vrfName=args.get( 'VRF', DEFAULT_VRF ),
                     leakToVrf=args.get( 'LEAK_TO_VRF' ) )

def handlerIpv6RouteNexthopGroupCmd( mode, args ):
   manageIp6Route( mode, args[ 'PREFIX' ], args[ 'NEXTHOP_INTF' ],
                     routeOptions=args.get( 'OPTION' ),
                     vrfName=args.get( 'VRF', DEFAULT_VRF ),
                     leakToVrf=args.get( 'LEAK_TO_VRF' ) )

def handlerIpv6RouteNexthopOrComboCmd( mode, args ):
   manageIp6Route( mode, args[ 'PREFIX' ], args[ 'NEXTHOP_INTF' ],
                     routeOptions=args.get( 'OPTION' ),
                     vrfName=args.get( 'VRF', DEFAULT_VRF ),
                     leakToVrf=args.get( 'LEAK_TO_VRF' ) )

def handlerIpv6RouteNexthopResRibProfileCmd( mode, args ):
   # nexthop in manageIpv6Route() is expected to be a dictionary
   # with the address set for 'nexthop' key
   nexthop = {}
   nexthop[ 'nexthop' ] = args.pop( 'NEXTHOP_ADDR' )
   manageIp6Route( mode, args[ 'PREFIX' ], nexthop,
                   routeOptions=args.get( 'OPTION' ),
                   vrfName=args.get( 'VRF', DEFAULT_VRF ),
                   viaResRibProfile=args.get( 'SYS_CON_ONLY_RIBS' ) )

def handlerIpv6RouteInterfaceCmd( mode, args ):
   if 'NEXTHOP_INTF' not in args:
      args[ 'NEXTHOP_INTF' ] = {}
   args[ 'NEXTHOP_INTF' ][ 'intf' ] = {}
   args[ 'NEXTHOP_INTF' ][ 'intf' ][ 'intf' ] = args.pop( 'INTF' )
   args[ 'NEXTHOP_INTF' ][ 'intf' ][ 'intfnexthop' ] = None

   manageIp6Route( mode, args[ 'PREFIX' ], args[ 'NEXTHOP_INTF' ],
                     routeOptions=args.get( 'OPTION' ),
                     vrfName=args.get( 'VRF', DEFAULT_VRF ),
                     leakToVrf=args.get( 'LEAK_TO_VRF' ) )

def handlerIpv6RouteEgressVrfCmd( mode, args ):
   manageIp6Route( mode, args[ 'PREFIX' ], args[ 'NEXTHOP_INTF' ],
                     routeOptions=args.get( 'OPTION' ),
                     vrfName=args.get( 'VRF', DEFAULT_VRF ),
                     leakToVrf=args.get( 'LEAK_TO_VRF' ),
                     egressVrf=args.get( 'EGRESS_VRF' ) )

def handlerIpv6RouteEgressVrfNexthopOrComboCmd( mode, args ):
   manageIp6Route( mode, args[ 'PREFIX' ], args[ 'NEXTHOP_INTF' ],
                     routeOptions=args.get( 'OPTION' ),
                     vrfName=args.get( 'VRF', DEFAULT_VRF ),
                     leakToVrf=args.get( 'LEAK_TO_VRF' ),
                     egressVrf=args.get( 'EGRESS_VRF' ) )

def handlerIpv6RouteIntfEgressVrfCmd( mode, args ):
   manageIp6Route( mode, args[ 'PREFIX' ], args[ 'NEXTHOP_INTF' ],
                     routeOptions=args.get( 'OPTION' ),
                     vrfName=args.get( 'VRF', DEFAULT_VRF ),
                     leakToVrf=args.get( 'LEAK_TO_VRF' ),
                     egressVrf=args.get( 'EGRESS_VRF' ) )

def noOrDefaultHandlerIpv6RouteIntfEgressVrfCmd( mode, args ):
   noIp6Route( mode, args[ 'PREFIX' ], args[ 'NEXTHOP_INTF' ],
               args.get( 'PREFERENCE' ),
               args.get( 'VRF', DEFAULT_VRF ),
               args.get( 'EGRESS_VRF' ) )

def showRouteFec( mode, prefix=None, vrfName=None ):
   vrf = vrfName
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   vrfFecRoutesModel = IraRouteCommon.VrfFecRoutesModel()
   if vrfName != ALL_VRF_NAME and not vrfExists( vrfName ):
      mode.addError( noIpv6RouteTableForVrfMsg % vrfName )
      return vrfFecRoutesModel

   vrfFilterWarningCheck( mode, vrf, prefix )

   vrfs = []

   # If VRF name is given, we will validate it; 'all' is a special case
   # we will print info for the default VRF or all VRFs
   if vrfName == ALL_VRF_NAME:
      vrfs.append( DEFAULT_VRF )
      for vrf in sorted( allVrfStatusLocal.vrf ):
         vrfs.append( vrf )
   else:
      vrfs.append( vrfName )

   if prefix != None: # pylint: disable=singleton-comparison
      # Ensure that I have an Ip6Prefix
      tn = prefix.tacType.fullTypeName
      if tn in ( 'Arnet::Ip6Addr', 'Arnet::Ip6AddrWithMask' ):
         v = ( prefix.stringValue
               if tn == "Arnet::Ip6Addr"
               else prefix.subnet.stringValue )
         prefix = Arnet.Ip6Prefix( v )

   for vrf in vrfs:
      if vrfName == ALL_VRF_NAME:
         print( "VRF: " + vrf )
      vrfFecRoutesModel.vrfs[ vrf ] = routing6.showFecs( mode,
                                                         prefix=prefix,
                                                         vrfName=vrf )
      if not routing6.config( vrf ).routing:
         mode.addWarning( "IPv6 routing not enabled" )
   return vrfFecRoutesModel

def handlerShowIpv6RouteFecCmd( mode, args ):
   return showRouteFec( mode, prefix=args.get( 'PREFIX' ),
                        vrfName=args.get( 'VRF' ) )

def showIpFecSummary( mode, vrfName=None ):
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   return routing6.showFecs( mode, vrfName=vrfName, summary=True )

def handlerShowIpv6FecSummaryCmd( mode, args ):
   return showIpFecSummary( mode, vrfName=args.get( 'VRF' ) )

def handlerShowIpv6RouteNextHopGroupCmd( mode, args ):
   NexthopGroupId = Tac.Type( "Routing::NexthopGroup::NexthopGroupId" )
   routeFilterList = Tac.newInstance( "Ira::RouteFilterColl", True )
   nhgName = args.get( 'NHG_NAME' )
   if nhgName:
      wildcard = False
      nhgId = routing6.getNexthopGroupId( nhgName )
      if nhgId == NexthopGroupId.null:
         routeFilterList.filterAll = True
   else:
      nhgId = NexthopGroupId.null
      wildcard = True
   routeFilterNhg = Tac.newInstance( "Ira::RouteFilterNextHopGroup", nhgId,
                                       wildcard )
   routeFilterList.filterColl.add( routeFilterNhg )
   return showRoute( mode, vrfName=args.get( 'VRF' ),
                     routeFilter=routeFilterList )

def showV6NotInstalledBfd( mode, vrfName=None ):
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )

   if not ( vrfName is None or vrfName == ALL_VRF_NAME or vrfExists( vrfName ) ):
      # pylint: disable-next=consider-using-f-string
      mode.addError( "IPv6 routing table %s does not exist." % vrfName )
      return None

   vrfs = []
   if vrfName == ALL_VRF_NAME:
      vrfs = [ DEFAULT_VRF ] + sorted( allVrfStatusLocal.vrf )
   elif vrfName is not None:
      vrfs.append( vrfName )
   else:
      vrfs.append( DEFAULT_VRF )

   return IraCommonCli.showCommonNotInstalledBfd( vrfs=vrfs, v4=False )

def handlerShowIpv6NotInstalledBfdCmd( mode, args ):
   return showV6NotInstalledBfd( mode, vrfName=args.get( 'VRF' ) )

def handlerShowIpv6RouteLongerPrefixesCmd( mode, args ):
   # pylint check fails if appended directly to set
   prefix = args[ 'PREFIX' ]
   optionsSet = args[ 'OPTIONS' ]
   vrfName = args.get( 'VRF' )
   optionsDict = dict( optionsSet or [] )
   optionsDict[ 'longerPrefixes' ] = 'longer-prefixes'
   optionsSet = list( optionsDict.items() )
   return showRoute( mode, prefix, optionsSet, vrfName )

def showRouteVrfAllSummaryBrief( mode, quantify=False ):
   beforeSum = Tac.now()
   vrfs = []

   for vrf in sorted( allVrfStatusLocal.vrf ):
      vrfs.append( vrf )
   vrfs.insert( 0, DEFAULT_VRF )

   ip6RouteSummaryBriefModel = Ip6RouteSummaryForVrf()
   ip6RouteSummaryBriefModel.maskLen = {}
   ip6RouteSummaryBriefModel.totalRoutes = 0
   ip6RouteSummaryBriefModel.vrfCount = len( vrfs )
   ip6RouteSummaryBriefModel = vrfSummaryCommon( mode, ip6RouteSummaryBriefModel,
                                                 vrfs )

   # pylint: disable=protected-access
   if quantify:
      afterSum = Tac.now()
      ip6RouteSummaryBriefModel._quantify = afterSum - beforeSum
   return ip6RouteSummaryBriefModel

def handlerShowIpv6RouteSummaryBriefCmd( mode, args ):
   quantify = 'quantify' in args
   return showRouteVrfAllSummaryBrief( mode, quantify=quantify )

def noipv6AggPrefix( mode, prefix ):
   if Arnet.Ip6Prefix( prefix.stringValue ) in routing6.hwConfig( vrfName=DEFAULT_VRF
         ).aggregateRoute:
      del routing6.hwConfig( vrfName=DEFAULT_VRF ).\
          aggregateRoute[ Arnet.Ip6Prefix( prefix.stringValue ) ]

def doipv6AggPrefix( mode, prefix ):
   routing6.hwConfig( vrfName=DEFAULT_VRF ).aggregateRoute[
         Arnet.Ip6Prefix( prefix.stringValue ) ] = True

def handlerIpv6FibAggregateAddrCmd( mode, args ):
   doipv6AggPrefix( mode, args[ 'PREFIX' ] )

def noOrDefaultHandlerIpv6FibAggregateAddrCmd( mode, args ):
   noipv6AggPrefix( mode, args[ 'PREFIX' ] )

def showHardwareAggregateRoute( mode, prefix=None ):
   if prefix:
      # Ensure I have an Ip6AddrWithMask and not an Ip6Addr
      if prefix.tacType.fullTypeName == 'Arnet::Ip6Addr':
         prefix = Tac.Value( 'Arnet::Ip6AddrWithMask', address=prefix, len=128 )
   routing6.showHwAggregateRoute( mode, prefix )

def handlerShowIpv6FibAggregateAddrCmd( mode, args ):
   showHardwareAggregateRoute( mode, prefix=args.get( 'PREFIX' ) )

def handlerOptimizeV6RoutesCmd( mode, args ):
   setRoutesInExactMatch( mode, IPV6, routing6HardwareConfig,
                           routing6HardwareStatus, args[ "LENGTH" ] )

def handlerVrfOptimizeV6RoutesCmd( mode, args ):
   setRoutesInExactMatch( mode, IPV6, routing6HardwareConfig,
                           routing6HardwareStatus, args[ "LENGTH" ],
                           vrfs=args[ "VRFS" ] )

def noOrDefaultHandlerVrfOptimizeV6RoutesCmd( mode, args ):
   unsetRoutesInExactMatch( mode, IPV6, routing6HardwareConfig,
                              routing6HardwareStatus, vrfs=args[ "VRFS" ] )

def handlerIpv6OptimizePrefixesCmd( mode, args ):
   enableOptimizePrefixes( mode, IPV6, routing6HardwareConfig,
                           profileType=args[ 'PROFILE' ] )

def noOrDefaultHandlerIpv6OptimizePrefixesCmd( mode, args ):
   unsetRoutesInExactMatch( mode, IPV6, routing6HardwareConfig,
                              routing6HardwareStatus )

def handlerVrfIpv6OptimizePrefixesCmd( mode, args ):
   enableOptimizePrefixes( mode, IPV6, routing6HardwareConfig,
                           profileType=args[ 'PROFILE' ],
                           vrfs=args[ 'VRFS' ] )

def resilientEcmpIp6PrefixIs( mode, prefix, capacity, redundancy, ordered, marked ):

   # if C * R > maxResilientEcmp value, throw max value error
   if ( routing6HardwareStatus.maxResilientEcmp and
        ( capacity * redundancy ) > routing6HardwareStatus.maxResilientEcmp ):
      # pylint: disable-next=consider-using-f-string
      mode.addError( "Maximum value supported for capacity * redundancy is %d"
          % ( routing6HardwareStatus.maxResilientEcmp ) )
      return

   ecmpConfig = Tac.Value(
         "Routing::Hardware::ResilientEcmpConfig", capacity, redundancy )
   ecmpConfig.orderedVias = ( ordered is not None )
   if marked:
      routing6HardwareConfig.policyBasedResilientEcmp = ecmpConfig
   else:
      genPrefix = Tac.Value( "Arnet::IpGenPrefix" )
      genPrefix.handleV6Prefix( prefix.subnet )
      routing6HardwareConfig.resilientEcmpPrefix[ genPrefix ] = ecmpConfig

def handlerIpv6ResilientEcmpCmd( mode, args ):
   resilientEcmpIp6PrefixIs( mode, args.get( "PREFIX" ), args[ 'CAPACITY' ],
                              args[ 'REDUNDANCY' ], args.get( 'ordered' ),
                              args.get( 'marked' ) )

def noOrDefaultHandlerIpv6ResilientEcmpCmd( mode, args ):
   prefix = args.get( "PREFIX" )
   # You must check that prefix is not None because the IPv6 default prefix (::/0)
   # will return a falsy object. Otherwise you may end up deleting all configured
   # V6 RECMP prefixes.
   if prefix is not None:
      genPrefix = Tac.Value( "Arnet::IpGenPrefix" )
      genPrefix.handleV6Prefix( prefix.subnet )
      del routing6HardwareConfig.resilientEcmpPrefix[ genPrefix ]
   else:
      # config for policy marked needs to be cleared whether 'marked' keyword
      # is specified or not
      ecmpConfig = Tac.Value( "Routing::Hardware::ResilientEcmpConfig", 0, 0 )
      routing6HardwareConfig.policyBasedResilientEcmp = ecmpConfig
      if 'marked' not in args:
         # user asked to clear config for all the prefix as well
         routing6HardwareConfig.resilientEcmpPrefix.clear()

def v6ResolvedUrpfStatus( intfId, v6Mode, v6AllowDefault, v6LookupVrf ):
   if not routingHardwareStatus.urpfSupportedOnV4AndV6:
      if intfId in ip4Config.ipIntfConfig:
         v4Mode = ip4Config.ipIntfConfig[ intfId ].urpf.mode
         v4AllowDefault = ip4Config.ipIntfConfig[ intfId ].urpf.allowDefaultRoute
         v4LookupVrf = ip4Config.ipIntfConfig[ intfId ].urpf.lookupVrf
         if v4Mode != 'disable' and \
            ( v6Mode != v4Mode or v6AllowDefault != v4AllowDefault or
              v6LookupVrf != v4LookupVrf ):
            # v4 and v6 mode in conflict v6 to be disabled by platform
            return 'disable', False, ''
   # TODO BUG642564: When strictDefault is removed from UrpfMode enum,
   # update the check here.
   allowDefault = ( v6Mode == UrpfMode.strictDefault or
                    ( toggleUrpfIgnoreDefaultRouteEnabled() and
                      routing6HardwareStatus.urpfLooseNonDefaultSupported and
                      v6Mode == UrpfMode.loose and v6AllowDefault ) )
   return v6Mode, allowDefault, v6LookupVrf

def populateIpv6UrpfModel( mode, intfs=None ):
   uRpfModel = Ip6UrpfInterfaces()
   uRpfModel.viaDefaultRoute = False
   for intf in intfs:
      if not intf.name in ip6.config.intf:
         # The command has no interface filter so skip interfaces that have
         # been deleted before this loop could get to them.
         continue
      v6Mode = ip6.config.intf[ intf.name ].urpf.mode
      v6AllowDefault = ip6.config.intf[ intf.name ].urpf.allowDefaultRoute
      v6LookupVrf = ip6.config.intf[ intf.name ].urpf.lookupVrf
      resolvedMode, resolvedAllowDefault, resolvedLookupVrf = \
         v6ResolvedUrpfStatus( intf.name, v6Mode, v6AllowDefault, v6LookupVrf )

      if resolvedMode != 'disable':
         uRpfIntf = Ip6UrpfInterface()
         uRpfIntf.uRpfMode = resolvedMode
         uRpfIntf.allowDefault = resolvedAllowDefault
         uRpfIntf.lookupVrf = resolvedLookupVrf
         # pylint: disable-next=unsupported-assignment-operation
         uRpfModel.interfaces[ intf.name ] = uRpfIntf
   return uRpfModel

def populateIpv6UrpfPrefixModel( mode, prefix, vrfName ):
   uRpfModel = Ip6UrpfInterfaces()
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, vrfName )
   if not vrfExists( vrfName ):
      mode.addError( noIpv6RouteTableForVrfMsg % vrfName )
      uRpfModel.interfaces = None
      return uRpfModel

   # Ensure I have an Ip6AddrWithMask and not an Ip6Addr
   if prefix.tacType.fullTypeName == 'Arnet::Ip6Addr':
      prefix = Tac.Value( 'Arnet::Ip6AddrWithMask', address=prefix, len=128 )
   routeEntry = routing6.route( prefix, None, vrfName=vrfName )
   if not routeEntry:
      uRpfModel.interfaces = None
      return uRpfModel

   uRpfModel.prefix = prefix.stringValue
   uRpfModel.viaDefaultRoute = \
            routeEntry.prefix == routing6.ip.defaultRoute
   status = routing6.fGenStatus
   fec = status( vrfName ).fec.get( routeEntry.fecId )

   for via in fec.via.values():
      intf = via.intfId
      intfConfig = ip6.config.intf.get( intf )
      v6Mode = intfConfig.urpf.mode if intfConfig else UrpfMode.disable
      v6AllowDefault = intfConfig.urpf.allowDefaultRoute if intfConfig else False
      v6LookupVrf = intfConfig.urpf.lookupVrf if intfConfig else ''
      resolvedMode, resolvedAllowDefault, resolvedLookupVrf = \
         v6ResolvedUrpfStatus( intf, v6Mode, v6AllowDefault, v6LookupVrf )
      uRpfIntf = Ip6UrpfInterface()
      uRpfIntf.uRpfMode = resolvedMode
      uRpfIntf.allowDefault = resolvedAllowDefault
      uRpfIntf.lookupVrf = resolvedLookupVrf
      # pylint: disable-next=unsupported-assignment-operation
      uRpfModel.interfaces[ intf ] = uRpfIntf
   return uRpfModel

def showIpv6Urpf( mode, vrfName=None, prefix=None ):

   # Get all routeable intefaces
   intfs = routedIntfsWithIp6Configured( mode, None, vrfName )

   # Use the ipv6 one which is in intfs[ 1 ]
   if prefix is None:
      return populateIpv6UrpfModel( mode, intfs[ 1 ] )
   else:
      return populateIpv6UrpfPrefixModel( mode, prefix, vrfName )

def handlerShowIpv6UrpfCmd( mode, args ):
   return showIpv6Urpf( mode, vrfName=args.get( 'VRF' ),
                        prefix=args.get( 'PREFIX' ) )

def setFibFilter( mode, enable=True ):
   routing6HardwareConfig.fibSuppression = enable

def noSetFibFilter( mode ):
   setFibFilter( mode, enable=False )

def handlerIpv6FibFilterCmd( mode, args ):
   setFibFilter( mode )

def noOrDefaultHandlerIpv6FibFilterCmd( mode, args ):
   noSetFibFilter( mode )

def handlerShowIpv6UserRouteCmd( mode, args ):
   vrfName = VrfCli.vrfMap.lookupCliModeVrf( mode, args.get( 'VRF' ) )
   if not vrfExists( vrfName ):
      mode.addError( noIpv6RouteTableForVrfMsg % vrfName )
      return None

   fmt = mode.session_.outputFormat()
   tag = args.get( 'TAGNUM', 0 )
   configTag = args.get( 'CONFIG_TAG', "" )
   routing6.showUserRoutes( vrfName=vrfName, tag=tag,
                            configTag=configTag,
                            fmt=fmt, mlib=printer )
   return VrfIpRibRoutes

# -------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
# -------------------------------------------------------------------------------
def Plugin( entityManager ):
   global routing6HardwareConfig
   global ip4Config
   global ip6Config
   global runnabilityConfig

   routing6HardwareConfig = ConfigMount.mount( entityManager,
         "routing6/hardware/config", "Routing6::Hardware::Config", "w" )
   ip4Config = ConfigMount.mount( entityManager, "ip/config", "Ip::Config", "w" )
   ip6Config = ConfigMount.mount( entityManager, "ip6/config", "Ip6::Config", "w" )
   runnabilityConfig = ConfigMount.mount( entityManager,
                                          "routing6/runnability/config",
                                          "Routing6::Runnability::Config", "w" )
