# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import (
      Dict,
      Model,
      Str,
)
from TableOutput import (
      createTable,
      Format
)
import textwrap

class ShowCliVrfCtxModel( Model ):
   vrfRoutingContext = Str( help="Name of VRF for the current routing-context." )

   def render( self ):
      # pylint: disable-next=consider-using-f-string
      print( "Current VRF routing-context is %s" % self.vrfRoutingContext )

class VrfReservedPortsEntry( Model ):
   config = \
         Str( help="Reserved Ports configuration associated with this VRF",
              default="None" )
   status = \
         Str( help="Reserved Ports status associated with this VRF",
              default="None" )

   def formatReservedPortsStr( self ):
      if self.config == "None":
         return "None"
      else:
         # break on commas if more than 40 wide
         ports = self.config
         ports = textwrap.wrap( ports.replace( ',', ' ' ), 40,
                                break_long_words=False )
         ports = ',\n'.join( ports ).replace( ' ', ',' )
         return ports

   def formatComment( self ):
      if self.config != self.status:
         if self.status == "":
            return "Overridden by: None"
         else:
            # break on commas if more than 25 wide
            # We choose 25 to keep the text with the indent no longer than 40 wide
            ports = self.status
            ports = textwrap.wrap( ports.replace( ',', ' ' ), 25,
                                   break_long_words=False )
            ports = ',\n'.join( ports ).replace( ' ', ',' )
            # add indent the length of "Overridden by: "
            indent = "               "
            # pylint: disable-next=consider-using-f-string
            ports = ports.replace( '\n', '\n%s' % indent )
            # pylint: disable-next=consider-using-f-string
            return "Overridden by: %s" % ports
      return ""

# --------------------------------------------------------
# show vrf reserved-ports Model
# Contains a dictionary of vrf entries based on vrf name
# and the output is rendered in the form of a table
# ---------------------------------------------------------
class VrfReservedPorts( Model ):
   ''' render all the goodies in the Vrf '''
   vrfs = Dict( keyType=str, valueType=VrfReservedPortsEntry,
         help="Mapping of VRF name to its reserved ports" )

   def render( self ):
      t = createTable( ( 'VRF', 'Reserved ports', 'Comment' ) )
      f = Format( justify='left' )
      t.formatColumns( f, f, f )
      for vrfName, vrfRpEntry in self.vrfs.items():
         t.newRow( vrfName, vrfRpEntry.formatReservedPortsStr(),
                   vrfRpEntry.formatComment() )
      print( t.output() )
