# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliDynamicSymbol import LazyCallback
from CliGlobal import CliGlobal
import LazyMount
from TypeFuture import TacLazyType

CardProfileReader = TacLazyType( 'L1Profile::CardProfileReader' )
CardProfileSource = TacLazyType( 'L1Profile::CardProfileSource::CardProfileSource' )
EntityMibStatus = TacLazyType( 'EntityMib::Status' )
MountConstants = TacLazyType( 'L1Profile::MountConstants' )

def allInstalledProfiles( profileReader ):
   for lib in ( profileReader.sourceLibrary( CardProfileSource.builtin ),
                profileReader.sourceLibrary( CardProfileSource.cli ) ):
      for profile in lib.cardProfile.values():
         yield profile

# TODO BUG790391: Applicability handling for CLI defined profiles works differently
def applicableModules( moduleToModelName, applicability ):
   return [ module
            for module, modelName in moduleToModelName.items()
            if modelName in applicability ]

gv = CliGlobal( dict( entityMib=None,
                      cardProfileLibraryRootDir=None ) )

def handler( mode, args ):
   summary = LazyCallback( 'L1CardProfileEnumerationModel.Summary' )()

   # Lazy mounting causes issues when passing mounted entities to TACC code.
   # Preemptively force the mounts to complete to avoid deadlocking later on.
   LazyMount.force( gv.cardProfileLibraryRootDir )

   insertedModules = {}
   if fixedSystem := gv.entityMib.fixedSystem:
      insertedModules = { 'Switch': fixedSystem.modelName }
   elif modularSystem := gv.entityMib.chassis:
      insertedModules = { f'{ cardSlot.tag }{ cardSlot.label }': card.modelName
         for cardSlot in modularSystem.cardSlot.values()
         if ( card := cardSlot.card ) }

   applicabilitySpan = set()
   if argsSlot := args.get( 'MOD' ):
      for argSlotId in argsSlot.values():
         module = argsSlot.type.tagLong + str( argSlotId )
         if module not in insertedModules:
            applicabilitySpan.add( 'NONINSERT' )
            continue
         applicabilitySpan.add( insertedModules[ module ] )

   searchCriteria = args.get( 'NAME', '' ).lower()

   for profile in allInstalledProfiles(
         CardProfileReader( gv.cardProfileLibraryRootDir ) ):
      if not profile.visible:
         continue

      if searchCriteria not in profile.displayName.lower():
         continue

      profileApplicability = set( profile.applicability.keys() )
      profileApplicable = profileApplicability & applicabilitySpan
      if applicabilitySpan and not profileApplicable:
         continue

      enumeration = LazyCallback( 'L1CardProfileEnumerationModel.Enumeration' )()
      enumeration.description = profile.description

      # TODO BUG790391: Logic is needed here to populate CLI defined profiles
      # TODO BUG738920: For CLI defined profiles, applicability information needs
      #                 to be determined
      if profile.source != CardProfileSource.builtin:
         continue

      enumeration.applicableModules = applicableModules( insertedModules,
                                                         profileApplicability )

      summary.builtin[ profile.displayName ] = enumeration

   return summary

def Plugin( em ):
   gv.entityMib = LazyMount.mount(
      em,
      'hardware/entmib',
      EntityMibStatus.tacType.fullTypeName,
      'r' )
   gv.cardProfileLibraryRootDir = LazyMount.mount(
      em,
      MountConstants.cardProfileLibraryRootDirPath(),
      'Tac::Dir',
      'ri' )
