# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliDynamicSymbol import LazyCallback
from CliGlobal import CliGlobal
from CliPlugin.L1ProfileCliCommon import (
   FIXEDSYSTEM_DISPLAY_NAME,
   PROFILE_STATUS_APPLIED,
   PROFILE_STATUS_ERROR,
   PROFILE_STATUS_NOT_INSERTED,
   PROFILE_STATUS_PENDING,
   SUPPORTED_MODULE_TYPES,
)
import LazyMount
from TypeFuture import TacLazyType

CardConfigDir = TacLazyType( 'L1Profile::CardConfigDir' )
CardConfigGenerator = TacLazyType( 'L1Profile::CardConfigGenerator' )
CardDefault = TacLazyType( 'L1Profile::CardDefault' )
CardDefaultDir = TacLazyType( 'L1Profile::CardDefaultDir' )
CardProfileDescriptor = TacLazyType( 'L1Profile::CardProfileDescriptor' )
CardProfileReader = TacLazyType( 'L1Profile::CardProfileReader' )
CardProfileSource = TacLazyType( 'L1Profile::CardProfileSource::CardProfileSource' )
CliConfig = TacLazyType( 'L1Profile::CliConfig' )
CompareUtilities = TacLazyType( 'L1Profile::CompareUtilities' )
EntityMibStatus = TacLazyType( 'EntityMib::Status' )
MountConstants = TacLazyType( 'L1Profile::MountConstants' )

gv = CliGlobal( dict( entityMib=None,
                      cardProfileLibraryRootDir=None,
                      intfSlotProfileLibraryRootDir=None,
                      cliConfig=None,
                      cardConfigDir=None,
                      cardDefaultDir=None ) )

def handler( mode, args ):
   summary = LazyCallback( 'L1CardProfileStatusModel.Summary' )()

   # Lazy mounting causes issues when passing mounted entities to TACC code.
   # Preemptively force the mounts to complete to avoid deadlocking later on.
   LazyMount.force( gv.cardProfileLibraryRootDir )
   LazyMount.force( gv.intfSlotProfileLibraryRootDir )

   profileReader = CardProfileReader( gv.cardProfileLibraryRootDir )

   isFixed = gv.entityMib.fixedSystem
   cardSlots = {}
   if isFixed:
      fixedSystemEntMib = gv.entityMib.fixedSystem
      cardSlots[ fixedSystemEntMib.tag ] = fixedSystemEntMib.modelName
   else:
      for cardSlot in gv.entityMib.chassis.cardSlot.values():
         if cardSlot.tag not in SUPPORTED_MODULE_TYPES:
            continue
         cardSlotId = f'{cardSlot.tag}{cardSlot.label}'
         cardSlots[ cardSlotId ] = getattr( cardSlot.card, 'modelName', '' )

   # Determines the set of cards which need to be displayed. For fixed systems,
   # it will always display the single 'switch', but for modular systems we have
   # the option to display a subset of the chassis' cards depending on the MOD
   # parameter, which is optional. Care has to be taken when ingesting this input
   # as guards may be disabled and the user may have specified normally acceptable
   # card slot IDs that do not exist on the chassis
   displayCardSlots = set( cardSlots )
   if not isFixed:
      argsSlot = args.get( 'MOD' )
      if argsSlot:
         argsSlotIds = { argsSlot.type.tagLong + str( argSlotId )
                         for argSlotId in argsSlot.values() }
         displayCardSlots &= argsSlotIds

   def isDefault( cardSlotId, desc ):
      '''Determines if the the specified card profile matches the SKU defaults or
      not.

      Args:
         cardSlotId ( str ): The name of the card whose profile name is being
                             normalized.
         desc ( CardProfileDescriptor ): The descriptor of the profile to be
                                         normalized.

      Returns:
         True if the specified CardProfileDescriptor matches the SKU defaults of
         the card inserted at the specified card slot ID.
      '''
      desc = profileReader.resolvedDescriptor(
            cardSlots[ cardSlotId ],
            desc ) or desc

      cardDefault = gv.cardDefaultDir.cardDefault.get( cardSlotId )
      if not cardDefault:
         return False

      cardDefaultDesc = profileReader.resolvedDescriptor(
            cardSlots[ cardSlotId ],
            cardDefault.cardProfile )
      if cardDefaultDesc != desc:
         return False

      return True

   for cardSlotId in displayCardSlots:
      cardStatus = LazyCallback( 'L1CardProfileStatusModel.Status' )()
      cardSlotIdKey = cardSlotId if not isFixed else FIXEDSYSTEM_DISPLAY_NAME
      summary.modules[ cardSlotIdKey ] = cardStatus

      # Determine the configured card profile
      configuredCardProfileDesc = gv.cliConfig.cardProfile.get( cardSlotId )
      if ( configuredCardProfileDesc and
           not isDefault( cardSlotId, configuredCardProfileDesc ) ):
         cardStatus.configured = configuredCardProfileDesc.name

      # Determine the operational card profile
      cardConfig = gv.cardConfigDir.cardConfig.get( cardSlotId )
      operationalCardProfileDesc = CardProfileDescriptor()
      if cardConfig and cardConfig.cardProfile:
         operationalCardProfileDesc = cardConfig.cardProfile.descriptor

      if ( operationalCardProfileDesc and
           not isDefault( cardSlotId, operationalCardProfileDesc ) ):
         cardStatus.operational = operationalCardProfileDesc.name
         cardProfile = profileReader.profile( operationalCardProfileDesc )
         if cardProfile and cardProfile.displayName:
            cardStatus.operational = cardProfile.displayName

      # If card is not inserted then the "not inserted" always trumps all other
      # statuses.
      #
      # NOTE: There is an implicit expectation that the operational card profile
      #       is set to NA or will be pretty soon.
      if not cardSlots[ cardSlotId ]:
         cardStatus.operational = None
         cardStatus.status = PROFILE_STATUS_NOT_INSERTED
         continue

      # If there is no customer configuration then the status reporting is much
      # simpler on account of not it not requiring any error checking. Special
      # care still has to be taken account when displaying default profile
      # application status.
      #
      # Note: This implementation does not compare the operational card
      #       configuration definition to the desired default card definition.
      #       This may expose us to bugs in case the default card profile
      #       definition ever changes.
      if not configuredCardProfileDesc:
         cardDefaultConfig = gv.cardDefaultDir.cardDefault.get(
               cardSlotId,
               CardDefault( cardSlotId ) )
         defaultProfileDesc = profileReader.resolvedDescriptor(
               cardSlots[ cardSlotId ],
               cardDefaultConfig.cardProfile )

         operationalProfileDesc = CardProfileDescriptor()
         if cardConfig and cardConfig.cardProfile:
            operationalProfileDesc = cardConfig.cardProfile.descriptor

         # In cases where there is a default card profile specified by the SKU
         # the comamnd may be run before the card config is generated. In such a
         # case we will confusingly print that the status is pending. As this
         # represents such a small wilndow of time, it's easier to ignore than
         # attempt to fix.
         if operationalProfileDesc == defaultProfileDesc:
            cardStatus.status = PROFILE_STATUS_APPLIED
         else:
            cardStatus.status = PROFILE_STATUS_PENDING

         continue

      # Generate the expected card configuration based on the current CLI inputs
      # so that it can be compared against what is actually operational on the
      # card.
      modelName = cardSlots[ cardSlotId ]
      generatedCardConfig = CardConfigGenerator.generateConfig(
         gv.cardProfileLibraryRootDir,
         gv.intfSlotProfileLibraryRootDir,
         cardSlotId,
         modelName,
         configuredCardProfileDesc )

      # At this point, if both the configured and operational profile names are
      # identical then differences between them might be due to a redefinition
      # of the configured profile. If there are no changes to the definition then
      # this flag will be cleared later.
      #
      # pylint: disable=protected-access
      cardStatus._definitionMismatch = (
         cardStatus.configured == cardStatus.operational )

      # If card configuration generation has failed, then the CLI inputs are most
      # likely inconsistent / refer to sub profiles that don't exist.
      if not generatedCardConfig:
         cardStatus.status = PROFILE_STATUS_ERROR
         continue

      # Next, the validity of the generated configuration is determined.
      #
      # NOTE: If there is an error here then we expect the generated card
      #       configuration to be different than the card configuration. However,
      #       there is no easy way for us to assert this without crashing the CLI.
      if ( cardSlots[ cardSlotId ]
           not in generatedCardConfig.cardProfile.applicability ):
         cardStatus.status = PROFILE_STATUS_ERROR
         continue

      # At this point, if an operational card profile is not present then the
      # system is pending a reboot to latch the currently configured profile.
      if not cardConfig or not cardConfig.cardProfile:
         cardStatus.status = PROFILE_STATUS_PENDING
         continue

      # If there are no errors with the configured card profile, it is valid
      # and only has to be compared against the operational card configuration to
      # determine if it's applied or pending a reboot.
      #
      # TODO BUG745413: Only the card profiles are compared as there is no support
      #                 for custom interface slot profiles.
      if CompareUtilities.compareCardProfile( generatedCardConfig.cardProfile,
                                              cardConfig.cardProfile ).mismatch():
         cardStatus.status = PROFILE_STATUS_PENDING
         continue

      # At this stage there are no changes between the configured and operational
      # card profile configurations and the card is in steady state.
      #
      # pylint: disable=protected-access
      cardStatus._definitionMismatch = False
      cardStatus.status = PROFILE_STATUS_APPLIED

   return summary

def Plugin( em ):
   gv.entityMib = LazyMount.mount(
      em,
      'hardware/entmib',
      EntityMibStatus.tacType.fullTypeName,
      'r' )
   gv.cardProfileLibraryRootDir = LazyMount.mount(
      em,
      MountConstants.cardProfileLibraryRootDirPath(),
      'Tac::Dir',
      'ri' )
   gv.intfSlotProfileLibraryRootDir = LazyMount.mount(
      em,
      MountConstants.intfSlotProfileLibraryRootDirPath(),
      'Tac::Dir',
      'ri' )
   gv.cliConfig = LazyMount.mount(
      em,
      MountConstants.cliConfigPath(),
      CliConfig.tacType.fullTypeName,
      'r' )
   gv.cardDefaultDir = LazyMount.mount(
      em,
      MountConstants.cardDefaultDirPath(),
      CardDefaultDir.tacType.fullTypeName,
      'r' )
   gv.cardConfigDir = LazyMount.mount(
      em,
      MountConstants.cardConfigPath(),
      CardConfigDir.tacType.fullTypeName,
      'r' )
