# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''This module provides the CLI models used for rendering / capi-fying the L1 profile
card status command.
'''

import natsort

import CliModel
from CliPlugin.L1ProfileCliCommon import (
   PROFILE_NAME_NA,
   PROFILE_STATUS_APPLIED,
   PROFILE_STATUS_ERROR,
   PROFILE_STATUS_NOT_INSERTED,
   PROFILE_STATUS_PENDING,
   PROFILE_STATUS_RENDER_MAP,
)
import TableOutput

class Status( CliModel.Model ):
   '''A model representing the L1 card profile configuration status for a particular
   card.
   '''

   operational = CliModel.Str( help='The operational L1 module profile',
                               optional=True )
   configured = CliModel.Str( help='The configured L1 module profile',
                              optional=True )
   status = CliModel.Enum( values={ PROFILE_STATUS_APPLIED,
                                    PROFILE_STATUS_ERROR,
                                    PROFILE_STATUS_NOT_INSERTED,
                                    PROFILE_STATUS_PENDING },
                           help='The status of the L1 module profile configuration' )
   _definitionMismatch = CliModel.Bool( help='Indicates that the operational '
                                             'profile definition has diverged from '
                                             'that of the configured profile' )

   def qualifiedOperational( self ):
      '''Returns the operation profile string to be rendered.'''
      return ( f'{self.operational or PROFILE_NAME_NA}'
               f'{"*" if self._definitionMismatch else ""}' )

   def qualifiedConfigured( self ):
      '''Returns the configured profile string to be rendered.'''
      return self.configured or PROFILE_NAME_NA

   def qualifiedStatus( self ):
      '''Returns the status of string as it's meant to be rendered.'''
      return PROFILE_STATUS_RENDER_MAP.get( self.status, self.status )

class Summary( CliModel.Model ):
   '''A model representing a summary of the L1 card profile configuration for every
   card on the system.
   '''

   modules = CliModel.Dict( valueType=Status,
                            help='The L1 module profile configuration status for a '
                                 'particular module' )

   def render( self ):
      print( '\n* Indicates that the configured profile definition has changed \n' )

      table = TableOutput.TableFormatter()

      # Apply the headers to the table
      hdr = TableOutput.Headings(
         [ 'Module', 'Operational', 'Configured', 'Status' ] )
      hdr.doApplyHeaders( table )

      # Define the table's format
      fmt = TableOutput.Format( justify='left' )
      table.formatColumns( fmt, fmt, fmt, fmt )

      for fmt in table.columnFormats_:
         fmt.padLimitIs( True )

      for cardSlotId, cardProfileStatus in natsort.natsorted( self.modules.items() ):
         table.newRow(
            cardSlotId.lower(),
            cardProfileStatus.qualifiedOperational(),
            cardProfileStatus.qualifiedConfigured(),
            cardProfileStatus.qualifiedStatus() )

      print( table.output() )
