# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliDynamicSymbol import CliDynamicPlugin

LSCL = CliDynamicPlugin( 'LineSystemCliLib' )
LSEM = CliDynamicPlugin( 'LineSystemShowEepromModel' )
XSEH = CliDynamicPlugin( 'XcvrShowEepromHandler' )
xcvrIdpromDumpFunc = CliDynamicPlugin( 'XcvrShowIdpromHandler' ).xcvrIdpromDumpFunc

# -------------------------------------------------------------------------------
# Handlers for "show line system [PORT X] eeprom"
# -------------------------------------------------------------------------------

def lineSystemDoEepromDump( mode, xcvrStatus, xcvrDir, intfs, model ) -> None:
   """
   Wrapper for Xcvr eeprom dump func
   """
   # pylint: disable=protected-access
   XSEH._doDecodedEepromDump( mode, xcvrStatus, xcvrDir, intfs, model )

def subEthernetToPort( model: object ) -> None:
   """
   Converts LineSystemEepromBase "Ethernet X" keys to "Port X"
   """
   # Change "Ethernet" strings to "Port" after command completion. We want our
   # CLI to use the word "Port" for LineSystem CLI, as it interacts with
   # front-panel hardware slots.
   for intf in sorted( model.interfaces.keys(), key=LSCL.subPrefixToPort ):
      portName = LSCL.subPrefixToPort( intf )
      model.interfaces[ portName ] = model.interfaces[ intf ]
      # pylint: disable=protected-access
      model.interfaces[ portName ]._name = portName
      del model.interfaces[ intf ]

def showLineSystemEepromHandler( mode, args ):
   """
   Handler function for "show line system eeprom" command.
   """
   # Uses the pre-existing handler methods from the transceiver-based eeprom command,
   # with added limitations of ports with linesystems, and editing "Ethernet"
   # strings to "Port"
   model = LSEM.LineSystemEepromBase()
   portRange = args.get( 'PORTRANGE' )
   applicableIntfs = LSCL.lineSystemIntfsInPortRange( mode, portRange )
   if not applicableIntfs:
      return model

   model = LSEM.LineSystemEepromBase()
   xcvrIdpromDumpFunc( mode, lineSystemDoEepromDump, model,
                       intf=applicableIntfs, includeAuxiliary=True )

   subEthernetToPort( model )
   return model
