#!/usr/bin/env python3
# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliMatcher
from CliMode.LoopProtect import MonitorLoopProtectMode
from CliPlugin.LoopProtectCli import config
import CliPlugin.VlanCli as VlanCli # pylint: disable=consider-using-from-import

TRANSMIT_INTERVAL_DEFAULT = 5
DISABLED_TIME_DEFAULT = 604800
RATE_LIMIT_DEFAULT = 1000

class MonitorLoopProtectConfigMode( MonitorLoopProtectMode,
                                    BasicCli.ConfigModeBase ):
   name = 'monitor-loop-protect'

   def __init__( self, parent, session ):
      MonitorLoopProtectMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      if not config.globalIntfConfig:
         config.globalIntfConfig = ( 'globalIntfConfig', )
      if not config.transmitConfig:
         config.transmitConfig = ( 'transmitConfig', )

   def noShutdown( self, args ):
      config.globalEnabled = True

   def shutdown( self, args ):
      config.globalEnabled = False

   def protectVlan( self, args ):
      vlanSet = args[ 'VLANSET' ]
      for vlan in vlanSet:
         config.vlanEnabled.add( vlan )

   def noProtectVlan( self, args ):
      vlanSet = args[ 'VLANSET' ]
      for vlan in vlanSet:
         config.vlanEnabled.remove( vlan )

   def transmitInterval( self, args ):
      interval = args.get( 'INTERVAL', TRANSMIT_INTERVAL_DEFAULT )
      config.globalIntfConfig.transmitInterval = interval

   def disabledTime( self, args ):
      seconds = args.get( 'SECONDS', DISABLED_TIME_DEFAULT )
      config.globalIntfConfig.disabledTime = seconds

   def rateLimit( self, args ):
      limit = args.get( 'LIMIT', RATE_LIMIT_DEFAULT )
      config.transmitConfig.rateLimit = limit

#-------------------------------------------------------------------------------
# This module implements the following commands:
#-------------------------------------------------------------------------------
# Loop protect mode:
#      -  ( no | default ) disabled-time [ DISABLEDTIME ]
#      -  [ no ] [ default ] protect vlan VLANSET
#      -  ( no | default ) rate-limit [ RATELIMIT ]
#      -  [ no ] [ default ] shutdown
#      -  ( no | default ) transmit-interval [ TRANSMITINTERVAL ]
#--------------------------------------------------------------------------------

# pylint: disable=ungrouped-imports

matcherDisabledTime = CliMatcher.KeywordMatcher( 'disabled-time', 
      helpdesc='Set how long looped interfaces are disabled for' )
matcherRateLimit = CliMatcher.KeywordMatcher( 'rate-limit', 
      helpdesc='Set throttling of loop protection messages' )
matcherTransmitInterval = CliMatcher.KeywordMatcher( 'transmit-interval', 
      helpdesc='Set message transmission interval' )

#--------------------------------------------------------------------------------
# ( no | default ) disabled-time [ SECONDS ]
#--------------------------------------------------------------------------------
# Defaults to 604800

class DisabledTimeCmd( CliCommand.CliCommandClass ):
   syntax = 'disabled-time SECONDS'
   noOrDefaultSyntax = 'disabled-time ...'
   data = {
      'disabled-time': matcherDisabledTime,
      'SECONDS': CliMatcher.IntegerMatcher( 0, DISABLED_TIME_DEFAULT, 
         helpdesc='How long to disable looped interfaces for, in seconds' ),
   }
   
   handler = MonitorLoopProtectConfigMode.disabledTime
   noOrDefaultHandler = handler

MonitorLoopProtectConfigMode.addCommandClass( DisabledTimeCmd )

#--------------------------------------------------------------------------------
# [ no ] [ default ] protect vlan VLANSET
#--------------------------------------------------------------------------------
# Defaults to no vlans protected

class ProtectVlanVlansetCmd( CliCommand.CliCommandClass ):
   syntax = 'protect vlan VLANSET'
   noOrDefaultSyntax = syntax
   data = {
      'protect': 'Protect VLAN from loops',
      'vlan': 'VLANs to be involved in loop protection',
      'VLANSET': VlanCli.vlanSetMatcher, 
   }
   
   handler = MonitorLoopProtectConfigMode.protectVlan
   noHandler = MonitorLoopProtectConfigMode.noProtectVlan
   defaultHandler = handler

MonitorLoopProtectConfigMode.addCommandClass( ProtectVlanVlansetCmd )

#--------------------------------------------------------------------------------
# ( no | default ) rate-limit [ LIMIT ]
#--------------------------------------------------------------------------------
# Defaults to 1000

class RateLimitCmd( CliCommand.CliCommandClass ):
   syntax = 'rate-limit LIMIT'
   noOrDefaultSyntax = 'rate-limit ...'
   data = {
      'rate-limit': matcherRateLimit,
      'LIMIT': CliMatcher.IntegerMatcher( 0, RATE_LIMIT_DEFAULT, 
      helpdesc='Maximum number of loop protection messages generated per second' ),
   }
   
   handler = MonitorLoopProtectConfigMode.rateLimit
   noOrDefaultHandler = handler

MonitorLoopProtectConfigMode.addCommandClass( RateLimitCmd )

#--------------------------------------------------------------------------------
# [ no ] [ default ] shutdown
#--------------------------------------------------------------------------------
# Defaults to shutdown. Globally disabled initially

class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown': 'Disable loop protection globally',
   }
   
   handler = MonitorLoopProtectConfigMode.shutdown
   noHandler = MonitorLoopProtectConfigMode.noShutdown
   defaultHandler = handler

MonitorLoopProtectConfigMode.addCommandClass( ShutdownCmd )

#--------------------------------------------------------------------------------
# ( no | default ) transmit-interval [ INTERVAL ]
#--------------------------------------------------------------------------------
# Defaults to 5

class TransmitIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'transmit-interval [ INTERVAL ]'
   noOrDefaultSyntax = 'transmit-interval ...'
   data = {
      'transmit-interval': matcherTransmitInterval,
      'INTERVAL': CliMatcher.IntegerMatcher( 1, 10, 
         helpdesc='How often to transmit loop protection messages in seconds' ),
   }
   
   handler = MonitorLoopProtectConfigMode.transmitInterval
   noOrDefaultHandler = handler

MonitorLoopProtectConfigMode.addCommandClass( TransmitIntervalCmd )
